/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags;

import jaxx.CompilerException;
import jaxx.compiler.JAXXCompiler;
import org.w3c.dom.Element;

import java.io.IOException;

/**
 * Implementations of <code>TagHandler</code> produce Java source code from XML tags.
 * <code>TagHandlers</code> are mapped to particular XML tags (such as &lt;JFrame&gt;) in {@link JAXXCompiler}.
 * There is only one <code>TagHandler</code> for any given XML tag, and therefore implementations must be
 * stateless.
 *
 * @author Ethan Nicholas
 */
public interface TagHandler {
    /**
     * Performs the first pass of compilation on an XML tag from a JAXX source file.
     * <code>TagHandler</code> implementations affect the generated <code>.java</code>
     * file by calling methods in the <code>JAXXCompiler</code>.
     *
     * @param tag      the XML tag to compile
     * @param compiler the active JAXXCompiler
     * @throws CompilerException if a compilation error occurs
     * @throws IOException       if an I/O error occurs
     */
    void compileFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException;


    /**
     * Performs the second pass of compilation on an XML tag from a JAXX source file.
     * <code>TagHandler</code> implementations affect the generated <code>.java</code>
     * file by calling methods in the <code>JAXXCompiler</code>.
     *
     * @param tag      the XML tag to compile
     * @param compiler the active JAXXCompiler
     * @throws CompilerException if a compilation error occurs
     * @throws IOException       if an I/O error occurs
     */
    void compileSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException;
}