/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.compiler;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;

/**
 * Represents a method in a Java source file being generated for output.  <code>JavaMethods</code> are created
 * and added to a {@link JavaFile}, which can then output Java source code.  In addition to normal methods, a
 * <code>JavaMethod</code> can represent a constructor -- constructors should be named after their containing
 * classes and have a return type of <code>null</code>.
 */
public class JavaMethod implements Comparable<JavaMethod> {
    private int modifiers;
    private String returnType;
    private String name;
    private JavaArgument[] arguments;
    private String[] exceptions;
    private StringBuffer bodyCode;


    /**
     * Constructs a new no-argument <code>JavaMethod</code> which throws no checked exceptions.  The
     * <code>modifiers</code> parameter is a bit mask of the constants from {@link java.lang.reflect.Modifier},
     * and the <code>returnType</code> of the method should be represented as it would appear in Java source
     * code (<code>null</code> for a constructor).  The method body is initially empty.
     *
     * @param modifiers  the modifier keywords that should appear as part of the method's declaration
     * @param returnType the return type of the method as it would appear in Java source code
     * @param name       the method's name
     * @see #appendBodyCode
     */
    //public JavaMethod(int modifiers, String returnType, String name) {
    //    this(modifiers, returnType, name, null);
    //}


    /**
     * Constructs a new <code>JavaMethod</code> which throws no checked exceptions.  The <code>modifiers</code>
     * parameter is a bit mask of the constants from {@link java.lang.reflect.Modifier}, and the
     * <code>returnType</code> of the method should be represented as it would appear in Java source code
     * (<code>null</code> for a constructor).  The method body is initially empty.
     *
     * @param modifiers  the modifier keywords that should appear as part of the method's declaration
     * @param returnType the return type of the method as it would appear in Java source code
     * @param name       the method's name
     * @param arguments  the method's arguments
     * @see #appendBodyCode
     */
    //public JavaMethod(int modifiers, String returnType, String name, JavaArgument[] arguments) {
    //    this(modifiers, returnType, name, arguments, null);
    //}


    /**
     * Constructs a new <code>JavaMethod</code>.  The <code>modifiers</code> parameter is a bit mask of the
     * constants from {@link java.lang.reflect.Modifier}, and the <code>returnType</code> and <code>exceptions</code>
     * of the method should be represented as they would appear in Java source code (<code>null</code> for a
     * constructor).  The method body is initially empty.
     *
     * @param modifiers  the modifier keywords that should appear as part of the method's declaration
     * @param returnType the return type of the method as it would appear in Java source code
     * @param name       the method's name
     * @param arguments  the method's arguments
     * @param exceptions a list of exceptions the methods can throw, as they would be represented in Java source code
     * @see #appendBodyCode
     */
    //public JavaMethod(int modifiers, String returnType, String name, JavaArgument[] arguments, String[] exceptions) {
    //    this(modifiers, returnType, name, arguments, exceptions, null);
    //}


    /**
     * Constructs a new <code>JavaMethod</code> containing the specified body code.  The <code>modifiers</code> parameter
     * is a bit mask of the  constants from {@link java.lang.reflect.Modifier}, and the <code>returnType</code> and
     * <code>exceptions</code> of the method should be represented as they would appear in Java source code (<code>null</code>
     * for a constructor).  The method body is initially empty.
     *
     * @param modifiers  the modifier keywords that should appear as part of the method's declaration
     * @param returnType the return type of the method as it would appear in Java source code
     * @param name       the method's name
     * @param arguments  the method's arguments
     * @param exceptions a list of exceptions the methods can throw, as they would be represented in Java source code
     * @param bodyCode   Java source code which should appear in the method body
     */
    public JavaMethod(int modifiers, String returnType, String name, JavaArgument[] arguments, String[] exceptions, String bodyCode) {
        this.modifiers = modifiers;
        this.returnType = returnType;
        this.name = name;
        this.arguments = arguments;
        this.exceptions = exceptions;
        this.bodyCode = new StringBuffer(bodyCode != null ? bodyCode : "");
    }


    /**
     * Returns a bit mask describing the modifier keywords which should appear as part of this method's
     * declaration.  See <code>java.lang.reflect.Modifier</code> for more information on decoding this
     * field.
     *
     * @return the modifier bit mask
     */
    public int getModifiers() {
        return modifiers;
    }


    /**
     * Returns the method's return type, as it would be represented in Java source code.
     *
     * @return the method's return type
     */
    public String getReturnType() {
        return returnType;
    }


    /**
     * Returns the method's name.
     *
     * @return the method's name
     */
    public String getName() {
        return name;
    }


    /**
     * Returns a list of the method's arguments.
     *
     * @return the method's arguments
     */
    public JavaArgument[] getArguments() {
        return arguments;
    }


    /**
     * Returns a list of exceptions the method can throw.
     *
     * @return the method's exceptions
     */
    public String[] getExceptions() {
        return exceptions;
    }


    /**
     * Returns the Java source code for the method's body.
     *
     * @return the method's body code
     */
    public String getBodyCode() {
        return bodyCode.toString();
    }


    /**
     * Appends additional code to the method's body.
     *
     * @param extraCode     Java source code to append to the method's body
     * @param lineSeparator line separator
     */
    public void appendBodyCode(String extraCode, String lineSeparator) {
        if (extraCode.length() == 0) {
            return;
        }
        if (bodyCode.length() > 0 && !bodyCode.toString().endsWith(lineSeparator)) {
            bodyCode.append(lineSeparator);
        }
        bodyCode.append(extraCode);
    }


    /**
     * Returns the Java source code for this method.
     *
     * @param lineSeparator line separator
     * @return the Java source code for this method
     */
    public String toString(String lineSeparator) {
        StringBuffer result = new StringBuffer();
        result.append(JavaFile.getModifiersText(modifiers));
        if (returnType != null) {
            result.append(returnType);
            result.append(' ');
        }
        result.append(name);
        result.append('(');
        if (arguments != null) {
            for (int i = 0; i < arguments.length; i++) {
                if (i > 0) {
                    result.append(", ");
                }
                result.append(arguments[i].toString());
            }
        }
        result.append(") {");
        result.append(lineSeparator);
        if (bodyCode != null) {
            String formattedBodyCode = JavaFile.addIndentation(bodyCode.toString().trim(), 4, lineSeparator);
            if (formattedBodyCode.length() > 0) {
                result.append(formattedBodyCode);
                result.append(lineSeparator);
            }
        }
        result.append("}");
        return result.toString();
    }

    public int compareTo(JavaMethod o) {
        return COMPARATOR.compare(this, o);
    }

    public static JavaMethod newMethod(int modifiers, String returnType, String name, String initializer, String[] exceptions, JavaArgument... arguments) {
        return new JavaMethod(modifiers, returnType, name, arguments, exceptions, initializer);
    }

    public static JavaMethod newMethod(int modifiers, String returnType, String name, String initializer, JavaArgument... arguments) {
        return newMethod(modifiers, returnType, name, initializer, new String[0], arguments);
    }

    public enum MethodOrder {

        statics(Modifier.STATIC, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- Statics methods --------------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
        },

        constructors(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- Constructors -----------------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            public boolean accept(JavaMethod method) {
                return method.returnType == null;
            }
        },

        JAXXObject(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- JAXXObject implementation ----------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            private List<String> methods = Arrays.asList("applyDataBinding", "firePropertyChange", "getObjectById", "get$objectMap", "processDataBinding", "removeDataBinding");

            public boolean accept(JavaMethod method) {
                return methods.contains(method.name);
            }
        },

        JAXXContext(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- JAXXContext implementation ---------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            private List<String> methods = Arrays.asList("getContextValue", "getDelegateContext", "getParentContainer", "removeContextValue", "setContextValue");
            public boolean accept(JavaMethod method) {
                return methods.contains(method.name);
            }
        },

        JAXXValidation(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- JAXXValidation implementation ------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            private List<String> methods = Arrays.asList("getValidator", "getValidatorIds");

            public boolean accept(JavaMethod method) {
                return methods.contains(method.name);
            }
        },

        events(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- Event methods ----------------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            public boolean accept(JavaMethod method) {
                return (method.name.startsWith("do") && method.name.indexOf("__") > -1);
            }
        },

        publicGetters(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- public acessor methods -------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            public boolean accept(JavaMethod method) {
                return (method.name.startsWith("get") || method.name.startsWith("is"));
            }
        },

        publicSetters(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- public mutator methods -------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            public boolean accept(JavaMethod method) {
                return (method.name.startsWith("set"));
            }
        },

        otherPublic(Modifier.PUBLIC, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- public mutator methods -------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            @Override
            public boolean accept(int mod) {
                return super.accept(mod) && !Modifier.isStatic(mod);
            }
        },

        protectedGetters(Modifier.PROTECTED, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- protected acessors methods ---------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            public boolean accept(JavaMethod method) {
                return (method.name.startsWith("get") || method.name.startsWith("is"));
            }
        },

        createMethod(Modifier.PROTECTED | Modifier.PRIVATE, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- ui creation methods ----------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            public boolean accept(JavaMethod method) {
                return method.name.startsWith("create") || method.name.startsWith("add") ||
                        method.name.equals("$completeSetup") ||
                        method.name.equals("$initialize");
            }
        },

        protecteds(Modifier.PROTECTED, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- Other protected methods ------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
        },

        packageLocal(0, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- Package methods --------------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
            @Override
            public boolean accept(int mod) {
                return !Modifier.isStatic(mod) && !Modifier.isPublic(mod) && !Modifier.isProtected(mod);
            }},

        privates(Modifier.PRIVATE, "/*---------------------------------------------------------------------------------*/\n" +
                "/*-- Private methods --------------------------------------------------------------*/\n" +
                "/*---------------------------------------------------------------------------------*/") {
        };

        private final String header;

        private int modifier;

        MethodOrder(int modifier, String header) {
            this.header = header;
            this.modifier = modifier;
        }

        public String getHeader() {
            return header;
        }

        public boolean accept(JavaMethod method) {
            return true;
        }

        public boolean accept(int mod) {
            return (mod & modifier) != 0;
        }

        public boolean accept(int mod, JavaMethod method) {
            return accept(mod) && accept(method);
        }

        public static MethodOrder valueOf(JavaMethod method, int scope) {
            for (MethodOrder o : values()) {
                if (o.accept(scope, method)) {
                    return o;
                }
            }
            throw new IllegalArgumentException("could not find a " + MethodOrder.class + " for method " + method);
        }

        public void generate(StringBuffer buffer, List<JavaMethod> methods, String lineSeparator) {
            if (methods.isEmpty()) {
                return;
            }
            buffer.append(JavaFile.addIndentation(header, 4, lineSeparator));
            buffer.append(lineSeparator).append(lineSeparator);
            for (JavaMethod method : methods) {
                buffer.append(JavaFile.addIndentation(method.toString(lineSeparator), 4, lineSeparator));
                buffer.append(lineSeparator).append(lineSeparator);
            }
        }
    }

    public static EnumMap<MethodOrder, List<JavaMethod>> getSortedMethods(List<JavaMethod> methods) {

        EnumMap<MethodOrder, List<JavaMethod>> result = new EnumMap<MethodOrder, List<JavaMethod>>(MethodOrder.class);
        for (MethodOrder methodOrder : MethodOrder.values()) {
            result.put(methodOrder, new ArrayList<JavaMethod>());
        }

        EnumSet<MethodOrder> allConstants = EnumSet.allOf(MethodOrder.class);
        List<JavaMethod> allMethods = new ArrayList<JavaMethod>(methods);
        int[] scopes = new int[]{Modifier.STATIC, Modifier.PUBLIC, Modifier.PROTECTED, Modifier.PRIVATE};
        for (int scope : scopes) {
            EnumSet<MethodOrder> constants = getMethodOrderScope(allConstants, scope);

            Iterator<JavaMethod> itMethods = allMethods.iterator();
            while (itMethods.hasNext()) {
                JavaMethod method = itMethods.next();
                for (MethodOrder constant : constants) {
                    if (constant.accept(method.getModifiers(), method)) {
                        result.get(constant).add(method);
                        itMethods.remove();
                        break;
                    }
                }
            }
            constants.clear();
        }

        if (!allMethods.isEmpty()) {
            throw new IllegalArgumentException("could not find a " + MethodOrder.class + " for method " + allMethods);
        }

        for (MethodOrder methodOrder : MethodOrder.values()) {
            // sort methods
            java.util.Collections.sort(result.get(methodOrder));
        }
        return result;
    }

    public static EnumSet<MethodOrder> getMethodOrderScope(EnumSet<MethodOrder> allConstants, int scope) {
        EnumSet<MethodOrder> constants = EnumSet.noneOf(MethodOrder.class);
        for (MethodOrder order : allConstants) {
            if (order.accept(scope)) {
                constants.add(order);
            }
        }
        return constants;
    }

    public static final Comparator<JavaMethod> COMPARATOR = new Comparator<JavaMethod>() {

        public int compare(JavaMethod o1, JavaMethod o2) {

            /*int result;
            if ((result = compareStatic(o1, o2)) != 0) {
                return result;
            }
            if ((result = compareConstructor(o1, o2)) != 0) {
                return result;
            }
            if ((result = compareVisibility(o1, o2)) != 0) {
                return result;
            }*/
            return o1.name.compareTo(o2.name);
        }

        public int compareStatic(JavaMethod o1, JavaMethod o2) {
            if (Modifier.isStatic(o1.modifiers) && !Modifier.isStatic(o2.modifiers)) {
                return -1;
            }
            if (!Modifier.isStatic(o1.modifiers) && Modifier.isStatic(o2.modifiers)) {
                return 1;
            }
            return 0;
        }

        public int compareConstructor(JavaMethod o1, JavaMethod o2) {
            if (o1.returnType == null && o2.returnType != null) {
                return -1;
            }
            if (o1.returnType != null && o2.returnType == null) {
                return 1;
            }
            return 0;
        }

        public int compareVisibility(JavaMethod o1, JavaMethod o2) {
            if (Modifier.isPublic(o1.modifiers) && !Modifier.isPublic(o2.modifiers)) {
                return -1;
            }
            if (!Modifier.isPublic(o1.modifiers) && Modifier.isPublic(o2.modifiers)) {
                return 1;
            }
            if (Modifier.isProtected(o1.modifiers) && !Modifier.isProtected(o2.modifiers)) {
                return -1;
            }
            if (!Modifier.isProtected(o1.modifiers) && Modifier.isProtected(o2.modifiers)) {
                return 1;
            }
            if (Modifier.isPrivate(o1.modifiers) && !Modifier.isPrivate(o2.modifiers)) {
                return -1;
            }
            if (!Modifier.isPrivate(o1.modifiers) && Modifier.isPrivate(o2.modifiers)) {
                return 1;
            }
            return 0;
        }
    };
}