/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.logging.log4j;

import org.jboss.logging.AbstractLoggerPluginInstance;
import org.jboss.logging.LoggerPluginInstance;
import org.jboss.logging.Logger;
import org.jboss.logging.LoggerPlugin;

import org.apache.log4j.Category;
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;

import java.util.Map;
import java.util.EnumMap;

/**
 * Delegate for org.jboss.logging.Logger logging to log4j. Body of implementation
 * mainly copied from old Logger implementation.
 *
 * @see org.jboss.logging.Logger
 * @see org.jboss.logging.LoggerPlugin
 *
 * @author  <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 * @author Jason T. Greene
 * @version $Revision: 3404 $
 */
public class Log4jLoggerPluginInstance extends AbstractLoggerPluginInstance implements LoggerPluginInstance
{

   // Constants -----------------------------------------------------

   // Attributes ----------------------------------------------------

   /** The Log4j delegate logger. */
   private org.apache.log4j.Logger log;

   // Static --------------------------------------------------------

   // Constructors --------------------------------------------------

   public Log4jLoggerPluginInstance(final String name, final String resourceBundleName, final LoggerPlugin loggerPlugin)
   {
      super(name, resourceBundleName, loggerPlugin);
      log = LogManager.getLogger(name);
   }

   // Public --------------------------------------------------------

   public Category getCategory()
   {
      return log;
   }

   /**
    * Exposes the delegate Log4j Logger.
    * 
    * @return the underlying logger
    */
   public org.apache.log4j.Logger getDelegateLogger()
   {
      return log;
   }

   // LoggerPlugin implementation ----------------------------------------------

   private static final Map<Logger.Level, Level> LEVELS;

   static {
      final EnumMap<Logger.Level, Level> map = new EnumMap<Logger.Level, Level>(Logger.Level.class);
      map.put(Logger.Level.TRACE, Level.TRACE);
      map.put(Logger.Level.DEBUG, Level.DEBUG);
      map.put(Logger.Level.INFO, Level.INFO);
      map.put(Logger.Level.WARN, Level.WARN);
      map.put(Logger.Level.ERROR, Level.ERROR);
      map.put(Logger.Level.FATAL, Level.FATAL);
      LEVELS = map;
   }

   public boolean isEnabled(final Logger.Level level) {
      final Level l = LEVELS.get(level);
      return log.isEnabledFor(l) && l.isGreaterOrEqual(log.getEffectiveLevel());
   }

   protected void log(final Logger.Level level, final String loggerFqcn, final String message, final Throwable t) {
      log.log(loggerFqcn, LEVELS.get(level), message, t);
   }
}
