/*      */ package org.quartz.core;
/*      */ 
/*      */ import java.io.IOException;
/*      */ import java.io.InputStream;
/*      */ import java.rmi.NotBoundException;
/*      */ import java.rmi.RemoteException;
/*      */ import java.rmi.registry.LocateRegistry;
/*      */ import java.rmi.registry.Registry;
/*      */ import java.rmi.server.UnicastRemoteObject;
/*      */ import java.util.ArrayList;
/*      */ import java.util.Collections;
/*      */ import java.util.Date;
/*      */ import java.util.HashMap;
/*      */ import java.util.Iterator;
/*      */ import java.util.LinkedList;
/*      */ import java.util.List;
/*      */ import java.util.Properties;
/*      */ import java.util.Random;
/*      */ import java.util.Set;
/*      */ import org.apache.commons.logging.Log;
/*      */ import org.apache.commons.logging.LogFactory;
/*      */ import org.quartz.Calendar;
/*      */ import org.quartz.InterruptableJob;
/*      */ import org.quartz.Job;
/*      */ import org.quartz.JobDataMap;
/*      */ import org.quartz.JobDetail;
/*      */ import org.quartz.JobExecutionContext;
/*      */ import org.quartz.JobExecutionException;
/*      */ import org.quartz.JobListener;
/*      */ import org.quartz.JobPersistenceException;
/*      */ import org.quartz.ObjectAlreadyExistsException;
/*      */ import org.quartz.SchedulerContext;
/*      */ import org.quartz.SchedulerException;
/*      */ import org.quartz.SchedulerListener;
/*      */ import org.quartz.SimpleTrigger;
/*      */ import org.quartz.Trigger;
/*      */ import org.quartz.TriggerListener;
/*      */ import org.quartz.UnableToInterruptJobException;
/*      */ import org.quartz.impl.SchedulerRepository;
/*      */ import org.quartz.simpl.SimpleJobFactory;
/*      */ import org.quartz.spi.JobFactory;
/*      */ import org.quartz.spi.JobStore;
/*      */ import org.quartz.spi.SchedulerPlugin;
/*      */ import org.quartz.spi.SchedulerSignaler;
/*      */ import org.quartz.spi.ThreadPool;
/*      */ 
/*      */ public class QuartzScheduler
/*      */   implements RemotableQuartzScheduler
/*      */ {
/*   89 */   private static String VERSION_MAJOR = "UNKNOWN";
/*   90 */   private static String VERSION_MINOR = "UNKNOWN";
/*   91 */   private static String VERSION_ITERATION = "UNKNOWN";
/*      */   private QuartzSchedulerResources resources;
/*      */   private QuartzSchedulerThread schedThread;
/*      */   private ThreadGroup threadGroup;
/*  124 */   private SchedulerContext context = new SchedulerContext();
/*      */ 
/*  126 */   private HashMap jobListeners = new HashMap(10);
/*      */ 
/*  128 */   private ArrayList globalJobListeners = new ArrayList(10);
/*      */ 
/*  130 */   private HashMap triggerListeners = new HashMap(10);
/*      */ 
/*  132 */   private ArrayList globalTriggerListeners = new ArrayList(10);
/*      */ 
/*  134 */   private ArrayList schedulerListeners = new ArrayList(10);
/*      */ 
/*  136 */   private ArrayList schedulerPlugins = new ArrayList(10);
/*      */ 
/*  138 */   private JobFactory jobFactory = new SimpleJobFactory();
/*      */ 
/*  140 */   ExecutingJobsManager jobMgr = null;
/*      */ 
/*  142 */   ErrorLogger errLogger = null;
/*      */   private SchedulerSignaler signaler;
/*  146 */   private Random random = new Random();
/*      */ 
/*  148 */   private ArrayList holdToPreventGC = new ArrayList(5);
/*      */ 
/*  150 */   private boolean signalOnSchedulingChange = true;
/*      */ 
/*  152 */   private boolean closed = false;
/*      */ 
/*  154 */   private Date initialStart = null;
/*      */ 
/*      */   public QuartzScheduler(QuartzSchedulerResources resources, SchedulingContext ctxt, long idleWaitTime, long dbRetryInterval)
/*      */     throws SchedulerException
/*      */   {
/*  175 */     this.resources = resources;
/*      */     try {
/*  177 */       bind();
/*      */     } catch (Exception re) {
/*  179 */       throw new SchedulerException("Unable to bind scheduler to RMI Registry.", re);
/*      */     }
/*      */ 
/*  183 */     this.schedThread = new QuartzSchedulerThread(this, resources, ctxt);
/*  184 */     if (idleWaitTime > 0L) this.schedThread.setIdleWaitTime(idleWaitTime);
/*  185 */     if (dbRetryInterval > 0L) {
/*  186 */       this.schedThread.setDbFailureRetryInterval(dbRetryInterval);
/*      */     }
/*  188 */     this.jobMgr = new ExecutingJobsManager();
/*  189 */     addGlobalJobListener(this.jobMgr);
/*  190 */     this.errLogger = new ErrorLogger();
/*  191 */     addSchedulerListener(this.errLogger);
/*      */ 
/*  193 */     this.signaler = new SchedulerSignalerImpl(this);
/*      */ 
/*  195 */     getLog().info("Quartz Scheduler v." + getVersion() + " created.");
/*      */   }
/*      */ 
/*      */   public String getVersion()
/*      */   {
/*  207 */     return getVersionMajor() + "." + getVersionMinor() + "." + getVersionIteration();
/*      */   }
/*      */ 
/*      */   public static String getVersionMajor()
/*      */   {
/*  212 */     return VERSION_MAJOR;
/*      */   }
/*      */ 
/*      */   public static String getVersionMinor() {
/*  216 */     return VERSION_MINOR;
/*      */   }
/*      */ 
/*      */   public static String getVersionIteration() {
/*  220 */     return VERSION_ITERATION;
/*      */   }
/*      */ 
/*      */   public SchedulerSignaler getSchedulerSignaler() {
/*  224 */     return this.signaler;
/*      */   }
/*      */ 
/*      */   public static Log getLog() {
/*  228 */     return LogFactory.getLog(QuartzScheduler.class);
/*      */   }
/*      */ 
/*      */   private void bind()
/*      */     throws RemoteException
/*      */   {
/*  237 */     String host = this.resources.getRMIRegistryHost();
/*      */ 
/*  239 */     if ((host == null) || (host.length() == 0)) return;
/*      */ 
/*  241 */     RemotableQuartzScheduler exportable = null;
/*      */ 
/*  243 */     if (this.resources.getRMIServerPort() > 0) {
/*  244 */       exportable = (RemotableQuartzScheduler)UnicastRemoteObject.exportObject(this, this.resources.getRMIServerPort());
/*      */     }
/*      */     else {
/*  247 */       exportable = (RemotableQuartzScheduler)UnicastRemoteObject.exportObject(this);
/*      */     }
/*      */ 
/*  250 */     Registry registry = null;
/*      */ 
/*  252 */     if (this.resources.getRMICreateRegistryStrategy().equals("as_needed"))
/*      */     {
/*      */       try
/*      */       {
/*  260 */         registry = LocateRegistry.getRegistry(this.resources.getRMIRegistryPort());
/*      */ 
/*  262 */         registry.list();
/*      */       } catch (Exception e) {
/*  264 */         registry = LocateRegistry.createRegistry(this.resources.getRMIRegistryPort());
/*      */       }
/*      */     }
/*  267 */     else if (this.resources.getRMICreateRegistryStrategy().equals("always")) {
/*      */       try
/*      */       {
/*  270 */         registry = LocateRegistry.createRegistry(this.resources.getRMIRegistryPort());
/*      */       }
/*      */       catch (Exception e)
/*      */       {
/*  278 */         registry = LocateRegistry.getRegistry(this.resources.getRMIRegistryPort());
/*      */       }
/*      */     }
/*      */     else {
/*  282 */       registry = LocateRegistry.getRegistry(this.resources.getRMIRegistryHost(), this.resources.getRMIRegistryPort());
/*      */     }
/*      */ 
/*  286 */     registry.rebind(this.resources.getUniqueIdentifier(), exportable);
/*      */ 
/*  288 */     getLog().info("Scheduler bound to RMI registry.");
/*      */   }
/*      */ 
/*      */   private void unBind()
/*      */     throws RemoteException
/*      */   {
/*  297 */     String host = this.resources.getRMIRegistryHost();
/*      */ 
/*  299 */     if ((host == null) || (host.length() == 0)) return;
/*      */ 
/*  301 */     Registry registry = LocateRegistry.getRegistry(this.resources.getRMIRegistryHost(), this.resources.getRMIRegistryPort());
/*      */     try
/*      */     {
/*  305 */       registry.unbind(this.resources.getUniqueIdentifier());
/*  306 */       UnicastRemoteObject.unexportObject(this, true);
/*      */     }
/*      */     catch (NotBoundException nbe) {
/*      */     }
/*  310 */     getLog().info("Scheduler un-bound from RMI registry.");
/*      */   }
/*      */ 
/*      */   public String getSchedulerName()
/*      */   {
/*  319 */     return this.resources.getName();
/*      */   }
/*      */ 
/*      */   public String getSchedulerInstanceId()
/*      */   {
/*  328 */     return this.resources.getInstanceId();
/*      */   }
/*      */ 
/*      */   public ThreadGroup getSchedulerThreadGroup()
/*      */   {
/*  337 */     if (this.threadGroup == null) {
/*  338 */       this.threadGroup = new ThreadGroup("QuartzScheduler:" + getSchedulerName());
/*      */     }
/*      */ 
/*  342 */     return this.threadGroup;
/*      */   }
/*      */ 
/*      */   public void addNoGCObject(Object obj) {
/*  346 */     this.holdToPreventGC.add(obj);
/*      */   }
/*      */ 
/*      */   public boolean removeNoGCObject(Object obj) {
/*  350 */     return this.holdToPreventGC.remove(obj);
/*      */   }
/*      */ 
/*      */   public SchedulerContext getSchedulerContext()
/*      */     throws SchedulerException
/*      */   {
/*  359 */     return this.context;
/*      */   }
/*      */ 
/*      */   public boolean isSignalOnSchedulingChange() {
/*  363 */     return this.signalOnSchedulingChange;
/*      */   }
/*      */ 
/*      */   public void setSignalOnSchedulingChange(boolean signalOnSchedulingChange) {
/*  367 */     this.signalOnSchedulingChange = signalOnSchedulingChange;
/*      */   }
/*      */ 
/*      */   public void start()
/*      */     throws SchedulerException
/*      */   {
/*  388 */     if (this.closed) {
/*  389 */       throw new SchedulerException("The Scheduler cannot be restarted after shutdown() has been called.");
/*      */     }
/*      */ 
/*  392 */     this.schedThread.togglePause(false);
/*      */ 
/*  394 */     if (this.initialStart == null) {
/*  395 */       this.initialStart = new Date();
/*  396 */       this.resources.getJobStore().schedulerStarted();
/*  397 */       startPlugins();
/*      */     }
/*      */ 
/*  400 */     getLog().info("Scheduler " + this.resources.getUniqueIdentifier() + " started.");
/*      */   }
/*      */ 
/*      */   public void standby()
/*      */   {
/*  414 */     this.schedThread.togglePause(true);
/*  415 */     getLog().info("Scheduler " + this.resources.getUniqueIdentifier() + " paused.");
/*      */   }
/*      */ 
/*      */   public boolean isInStandbyMode()
/*      */   {
/*  425 */     return this.schedThread.isPaused();
/*      */   }
/*      */ 
/*      */   public Date runningSince() {
/*  429 */     return this.initialStart;
/*      */   }
/*      */ 
/*      */   public int numJobsExecuted() {
/*  433 */     return this.jobMgr.getNumJobsFired();
/*      */   }
/*      */ 
/*      */   public Class getJobStoreClass() {
/*  437 */     return this.resources.getJobStore().getClass();
/*      */   }
/*      */ 
/*      */   public boolean supportsPersistence() {
/*  441 */     return this.resources.getJobStore().supportsPersistence();
/*      */   }
/*      */ 
/*      */   public Class getThreadPoolClass() {
/*  445 */     return this.resources.getThreadPool().getClass();
/*      */   }
/*      */ 
/*      */   public int getThreadPoolSize() {
/*  449 */     return this.resources.getThreadPool().getPoolSize();
/*      */   }
/*      */ 
/*      */   public void shutdown()
/*      */   {
/*  464 */     shutdown(false);
/*      */   }
/*      */ 
/*      */   public void shutdown(boolean waitForJobsToComplete)
/*      */   {
/*  483 */     if (this.closed == true) {
/*  484 */       return;
/*      */     }
/*  486 */     getLog().info("Scheduler " + this.resources.getUniqueIdentifier() + " shutting down.");
/*      */ 
/*  489 */     standby();
/*      */ 
/*  491 */     this.closed = true;
/*      */ 
/*  493 */     this.schedThread.halt();
/*      */ 
/*  495 */     this.resources.getThreadPool().shutdown(waitForJobsToComplete);
/*      */ 
/*  497 */     if (waitForJobsToComplete)
/*  498 */       while (this.jobMgr.getNumJobsCurrentlyExecuting() > 0)
/*      */         try {
/*  500 */           Thread.sleep(100L);
/*      */         }
/*      */         catch (Exception ignore)
/*      */         {
/*      */         }
/*  505 */     this.resources.getJobStore().shutdown();
/*      */ 
/*  507 */     notifySchedulerListenersShutdown();
/*      */ 
/*  509 */     shutdownPlugins();
/*      */ 
/*  511 */     SchedulerRepository.getInstance().remove(this.resources.getName());
/*      */ 
/*  513 */     this.holdToPreventGC.clear();
/*      */     try
/*      */     {
/*  516 */       unBind();
/*      */     }
/*      */     catch (RemoteException re) {
/*      */     }
/*  520 */     getLog().info("Scheduler " + this.resources.getUniqueIdentifier() + " shutdown complete.");
/*      */   }
/*      */ 
/*      */   public boolean isShutdown()
/*      */   {
/*  531 */     return this.closed;
/*      */   }
/*      */ 
/*      */   public void validateState() throws SchedulerException {
/*  535 */     if (isShutdown())
/*  536 */       throw new SchedulerException("The Scheduler has been shutdown.");
/*      */   }
/*      */ 
/*      */   public List getCurrentlyExecutingJobs()
/*      */   {
/*  553 */     return this.jobMgr.getExecutingJobs();
/*      */   }
/*      */ 
/*      */   public Date scheduleJob(SchedulingContext ctxt, JobDetail jobDetail, Trigger trigger)
/*      */     throws SchedulerException
/*      */   {
/*  580 */     validateState();
/*      */ 
/*  582 */     jobDetail.validate();
/*      */ 
/*  584 */     if (trigger.getJobName() == null) {
/*  585 */       trigger.setJobName(jobDetail.getName());
/*  586 */       trigger.setJobGroup(jobDetail.getGroup()); } else {
/*  587 */       if ((trigger.getJobName() != null) && (!trigger.getJobName().equals(jobDetail.getName())))
/*      */       {
/*  589 */         throw new SchedulerException("Trigger does not reference given job!", 100);
/*      */       }
/*      */ 
/*  592 */       if ((trigger.getJobGroup() != null) && (!trigger.getJobGroup().equals(jobDetail.getGroup())))
/*      */       {
/*  594 */         throw new SchedulerException("Trigger does not reference given job!", 100);
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/*  599 */     trigger.validate();
/*      */ 
/*  601 */     Calendar cal = null;
/*  602 */     if (trigger.getCalendarName() != null) {
/*  603 */       cal = this.resources.getJobStore().retrieveCalendar(ctxt, trigger.getCalendarName());
/*      */     }
/*      */ 
/*  606 */     Date ft = trigger.computeFirstFireTime(cal);
/*      */ 
/*  608 */     if (ft == null) {
/*  609 */       throw new SchedulerException("Based on configured schedule, the given trigger will never fire.", 100);
/*      */     }
/*      */ 
/*  613 */     this.resources.getJobStore().storeJobAndTrigger(ctxt, jobDetail, trigger);
/*  614 */     notifySchedulerThread();
/*  615 */     notifySchedulerListenersSchduled(trigger);
/*      */ 
/*  617 */     return ft;
/*      */   }
/*      */ 
/*      */   public Date scheduleJob(SchedulingContext ctxt, Trigger trigger)
/*      */     throws SchedulerException
/*      */   {
/*  633 */     validateState();
/*      */ 
/*  635 */     trigger.validate();
/*      */ 
/*  637 */     Calendar cal = null;
/*  638 */     if (trigger.getCalendarName() != null) {
/*  639 */       cal = this.resources.getJobStore().retrieveCalendar(ctxt, trigger.getCalendarName());
/*      */ 
/*  641 */       if (cal == null) {
/*  642 */         throw new SchedulerException("Calendar not found: " + trigger.getCalendarName(), 420);
/*      */       }
/*      */     }
/*      */ 
/*  646 */     Date ft = trigger.computeFirstFireTime(cal);
/*      */ 
/*  648 */     if (ft == null) {
/*  649 */       throw new SchedulerException("Based on configured schedule, the given trigger will never fire.", 100);
/*      */     }
/*      */ 
/*  653 */     this.resources.getJobStore().storeTrigger(ctxt, trigger, false);
/*  654 */     notifySchedulerThread();
/*  655 */     notifySchedulerListenersSchduled(trigger);
/*      */ 
/*  657 */     return ft;
/*      */   }
/*      */ 
/*      */   public void addJob(SchedulingContext ctxt, JobDetail jobDetail, boolean replace)
/*      */     throws SchedulerException
/*      */   {
/*  680 */     validateState();
/*      */ 
/*  682 */     if ((!jobDetail.isDurable()) && (!replace)) {
/*  683 */       throw new SchedulerException("Jobs added with no trigger must be durable.", 100);
/*      */     }
/*      */ 
/*  687 */     this.resources.getJobStore().storeJob(ctxt, jobDetail, replace);
/*      */   }
/*      */ 
/*      */   public boolean deleteJob(SchedulingContext ctxt, String jobName, String groupName)
/*      */     throws SchedulerException
/*      */   {
/*  702 */     validateState();
/*      */ 
/*  704 */     if (groupName == null) {
/*  705 */       groupName = "DEFAULT";
/*      */     }
/*  707 */     return this.resources.getJobStore().removeJob(ctxt, jobName, groupName);
/*      */   }
/*      */ 
/*      */   public boolean unscheduleJob(SchedulingContext ctxt, String triggerName, String groupName)
/*      */     throws SchedulerException
/*      */   {
/*  718 */     validateState();
/*      */ 
/*  720 */     if (groupName == null) {
/*  721 */       groupName = "DEFAULT";
/*      */     }
/*  723 */     if (this.resources.getJobStore().removeTrigger(ctxt, triggerName, groupName)) {
/*  724 */       notifySchedulerThread();
/*  725 */       notifySchedulerListenersUnschduled(triggerName, groupName);
/*      */     } else {
/*  727 */       return false;
/*      */     }
/*  729 */     return true;
/*      */   }
/*      */ 
/*      */   public Date rescheduleJob(SchedulingContext ctxt, String triggerName, String groupName, Trigger newTrigger)
/*      */     throws SchedulerException
/*      */   {
/*  752 */     validateState();
/*      */ 
/*  754 */     if (groupName == null) {
/*  755 */       groupName = "DEFAULT";
/*      */     }
/*  757 */     newTrigger.validate();
/*      */ 
/*  759 */     Calendar cal = null;
/*  760 */     if (newTrigger.getCalendarName() != null) {
/*  761 */       cal = this.resources.getJobStore().retrieveCalendar(ctxt, newTrigger.getCalendarName());
/*      */     }
/*      */ 
/*  764 */     Date ft = newTrigger.computeFirstFireTime(cal);
/*      */ 
/*  766 */     if (ft == null) {
/*  767 */       throw new SchedulerException("Based on configured schedule, the given trigger will never fire.", 100);
/*      */     }
/*      */ 
/*  771 */     if (this.resources.getJobStore().replaceTrigger(ctxt, triggerName, groupName, newTrigger)) {
/*  772 */       notifySchedulerThread();
/*  773 */       notifySchedulerListenersUnschduled(triggerName, groupName);
/*  774 */       notifySchedulerListenersSchduled(newTrigger);
/*      */     } else {
/*  776 */       return null;
/*      */     }
/*  778 */     return ft;
/*      */   }
/*      */ 
/*      */   private String newTriggerId()
/*      */   {
/*  784 */     long r = this.random.nextLong();
/*  785 */     if (r < 0L) r = -r;
/*  786 */     return "MT_" + Long.toString(r, 30 + (int)(System.currentTimeMillis() % 7L));
/*      */   }
/*      */ 
/*      */   public void triggerJob(SchedulingContext ctxt, String jobName, String groupName, JobDataMap data)
/*      */     throws SchedulerException
/*      */   {
/*  798 */     validateState();
/*      */ 
/*  800 */     if (groupName == null) {
/*  801 */       groupName = "DEFAULT";
/*      */     }
/*  803 */     Trigger trig = new SimpleTrigger(newTriggerId(), "MANUAL_TRIGGER", jobName, groupName, new Date(), null, 0, 0L);
/*      */ 
/*  806 */     trig.setVolatility(false);
/*  807 */     trig.computeFirstFireTime(null);
/*  808 */     if (data != null) {
/*  809 */       trig.setJobDataMap(data);
/*      */     }
/*  811 */     boolean collision = true;
/*  812 */     while (collision) {
/*      */       try {
/*  814 */         this.resources.getJobStore().storeTrigger(ctxt, trig, false);
/*  815 */         collision = false;
/*      */       } catch (ObjectAlreadyExistsException oaee) {
/*  817 */         trig.setName(newTriggerId());
/*      */       }
/*      */     }
/*      */ 
/*  821 */     notifySchedulerThread();
/*  822 */     notifySchedulerListenersSchduled(trig);
/*      */   }
/*      */ 
/*      */   public void triggerJobWithVolatileTrigger(SchedulingContext ctxt, String jobName, String groupName, JobDataMap data)
/*      */     throws SchedulerException
/*      */   {
/*  833 */     validateState();
/*      */ 
/*  835 */     if (groupName == null) {
/*  836 */       groupName = "DEFAULT";
/*      */     }
/*  838 */     Trigger trig = new SimpleTrigger(newTriggerId(), "MANUAL_TRIGGER", jobName, groupName, new Date(), null, 0, 0L);
/*      */ 
/*  841 */     trig.setVolatility(true);
/*  842 */     trig.computeFirstFireTime(null);
/*  843 */     if (data != null) {
/*  844 */       trig.setJobDataMap(data);
/*      */     }
/*  846 */     boolean collision = true;
/*  847 */     while (collision) {
/*      */       try {
/*  849 */         this.resources.getJobStore().storeTrigger(ctxt, trig, false);
/*  850 */         collision = false;
/*      */       } catch (ObjectAlreadyExistsException oaee) {
/*  852 */         trig.setName(newTriggerId());
/*      */       }
/*      */     }
/*      */ 
/*  856 */     notifySchedulerThread();
/*  857 */     notifySchedulerListenersSchduled(trig);
/*      */   }
/*      */ 
/*      */   public void pauseTrigger(SchedulingContext ctxt, String triggerName, String groupName)
/*      */     throws SchedulerException
/*      */   {
/*  868 */     validateState();
/*      */ 
/*  870 */     if (groupName == null) {
/*  871 */       groupName = "DEFAULT";
/*      */     }
/*  873 */     this.resources.getJobStore().pauseTrigger(ctxt, triggerName, groupName);
/*  874 */     notifySchedulerThread();
/*  875 */     notifySchedulerListenersPausedTrigger(triggerName, groupName);
/*      */   }
/*      */ 
/*      */   public void pauseTriggerGroup(SchedulingContext ctxt, String groupName)
/*      */     throws SchedulerException
/*      */   {
/*  886 */     validateState();
/*      */ 
/*  888 */     if (groupName == null) {
/*  889 */       groupName = "DEFAULT";
/*      */     }
/*  891 */     this.resources.getJobStore().pauseTriggerGroup(ctxt, groupName);
/*  892 */     notifySchedulerThread();
/*  893 */     notifySchedulerListenersPausedTrigger(null, groupName);
/*      */   }
/*      */ 
/*      */   public void pauseJob(SchedulingContext ctxt, String jobName, String groupName)
/*      */     throws SchedulerException
/*      */   {
/*  905 */     validateState();
/*      */ 
/*  907 */     if (groupName == null) {
/*  908 */       groupName = "DEFAULT";
/*      */     }
/*  910 */     this.resources.getJobStore().pauseJob(ctxt, jobName, groupName);
/*  911 */     notifySchedulerThread();
/*  912 */     notifySchedulerListenersPausedJob(jobName, groupName);
/*      */   }
/*      */ 
/*      */   public void pauseJobGroup(SchedulingContext ctxt, String groupName)
/*      */     throws SchedulerException
/*      */   {
/*  924 */     validateState();
/*      */ 
/*  926 */     if (groupName == null) {
/*  927 */       groupName = "DEFAULT";
/*      */     }
/*  929 */     this.resources.getJobStore().pauseJobGroup(ctxt, groupName);
/*  930 */     notifySchedulerThread();
/*  931 */     notifySchedulerListenersPausedJob(null, groupName);
/*      */   }
/*      */ 
/*      */   public void resumeTrigger(SchedulingContext ctxt, String triggerName, String groupName)
/*      */     throws SchedulerException
/*      */   {
/*  948 */     validateState();
/*      */ 
/*  950 */     if (groupName == null) {
/*  951 */       groupName = "DEFAULT";
/*      */     }
/*  953 */     this.resources.getJobStore().resumeTrigger(ctxt, triggerName, groupName);
/*  954 */     notifySchedulerThread();
/*  955 */     notifySchedulerListenersResumedTrigger(triggerName, groupName);
/*      */   }
/*      */ 
/*      */   public void resumeTriggerGroup(SchedulingContext ctxt, String groupName)
/*      */     throws SchedulerException
/*      */   {
/*  972 */     validateState();
/*      */ 
/*  974 */     if (groupName == null) {
/*  975 */       groupName = "DEFAULT";
/*      */     }
/*  977 */     this.resources.getJobStore().resumeTriggerGroup(ctxt, groupName);
/*  978 */     notifySchedulerThread();
/*  979 */     notifySchedulerListenersResumedTrigger(null, groupName);
/*      */   }
/*      */ 
/*      */   public Set getPausedTriggerGroups(SchedulingContext ctxt) throws SchedulerException {
/*  983 */     return this.resources.getJobStore().getPausedTriggerGroups(ctxt);
/*      */   }
/*      */ 
/*      */   public void resumeJob(SchedulingContext ctxt, String jobName, String groupName)
/*      */     throws SchedulerException
/*      */   {
/* 1001 */     validateState();
/*      */ 
/* 1003 */     if (groupName == null) {
/* 1004 */       groupName = "DEFAULT";
/*      */     }
/* 1006 */     this.resources.getJobStore().resumeJob(ctxt, jobName, groupName);
/* 1007 */     notifySchedulerThread();
/* 1008 */     notifySchedulerListenersResumedJob(jobName, groupName);
/*      */   }
/*      */ 
/*      */   public void resumeJobGroup(SchedulingContext ctxt, String groupName)
/*      */     throws SchedulerException
/*      */   {
/* 1026 */     validateState();
/*      */ 
/* 1028 */     if (groupName == null) {
/* 1029 */       groupName = "DEFAULT";
/*      */     }
/* 1031 */     this.resources.getJobStore().resumeJobGroup(ctxt, groupName);
/* 1032 */     notifySchedulerThread();
/* 1033 */     notifySchedulerListenersResumedJob(null, groupName);
/*      */   }
/*      */ 
/*      */   public void pauseAll(SchedulingContext ctxt)
/*      */     throws SchedulerException
/*      */   {
/* 1052 */     validateState();
/*      */ 
/* 1054 */     this.resources.getJobStore().pauseAll(ctxt);
/* 1055 */     notifySchedulerThread();
/* 1056 */     notifySchedulerListenersPausedTrigger(null, null);
/*      */   }
/*      */ 
/*      */   public void resumeAll(SchedulingContext ctxt)
/*      */     throws SchedulerException
/*      */   {
/* 1073 */     validateState();
/*      */ 
/* 1075 */     this.resources.getJobStore().resumeAll(ctxt);
/* 1076 */     notifySchedulerThread();
/* 1077 */     notifySchedulerListenersResumedTrigger(null, null);
/*      */   }
/*      */ 
/*      */   public String[] getJobGroupNames(SchedulingContext ctxt)
/*      */     throws SchedulerException
/*      */   {
/* 1087 */     validateState();
/*      */ 
/* 1089 */     return this.resources.getJobStore().getJobGroupNames(ctxt);
/*      */   }
/*      */ 
/*      */   public String[] getJobNames(SchedulingContext ctxt, String groupName)
/*      */     throws SchedulerException
/*      */   {
/* 1100 */     validateState();
/*      */ 
/* 1102 */     if (groupName == null) {
/* 1103 */       groupName = "DEFAULT";
/*      */     }
/* 1105 */     return this.resources.getJobStore().getJobNames(ctxt, groupName);
/*      */   }
/*      */ 
/*      */   public Trigger[] getTriggersOfJob(SchedulingContext ctxt, String jobName, String groupName)
/*      */     throws SchedulerException
/*      */   {
/* 1116 */     validateState();
/*      */ 
/* 1118 */     if (groupName == null) {
/* 1119 */       groupName = "DEFAULT";
/*      */     }
/* 1121 */     return this.resources.getJobStore().getTriggersForJob(ctxt, jobName, groupName);
/*      */   }
/*      */ 
/*      */   public String[] getTriggerGroupNames(SchedulingContext ctxt)
/*      */     throws SchedulerException
/*      */   {
/* 1133 */     validateState();
/*      */ 
/* 1135 */     return this.resources.getJobStore().getTriggerGroupNames(ctxt);
/*      */   }
/*      */ 
/*      */   public String[] getTriggerNames(SchedulingContext ctxt, String groupName)
/*      */     throws SchedulerException
/*      */   {
/* 1146 */     validateState();
/*      */ 
/* 1148 */     if (groupName == null) {
/* 1149 */       groupName = "DEFAULT";
/*      */     }
/* 1151 */     return this.resources.getJobStore().getTriggerNames(ctxt, groupName);
/*      */   }
/*      */ 
/*      */   public JobDetail getJobDetail(SchedulingContext ctxt, String jobName, String jobGroup)
/*      */     throws SchedulerException
/*      */   {
/* 1162 */     validateState();
/*      */ 
/* 1164 */     if (jobGroup == null) {
/* 1165 */       jobGroup = "DEFAULT";
/*      */     }
/* 1167 */     return this.resources.getJobStore().retrieveJob(ctxt, jobName, jobGroup);
/*      */   }
/*      */ 
/*      */   public Trigger getTrigger(SchedulingContext ctxt, String triggerName, String triggerGroup)
/*      */     throws SchedulerException
/*      */   {
/* 1178 */     validateState();
/*      */ 
/* 1180 */     if (triggerGroup == null) {
/* 1181 */       triggerGroup = "DEFAULT";
/*      */     }
/* 1183 */     return this.resources.getJobStore().retrieveTrigger(ctxt, triggerName, triggerGroup);
/*      */   }
/*      */ 
/*      */   public int getTriggerState(SchedulingContext ctxt, String triggerName, String triggerGroup)
/*      */     throws SchedulerException
/*      */   {
/* 1199 */     validateState();
/*      */ 
/* 1201 */     if (triggerGroup == null) {
/* 1202 */       triggerGroup = "DEFAULT";
/*      */     }
/* 1204 */     return this.resources.getJobStore().getTriggerState(ctxt, triggerName, triggerGroup);
/*      */   }
/*      */ 
/*      */   public void addCalendar(SchedulingContext ctxt, String calName, Calendar calendar, boolean replace, boolean updateTriggers)
/*      */     throws SchedulerException
/*      */   {
/* 1220 */     validateState();
/*      */ 
/* 1222 */     this.resources.getJobStore().storeCalendar(ctxt, calName, calendar, replace, updateTriggers);
/*      */   }
/*      */ 
/*      */   public boolean deleteCalendar(SchedulingContext ctxt, String calName)
/*      */     throws SchedulerException
/*      */   {
/* 1236 */     validateState();
/*      */ 
/* 1238 */     return this.resources.getJobStore().removeCalendar(ctxt, calName);
/*      */   }
/*      */ 
/*      */   public Calendar getCalendar(SchedulingContext ctxt, String calName)
/*      */     throws SchedulerException
/*      */   {
/* 1248 */     validateState();
/*      */ 
/* 1250 */     return this.resources.getJobStore().retrieveCalendar(ctxt, calName);
/*      */   }
/*      */ 
/*      */   public String[] getCalendarNames(SchedulingContext ctxt)
/*      */     throws SchedulerException
/*      */   {
/* 1260 */     validateState();
/*      */ 
/* 1262 */     return this.resources.getJobStore().getCalendarNames(ctxt);
/*      */   }
/*      */ 
/*      */   public void addGlobalJobListener(JobListener jobListener)
/*      */   {
/* 1277 */     if ((jobListener.getName() == null) || (jobListener.getName().length() == 0))
/*      */     {
/* 1279 */       throw new IllegalArgumentException("JobListener name cannot be empty.");
/*      */     }
/*      */ 
/* 1282 */     this.globalJobListeners.add(jobListener);
/*      */   }
/*      */ 
/*      */   public void addJobListener(JobListener jobListener)
/*      */   {
/* 1291 */     if ((jobListener.getName() == null) || (jobListener.getName().length() == 0))
/*      */     {
/* 1293 */       throw new IllegalArgumentException("JobListener name cannot be empty.");
/*      */     }
/*      */ 
/* 1296 */     this.jobListeners.put(jobListener.getName(), jobListener);
/*      */   }
/*      */ 
/*      */   public boolean removeGlobalJobListener(JobListener jobListener)
/*      */   {
/* 1309 */     return this.globalJobListeners.remove(jobListener);
/*      */   }
/*      */ 
/*      */   public boolean removeJobListener(String name)
/*      */   {
/* 1322 */     Object o = this.jobListeners.remove(name);
/*      */ 
/* 1324 */     return o != null;
/*      */   }
/*      */ 
/*      */   public List getGlobalJobListeners()
/*      */   {
/* 1336 */     return new LinkedList(this.globalJobListeners);
/*      */   }
/*      */ 
/*      */   public Set getJobListenerNames()
/*      */   {
/* 1346 */     return Collections.unmodifiableSet(this.jobListeners.keySet());
/*      */   }
/*      */ 
/*      */   public JobListener getJobListener(String name)
/*      */   {
/* 1356 */     return (JobListener)this.jobListeners.get(name);
/*      */   }
/*      */ 
/*      */   public void addGlobalTriggerListener(TriggerListener triggerListener)
/*      */   {
/* 1371 */     if ((triggerListener.getName() == null) || (triggerListener.getName().length() == 0))
/*      */     {
/* 1373 */       throw new IllegalArgumentException("TriggerListener name cannot be empty.");
/*      */     }
/*      */ 
/* 1376 */     this.globalTriggerListeners.add(triggerListener);
/*      */   }
/*      */ 
/*      */   public void addTriggerListener(TriggerListener triggerListener)
/*      */   {
/* 1385 */     if ((triggerListener.getName() == null) || (triggerListener.getName().length() == 0))
/*      */     {
/* 1387 */       throw new IllegalArgumentException("TriggerListener name cannot be empty.");
/*      */     }
/*      */ 
/* 1390 */     this.triggerListeners.put(triggerListener.getName(), triggerListener);
/*      */   }
/*      */ 
/*      */   public boolean removeGlobalTriggerListener(TriggerListener triggerListener)
/*      */   {
/* 1403 */     return this.globalTriggerListeners.remove(triggerListener);
/*      */   }
/*      */ 
/*      */   public boolean removeTriggerListener(String name)
/*      */   {
/* 1416 */     Object o = this.triggerListeners.remove(name);
/*      */ 
/* 1418 */     return o != null;
/*      */   }
/*      */ 
/*      */   public List getGlobalTriggerListeners()
/*      */   {
/* 1430 */     return new LinkedList(this.globalTriggerListeners);
/*      */   }
/*      */ 
/*      */   public Set getTriggerListenerNames()
/*      */   {
/* 1440 */     return Collections.unmodifiableSet(this.triggerListeners.keySet());
/*      */   }
/*      */ 
/*      */   public TriggerListener getTriggerListener(String name)
/*      */   {
/* 1450 */     return (TriggerListener)this.triggerListeners.get(name);
/*      */   }
/*      */ 
/*      */   public void addSchedulerListener(SchedulerListener schedulerListener)
/*      */   {
/* 1460 */     this.schedulerListeners.add(schedulerListener);
/*      */   }
/*      */ 
/*      */   public boolean removeSchedulerListener(SchedulerListener schedulerListener)
/*      */   {
/* 1473 */     return this.schedulerListeners.remove(schedulerListener);
/*      */   }
/*      */ 
/*      */   public List getSchedulerListeners()
/*      */   {
/* 1483 */     return (List)this.schedulerListeners.clone();
/*      */   }
/*      */ 
/*      */   protected void notifyJobStoreJobComplete(SchedulingContext ctxt, Trigger trigger, JobDetail detail, int instCode)
/*      */     throws JobPersistenceException
/*      */   {
/* 1490 */     this.resources.getJobStore().triggeredJobComplete(ctxt, trigger, detail, instCode);
/*      */   }
/*      */ 
/*      */   protected void notifySchedulerThread()
/*      */   {
/* 1495 */     if (isSignalOnSchedulingChange()) this.schedThread.signalSchedulingChange(); 
/*      */   }
/*      */ 
/*      */   private List buildTriggerListenerList(String[] additionalLstnrs)
/*      */     throws SchedulerException
/*      */   {
/* 1500 */     List triggerListeners = getGlobalTriggerListeners();
/* 1501 */     for (int i = 0; i < additionalLstnrs.length; i++) {
/* 1502 */       TriggerListener tl = getTriggerListener(additionalLstnrs[i]);
/*      */ 
/* 1504 */       if (tl != null) triggerListeners.add(tl);
/*      */       else {
/* 1506 */         throw new SchedulerException("TriggerListener '" + additionalLstnrs[i] + "' not found.", 710);
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/* 1511 */     return triggerListeners;
/*      */   }
/*      */ 
/*      */   private List buildJobListenerList(String[] additionalLstnrs) throws SchedulerException
/*      */   {
/* 1516 */     List jobListeners = getGlobalJobListeners();
/* 1517 */     for (int i = 0; i < additionalLstnrs.length; i++) {
/* 1518 */       JobListener jl = getJobListener(additionalLstnrs[i]);
/*      */ 
/* 1520 */       if (jl != null) jobListeners.add(jl);
/*      */       else {
/* 1522 */         throw new SchedulerException("JobListener '" + additionalLstnrs[i] + "' not found.", 610);
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/* 1527 */     return jobListeners;
/*      */   }
/*      */ 
/*      */   public boolean notifyTriggerListenersFired(JobExecutionContext jec)
/*      */     throws SchedulerException
/*      */   {
/* 1533 */     List triggerListeners = buildTriggerListenerList(jec.getTrigger().getTriggerListenerNames());
/*      */ 
/* 1536 */     boolean vetoedExecution = false;
/*      */ 
/* 1539 */     Iterator itr = triggerListeners.iterator();
/* 1540 */     while (itr.hasNext()) {
/* 1541 */       TriggerListener tl = (TriggerListener)itr.next();
/*      */       try {
/* 1543 */         tl.triggerFired(jec.getTrigger(), jec);
/*      */ 
/* 1545 */         if (tl.vetoJobExecution(jec.getTrigger(), jec))
/* 1546 */           vetoedExecution = true;
/*      */       } catch (Exception e) {
/* 1548 */         SchedulerException se = new SchedulerException("TriggerListener '" + tl.getName() + "' threw exception: " + e.getMessage(), e);
/*      */ 
/* 1551 */         se.setErrorCode(700);
/* 1552 */         throw se;
/*      */       }
/*      */     }
/*      */ 
/* 1556 */     return vetoedExecution;
/*      */   }
/*      */ 
/*      */   public void notifyTriggerListenersMisfired(Trigger trigger)
/*      */     throws SchedulerException
/*      */   {
/* 1563 */     List triggerListeners = buildTriggerListenerList(trigger.getTriggerListenerNames());
/*      */ 
/* 1567 */     Iterator itr = triggerListeners.iterator();
/* 1568 */     while (itr.hasNext()) {
/* 1569 */       TriggerListener tl = (TriggerListener)itr.next();
/*      */       try {
/* 1571 */         tl.triggerMisfired(trigger);
/*      */       } catch (Exception e) {
/* 1573 */         SchedulerException se = new SchedulerException("TriggerListener '" + tl.getName() + "' threw exception: " + e.getMessage(), e);
/*      */ 
/* 1576 */         se.setErrorCode(700);
/* 1577 */         throw se;
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifyTriggerListenersComplete(JobExecutionContext jec, int instCode)
/*      */     throws SchedulerException
/*      */   {
/* 1585 */     List triggerListeners = buildTriggerListenerList(jec.getTrigger().getTriggerListenerNames());
/*      */ 
/* 1589 */     Iterator itr = triggerListeners.iterator();
/* 1590 */     while (itr.hasNext()) {
/* 1591 */       TriggerListener tl = (TriggerListener)itr.next();
/*      */       try {
/* 1593 */         tl.triggerComplete(jec.getTrigger(), jec, instCode);
/*      */       } catch (Exception e) {
/* 1595 */         SchedulerException se = new SchedulerException("TriggerListener '" + tl.getName() + "' threw exception: " + e.getMessage(), e);
/*      */ 
/* 1598 */         se.setErrorCode(700);
/* 1599 */         throw se;
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifyJobListenersToBeExecuted(JobExecutionContext jec)
/*      */     throws SchedulerException
/*      */   {
/* 1607 */     List jobListeners = buildJobListenerList(jec.getJobDetail().getJobListenerNames());
/*      */ 
/* 1611 */     Iterator itr = jobListeners.iterator();
/* 1612 */     while (itr.hasNext()) {
/* 1613 */       JobListener jl = (JobListener)itr.next();
/*      */       try {
/* 1615 */         jl.jobToBeExecuted(jec);
/*      */       } catch (Exception e) {
/* 1617 */         SchedulerException se = new SchedulerException("JobListener '" + jl.getName() + "' threw exception: " + e.getMessage(), e);
/*      */ 
/* 1620 */         se.setErrorCode(600);
/* 1621 */         throw se;
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifyJobListenersWasVetoed(JobExecutionContext jec)
/*      */     throws SchedulerException
/*      */   {
/* 1629 */     List jobListeners = buildJobListenerList(jec.getJobDetail().getJobListenerNames());
/*      */ 
/* 1633 */     Iterator itr = jobListeners.iterator();
/* 1634 */     while (itr.hasNext()) {
/* 1635 */       JobListener jl = (JobListener)itr.next();
/*      */       try {
/* 1637 */         jl.jobExecutionVetoed(jec);
/*      */       } catch (Exception e) {
/* 1639 */         SchedulerException se = new SchedulerException("JobListener '" + jl.getName() + "' threw exception: " + e.getMessage(), e);
/*      */ 
/* 1642 */         se.setErrorCode(600);
/* 1643 */         throw se;
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifyJobListenersWasExecuted(JobExecutionContext jec, JobExecutionException je)
/*      */     throws SchedulerException
/*      */   {
/* 1651 */     List jobListeners = buildJobListenerList(jec.getJobDetail().getJobListenerNames());
/*      */ 
/* 1655 */     Iterator itr = jobListeners.iterator();
/* 1656 */     while (itr.hasNext()) {
/* 1657 */       JobListener jl = (JobListener)itr.next();
/*      */       try {
/* 1659 */         jl.jobWasExecuted(jec, je);
/*      */       } catch (Exception e) {
/* 1661 */         SchedulerException se = new SchedulerException("JobListener '" + jl.getName() + "' threw exception: " + e.getMessage(), e);
/*      */ 
/* 1664 */         se.setErrorCode(600);
/* 1665 */         throw se;
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifySchedulerListenersError(String msg, SchedulerException se)
/*      */   {
/* 1672 */     List schedListeners = getSchedulerListeners();
/*      */ 
/* 1675 */     Iterator itr = schedListeners.iterator();
/* 1676 */     while (itr.hasNext()) {
/* 1677 */       SchedulerListener sl = (SchedulerListener)itr.next();
/*      */       try {
/* 1679 */         sl.schedulerError(msg, se);
/*      */       } catch (Exception e) {
/* 1681 */         getLog().error("Error while notifying SchedulerListener of error: ", e);
/*      */ 
/* 1685 */         getLog().error("  Original error (for notification) was: " + msg, se);
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifySchedulerListenersSchduled(Trigger trigger)
/*      */   {
/* 1693 */     List schedListeners = getSchedulerListeners();
/*      */ 
/* 1696 */     Iterator itr = schedListeners.iterator();
/* 1697 */     while (itr.hasNext()) {
/* 1698 */       SchedulerListener sl = (SchedulerListener)itr.next();
/*      */       try {
/* 1700 */         sl.jobScheduled(trigger);
/*      */       } catch (Exception e) {
/* 1702 */         getLog().error("Error while notifying SchedulerListener of scheduled job.  Triger=" + trigger.getFullName(), e);
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifySchedulerListenersUnschduled(String triggerName, String triggerGroup)
/*      */   {
/* 1712 */     List schedListeners = getSchedulerListeners();
/*      */ 
/* 1715 */     Iterator itr = schedListeners.iterator();
/* 1716 */     while (itr.hasNext()) {
/* 1717 */       SchedulerListener sl = (SchedulerListener)itr.next();
/*      */       try {
/* 1719 */         sl.jobUnscheduled(triggerName, triggerGroup);
/*      */       } catch (Exception e) {
/* 1721 */         getLog().error("Error while notifying SchedulerListener of unscheduled job.  Triger=" + triggerGroup + "." + triggerName, e);
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifySchedulerListenersFinalized(Trigger trigger)
/*      */   {
/* 1731 */     List schedListeners = getSchedulerListeners();
/*      */ 
/* 1734 */     Iterator itr = schedListeners.iterator();
/* 1735 */     while (itr.hasNext()) {
/* 1736 */       SchedulerListener sl = (SchedulerListener)itr.next();
/*      */       try {
/* 1738 */         sl.triggerFinalized(trigger);
/*      */       } catch (Exception e) {
/* 1740 */         getLog().error("Error while notifying SchedulerListener of finalized trigger.  Triger=" + trigger.getFullName(), e);
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifySchedulerListenersPausedTrigger(String name, String group)
/*      */   {
/* 1749 */     List schedListeners = getSchedulerListeners();
/*      */ 
/* 1752 */     Iterator itr = schedListeners.iterator();
/* 1753 */     while (itr.hasNext()) {
/* 1754 */       SchedulerListener sl = (SchedulerListener)itr.next();
/*      */       try {
/* 1756 */         sl.triggersPaused(name, group);
/*      */       } catch (Exception e) {
/* 1758 */         getLog().error("Error while notifying SchedulerListener of paused trigger/group.  Triger=" + group + "." + name, e);
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifySchedulerListenersResumedTrigger(String name, String group)
/*      */   {
/* 1767 */     List schedListeners = getSchedulerListeners();
/*      */ 
/* 1770 */     Iterator itr = schedListeners.iterator();
/* 1771 */     while (itr.hasNext()) {
/* 1772 */       SchedulerListener sl = (SchedulerListener)itr.next();
/*      */       try {
/* 1774 */         sl.triggersResumed(name, group);
/*      */       } catch (Exception e) {
/* 1776 */         getLog().error("Error while notifying SchedulerListener of resumed trigger/group.  Triger=" + group + "." + name, e);
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifySchedulerListenersPausedJob(String name, String group)
/*      */   {
/* 1785 */     List schedListeners = getSchedulerListeners();
/*      */ 
/* 1788 */     Iterator itr = schedListeners.iterator();
/* 1789 */     while (itr.hasNext()) {
/* 1790 */       SchedulerListener sl = (SchedulerListener)itr.next();
/*      */       try {
/* 1792 */         sl.jobsPaused(name, group);
/*      */       } catch (Exception e) {
/* 1794 */         getLog().error("Error while notifying SchedulerListener of paused job/group.  Job=" + group + "." + name, e);
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifySchedulerListenersResumedJob(String name, String group)
/*      */   {
/* 1803 */     List schedListeners = getSchedulerListeners();
/*      */ 
/* 1806 */     Iterator itr = schedListeners.iterator();
/* 1807 */     while (itr.hasNext()) {
/* 1808 */       SchedulerListener sl = (SchedulerListener)itr.next();
/*      */       try {
/* 1810 */         sl.jobsResumed(name, group);
/*      */       } catch (Exception e) {
/* 1812 */         getLog().error("Error while notifying SchedulerListener of resumed job/group.  Job=" + group + "." + name, e);
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void notifySchedulerListenersShutdown()
/*      */   {
/* 1821 */     List schedListeners = getSchedulerListeners();
/*      */ 
/* 1824 */     Iterator itr = schedListeners.iterator();
/* 1825 */     while (itr.hasNext()) {
/* 1826 */       SchedulerListener sl = (SchedulerListener)itr.next();
/*      */       try {
/* 1828 */         sl.schedulerShutdown();
/*      */       } catch (Exception e) {
/* 1830 */         getLog().error("Error while notifying SchedulerListener of shutdown.", e);
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public void addSchedulerPlugin(SchedulerPlugin plugin)
/*      */   {
/* 1844 */     this.schedulerPlugins.add(plugin);
/*      */   }
/*      */ 
/*      */   public void setJobFactory(JobFactory factory)
/*      */     throws SchedulerException
/*      */   {
/* 1850 */     if (factory == null) {
/* 1851 */       throw new IllegalArgumentException("JobFactory cannot be set to null!");
/*      */     }
/* 1853 */     getLog().info("JobFactory set to: " + factory);
/*      */ 
/* 1855 */     this.jobFactory = factory;
/*      */   }
/*      */ 
/*      */   public JobFactory getJobFactory() {
/* 1859 */     return this.jobFactory;
/*      */   }
/*      */ 
/*      */   public boolean interrupt(SchedulingContext ctxt, String jobName, String groupName)
/*      */     throws UnableToInterruptJobException
/*      */   {
/* 1870 */     if (groupName == null) {
/* 1871 */       groupName = "DEFAULT";
/*      */     }
/* 1873 */     List jobs = getCurrentlyExecutingJobs();
/* 1874 */     Iterator it = jobs.iterator();
/*      */ 
/* 1876 */     JobExecutionContext jec = null;
/* 1877 */     JobDetail jobDetail = null;
/* 1878 */     Job job = null;
/*      */ 
/* 1880 */     boolean interrupted = false;
/*      */ 
/* 1882 */     while (it.hasNext()) {
/* 1883 */       jec = (JobExecutionContext)it.next();
/* 1884 */       jobDetail = jec.getJobDetail();
/* 1885 */       if ((!jobName.equals(jobDetail.getName())) || (!groupName.equals(jobDetail.getGroup())))
/*      */         continue;
/* 1887 */       job = jec.getJobInstance();
/* 1888 */       if ((job instanceof InterruptableJob)) {
/* 1889 */         ((InterruptableJob)job).interrupt();
/* 1890 */         interrupted = true; continue;
/*      */       }
/* 1892 */       throw new UnableToInterruptJobException("Job '" + jobName + "' of group '" + groupName + "' can not be interrupted, since it does not implement " + InterruptableJob.class.getName());
/*      */     }
/*      */ 
/* 1904 */     return interrupted;
/*      */   }
/*      */ 
/*      */   private void shutdownPlugins() {
/* 1908 */     Iterator itr = this.schedulerPlugins.iterator();
/* 1909 */     while (itr.hasNext()) {
/* 1910 */       SchedulerPlugin plugin = (SchedulerPlugin)itr.next();
/* 1911 */       plugin.shutdown();
/*      */     }
/*      */   }
/*      */ 
/*      */   private void startPlugins() {
/* 1916 */     Iterator itr = this.schedulerPlugins.iterator();
/* 1917 */     while (itr.hasNext()) {
/* 1918 */       SchedulerPlugin plugin = (SchedulerPlugin)itr.next();
/* 1919 */       plugin.start();
/*      */     }
/*      */   }
/*      */ 
/*      */   static
/*      */   {
/*   94 */     Properties props = new Properties();
/*      */     try {
/*   96 */       InputStream is = QuartzScheduler.class.getResourceAsStream("/build.properties");
/*      */ 
/*   98 */       if (is != null) {
/*   99 */         props.load(is);
/*  100 */         VERSION_MAJOR = props.getProperty("version.major");
/*  101 */         VERSION_MINOR = props.getProperty("version.minor");
/*  102 */         VERSION_ITERATION = props.getProperty("version.iter");
/*      */       }
/*      */     } catch (IOException e) {
/*  105 */       getLog().error("Error loading version info from build.properties.", e);
/*      */     }
/*      */   }
/*      */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.quartz.core.QuartzScheduler
 * JD-Core Version:    0.6.0
 */