package org.hsqldb;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import org.hsqldb.jdbc.jdbcConnection;
import org.hsqldb.lib.ArrayUtil;
import org.hsqldb.lib.HashMappedList;
import org.hsqldb.lib.HsqlArrayList;
import org.hsqldb.lib.IntKeyHashMap;
import org.hsqldb.lib.SimpleLog;
import org.hsqldb.lib.java.JavaSystem;
import org.hsqldb.persist.Logger;
import org.hsqldb.store.ValuePool;

public class Session
  implements SessionInterface
{
  private volatile boolean isAutoCommit;
  private volatile boolean isReadOnly;
  private volatile boolean isClosed;
  Database database;
  private User user;
  HsqlArrayList rowActionList;
  private boolean isNestedTransaction;
  private int nestedOldTransIndex;
  int isolationMode = 2;
  long actionTimestamp;
  long transactionTimestamp;
  private int currentMaxRows;
  private int sessionMaxRows;
  private Number lastIdentity = ValuePool.getInt(0);
  private final int sessionId;
  HashMappedList savepoints;
  private boolean script;
  private Tokenizer tokenizer;
  private Parser parser;
  static final Result emptyUpdateCount = new Result(1);
  private jdbcConnection intConnection;
  public HsqlNameManager.HsqlName currentSchema;
  public HsqlNameManager.HsqlName loggedSchema;
  private HsqlNameManager.HsqlName oldSchema;
  boolean isProcessingScript;
  boolean isProcessingLog;
  private IntKeyHashMap indexArrayMap;
  private IntKeyHashMap indexArrayKeepMap;
  private final long connectTime = System.currentTimeMillis();
  DatabaseCommandInterpreter dbCommandInterpreter;
  CompiledStatementExecutor compiledStatementExecutor;
  CompiledStatementManager compiledStatementManager;
  long currentDateTimeSCN;
  long currentMillis;
  Date currentDate;
  Time currentTime;
  Timestamp currentTimestamp;
  HsqlArrayList sqlWarnings;

  Session getSession()
  {
    return this;
  }

  Session(Database paramDatabase, User paramUser, boolean paramBoolean1, boolean paramBoolean2, int paramInt)
  {
    this.sessionId = paramInt;
    this.database = paramDatabase;
    this.user = paramUser;
    this.rowActionList = new HsqlArrayList(true);
    this.savepoints = new HashMappedList(4);
    this.isAutoCommit = paramBoolean1;
    this.isReadOnly = paramBoolean2;
    this.dbCommandInterpreter = new DatabaseCommandInterpreter(this);
    this.compiledStatementExecutor = new CompiledStatementExecutor(this);
    this.compiledStatementManager = paramDatabase.compiledStatementManager;
    this.tokenizer = new Tokenizer();
    this.parser = new Parser(this, this.database, this.tokenizer);
    resetSchema();
  }

  void resetSchema()
  {
    HsqlNameManager.HsqlName localHsqlName = this.user.getInitialSchema();
    this.currentSchema = (localHsqlName == null ? this.database.schemaManager.getDefaultSchemaHsqlName() : localHsqlName);
  }

  public int getId()
  {
    return this.sessionId;
  }

  public void close()
  {
    if (this.isClosed)
      return;
    synchronized (this.database)
    {
      if (this.isClosed)
        return;
      this.database.sessionManager.removeSession(this);
      rollback();
      try
      {
        this.database.logger.writeToLog(this, "DISCONNECT");
      }
      catch (HsqlException localHsqlException)
      {
      }
      clearIndexRoots();
      clearIndexRootsKeep();
      this.compiledStatementManager.removeSession(this.sessionId);
      this.database.closeIfLast();
      this.database = null;
      this.user = null;
      this.rowActionList = null;
      this.savepoints = null;
      this.intConnection = null;
      this.compiledStatementExecutor = null;
      this.compiledStatementManager = null;
      this.dbCommandInterpreter = null;
      this.lastIdentity = null;
      this.isClosed = true;
    }
  }

  public boolean isClosed()
  {
    return this.isClosed;
  }

  public void setIsolation(int paramInt)
    throws HsqlException
  {
    this.isolationMode = paramInt;
  }

  public int getIsolation()
    throws HsqlException
  {
    return this.isolationMode;
  }

  void setLastIdentity(Number paramNumber)
  {
    this.lastIdentity = paramNumber;
  }

  Number getLastIdentity()
  {
    return this.lastIdentity;
  }

  Database getDatabase()
  {
    return this.database;
  }

  String getUsername()
  {
    return this.user.getName();
  }

  public User getUser()
  {
    return this.user;
  }

  void setUser(User paramUser)
  {
    this.user = paramUser;
  }

  int getMaxRows()
  {
    return this.currentMaxRows;
  }

  int getSQLMaxRows()
  {
    return this.sessionMaxRows;
  }

  void setSQLMaxRows(int paramInt)
  {
    this.currentMaxRows = (this.sessionMaxRows = paramInt);
  }

  void checkAdmin()
    throws HsqlException
  {
    this.user.checkAdmin();
  }

  void check(HsqlNameManager.HsqlName paramHsqlName, int paramInt)
    throws HsqlException
  {
    this.user.check(paramHsqlName, paramInt);
  }

  void check(String paramString)
    throws HsqlException
  {
    this.user.check(paramString);
  }

  void checkReadWrite()
    throws HsqlException
  {
    if (this.isReadOnly)
      throw Trace.error(31);
  }

  void checkDDLWrite()
    throws HsqlException
  {
    if ((this.database.isFilesReadOnly()) && (!this.user.isSys()))
      throw Trace.error(31);
    checkReadWrite();
  }

  boolean addDeleteAction(Table paramTable, Row paramRow)
    throws HsqlException
  {
    if ((!this.isAutoCommit) || (this.isNestedTransaction))
    {
      Transaction localTransaction = new Transaction(true, paramTable, paramRow, this.actionTimestamp);
      this.rowActionList.add(localTransaction);
      this.database.txManager.addTransaction(this, localTransaction);
      return true;
    }
    paramTable.removeRowFromStore(paramRow);
    return false;
  }

  boolean addInsertAction(Table paramTable, Row paramRow)
    throws HsqlException
  {
    if ((!this.isAutoCommit) || (this.isNestedTransaction))
    {
      Transaction localTransaction = new Transaction(false, paramTable, paramRow, this.actionTimestamp);
      this.rowActionList.add(localTransaction);
      this.database.txManager.addTransaction(this, localTransaction);
      return true;
    }
    paramTable.commitRowToStore(paramRow);
    return false;
  }

  public void setAutoCommit(boolean paramBoolean)
  {
    if (this.isClosed)
      return;
    synchronized (this.database)
    {
      if (paramBoolean != this.isAutoCommit)
      {
        commit();
        this.isAutoCommit = paramBoolean;
        try
        {
          this.database.logger.writeToLog(this, getAutoCommitStatement());
        }
        catch (HsqlException localHsqlException)
        {
        }
      }
    }
  }

  public void startPhasedTransaction()
    throws HsqlException
  {
  }

  public void prepareCommit()
    throws HsqlException
  {
  }

  public void commit()
  {
    if (this.isClosed)
      return;
    synchronized (this.database)
    {
      if (!this.rowActionList.isEmpty())
        try
        {
          this.database.logger.writeCommitStatement(this);
        }
        catch (HsqlException localHsqlException)
        {
        }
      this.database.txManager.commit(this);
      clearIndexRoots();
    }
  }

  public void rollback()
  {
    if (this.isClosed)
      return;
    synchronized (this.database)
    {
      if (this.rowActionList.size() != 0)
        try
        {
          this.database.logger.writeToLog(this, "ROLLBACK");
        }
        catch (HsqlException localHsqlException)
        {
        }
      this.database.txManager.rollback(this);
      clearIndexRoots();
    }
  }

  public void resetSession()
    throws HsqlException
  {
    throw new HsqlException("", "", 0);
  }

  void savepoint(String paramString)
    throws HsqlException
  {
    this.savepoints.remove(paramString);
    this.savepoints.add(paramString, ValuePool.getInt(this.rowActionList.size()));
    try
    {
      this.database.logger.writeToLog(this, "SAVEPOINT " + paramString);
    }
    catch (HsqlException localHsqlException)
    {
    }
  }

  void rollbackToSavepoint(String paramString)
    throws HsqlException
  {
    if (this.isClosed)
      return;
    try
    {
      this.database.logger.writeToLog(this, "ROLLBACK TO SAVEPOINT " + paramString);
    }
    catch (HsqlException localHsqlException)
    {
    }
    this.database.txManager.rollbackSavepoint(this, paramString);
  }

  void releaseSavepoint(String paramString)
    throws HsqlException
  {
    int i = this.savepoints.getIndex(paramString);
    Trace.check(i >= 0, 44, paramString);
    while (this.savepoints.size() > i)
      this.savepoints.remove(this.savepoints.size() - 1);
  }

  void beginNestedTransaction()
    throws HsqlException
  {
    if (this.isNestedTransaction)
      Trace.doAssert(false, "beginNestedTransaction");
    this.nestedOldTransIndex = this.rowActionList.size();
    this.isNestedTransaction = true;
    if (this.isAutoCommit)
      try
      {
        this.database.logger.writeToLog(this, "SET AUTOCOMMIT FALSE");
      }
      catch (HsqlException localHsqlException)
      {
      }
  }

  void endNestedTransaction(boolean paramBoolean)
    throws HsqlException
  {
    if (!this.isNestedTransaction)
      Trace.doAssert(false, "endNestedTransaction");
    if (paramBoolean)
      this.database.txManager.rollbackTransactions(this, this.nestedOldTransIndex, true);
    this.isNestedTransaction = false;
    if (this.isAutoCommit)
    {
      this.database.txManager.commit(this);
      try
      {
        this.database.logger.writeToLog(this, "SET AUTOCOMMIT TRUE");
      }
      catch (HsqlException localHsqlException)
      {
      }
    }
  }

  public void setReadOnly(boolean paramBoolean)
    throws HsqlException
  {
    if ((!paramBoolean) && (this.database.databaseReadOnly))
      throw Trace.error(31);
    this.isReadOnly = paramBoolean;
  }

  public boolean isReadOnly()
  {
    return this.isReadOnly;
  }

  boolean isNestedTransaction()
  {
    return this.isNestedTransaction;
  }

  public boolean isAutoCommit()
  {
    return this.isAutoCommit;
  }

  void setScripting(boolean paramBoolean)
  {
    this.script = paramBoolean;
  }

  boolean getScripting()
  {
    return this.script;
  }

  public String getAutoCommitStatement()
  {
    return this.isAutoCommit ? "SET AUTOCOMMIT TRUE" : "SET AUTOCOMMIT FALSE";
  }

  jdbcConnection getInternalConnection()
    throws HsqlException
  {
    if (this.intConnection == null)
      this.intConnection = new jdbcConnection(this);
    return this.intConnection;
  }

  boolean isAdmin()
  {
    return this.user.isAdmin();
  }

  long getConnectTime()
  {
    return this.connectTime;
  }

  int getTransactionSize()
  {
    return this.rowActionList.size();
  }

  boolean isAccessible(String paramString)
    throws HsqlException
  {
    return this.user.isAccessible(paramString);
  }

  boolean isAccessible(HsqlNameManager.HsqlName paramHsqlName)
    throws HsqlException
  {
    return this.user.isAccessible(paramHsqlName);
  }

  CompiledStatement sqlCompileStatement(String paramString)
    throws HsqlException
  {
    this.parser.reset(paramString);
    int i = 0;
    String str = this.tokenizer.getString();
    int j = Token.get(str);
    CompiledStatement localCompiledStatement;
    switch (j)
    {
    case 313:
      i = this.parser.parseOpenBracketsSelect() + 1;
    case 193:
      localCompiledStatement = this.parser.compileSelectStatement(i);
      break;
    case 112:
      localCompiledStatement = this.parser.compileInsertStatement();
      break;
    case 222:
      localCompiledStatement = this.parser.compileUpdateStatement();
      break;
    case 62:
      localCompiledStatement = this.parser.compileDeleteStatement();
      break;
    case 23:
      localCompiledStatement = this.parser.compileCallStatement();
      break;
    default:
      localCompiledStatement = new CompiledStatement(this.currentSchema);
    }
    while ((localCompiledStatement.type != 9) && (this.tokenizer.getPosition() < this.tokenizer.getLength()))
    {
      str = this.tokenizer.getString();
      if ((str.length() == 0) || (str.equals(";")))
        continue;
      throw Trace.error(11, str);
    }
    localCompiledStatement.sql = paramString;
    return localCompiledStatement;
  }

  public Result execute(Result paramResult)
  {
    try
    {
      if (this.isClosed)
        Trace.check(false, 33, Trace.getMessage(222));
    }
    catch (Throwable localThrowable1)
    {
      return new Result(localThrowable1, null);
    }
    synchronized (this.database)
    {
      int i = paramResult.mode;
      if (this.sessionMaxRows == 0)
        this.currentMaxRows = paramResult.updateCount;
      this.actionTimestamp = this.database.txManager.nextActionTimestamp();
      JavaSystem.gc();
      Object localObject1;
      switch (i)
      {
      case 65548:
        localObject1 = sqlExecute(paramResult);
        localObject1 = performPostExecute((Result)localObject1);
        return localObject1;
      case 9:
        localObject1 = sqlExecuteBatch(paramResult);
        localObject1 = performPostExecute((Result)localObject1);
        return localObject1;
      case 65547:
        localObject1 = sqlExecuteDirectNoPreChecks(paramResult.getMainString());
        localObject1 = performPostExecute((Result)localObject1);
        return localObject1;
      case 8:
        localObject1 = sqlExecuteBatchDirect(paramResult);
        localObject1 = performPostExecute((Result)localObject1);
        return localObject1;
      case 65555:
        try
        {
          localObject1 = this.compiledStatementManager.compile(this, paramResult.getMainString());
        }
        catch (Throwable localThrowable5)
        {
          return new Result(localThrowable5, paramResult.getMainString());
        }
        Result localResult1 = ((CompiledStatement)localObject1).describeResult();
        Result localResult2 = ((CompiledStatement)localObject1).describeParameters();
        return Result.newPrepareResponse(((CompiledStatement)localObject1).id, localResult1, localResult2);
      case 65552:
        this.compiledStatementManager.freeStatement(paramResult.getStatementID(), this.sessionId, false);
        return emptyUpdateCount;
      case 7:
        return getAttributes();
      case 6:
        return setAttributes(paramResult);
      case 66541:
        switch (paramResult.getEndTranType())
        {
        case 0:
          commit();
          break;
        case 1:
          rollback();
          break;
        case 4:
          try
          {
            localObject1 = paramResult.getMainString();
            releaseSavepoint((String)localObject1);
          }
          catch (Throwable localThrowable2)
          {
            return new Result(localThrowable2, null);
          }
        case 2:
          try
          {
            rollbackToSavepoint(paramResult.getMainString());
          }
          catch (Throwable localThrowable3)
          {
            return new Result(localThrowable3, null);
          }
        case 3:
        }
        return emptyUpdateCount;
      case 66552:
        switch (paramResult.getConnectionAttrType())
        {
        case 10027:
          try
          {
            savepoint(paramResult.getMainString());
          }
          catch (Throwable localThrowable4)
          {
            return new Result(localThrowable4, null);
          }
        }
        return emptyUpdateCount;
      case 65545:
        close();
        return emptyUpdateCount;
      }
      return new Result(Trace.runtimeError(201, "Session.execute()"), null);
    }
  }

  private Result performPostExecute(Result paramResult)
  {
    try
    {
      if (this.database != null)
      {
        this.database.schemaManager.logSequences(this, this.database.logger);
        if (this.isAutoCommit)
        {
          clearIndexRoots();
          this.database.logger.synchLog();
        }
      }
      localResult1 = paramResult;
    }
    catch (Exception localException)
    {
      Result localResult1;
      Result localResult2 = new Result(localException, null);
      return localResult2;
    }
    finally
    {
      if ((this.database != null) && (this.database.logger.needsCheckpoint()))
        try
        {
          this.database.logger.checkpoint(false);
        }
        catch (HsqlException localHsqlException)
        {
          this.database.logger.appLog.logContext(SimpleLog.LOG_ERROR, "checkpoint did not complete");
        }
    }
  }

  public Result sqlExecuteDirectNoPreChecks(String paramString)
  {
    synchronized (this.database)
    {
      return this.dbCommandInterpreter.execute(paramString);
    }
  }

  Result sqlExecuteCompiledNoPreChecks(CompiledStatement paramCompiledStatement, Object[] paramArrayOfObject)
  {
    return this.compiledStatementExecutor.execute(paramCompiledStatement, paramArrayOfObject);
  }

  private Result sqlExecuteBatch(Result paramResult)
  {
    int i = paramResult.getStatementID();
    CompiledStatement localCompiledStatement = this.database.compiledStatementManager.getStatement(this, i);
    if (localCompiledStatement == null)
      return new Result(Trace.runtimeError(203, null), null);
    Expression[] arrayOfExpression = localCompiledStatement.parameters;
    int j = 0;
    int[] arrayOfInt = new int[paramResult.getSize()];
    for (Record localRecord = paramResult.rRoot; localRecord != null; localRecord = localRecord.next)
    {
      Object[] arrayOfObject = localRecord.data;
      Result localResult2 = sqlExecuteCompiledNoPreChecks(localCompiledStatement, arrayOfObject);
      if (localResult2.mode == 1)
      {
        arrayOfInt[(j++)] = localResult2.updateCount;
      }
      else if (localResult2.isData())
      {
        arrayOfInt[(j++)] = -2;
      }
      else
      {
        arrayOfInt = ArrayUtil.arraySlice(arrayOfInt, 0, j);
        break;
      }
    }
    Result localResult1 = new Result(65548, arrayOfInt, 0);
    return localResult1;
  }

  private Result sqlExecuteBatchDirect(Result paramResult)
  {
    int i = 0;
    int[] arrayOfInt = new int[paramResult.getSize()];
    for (Record localRecord = paramResult.rRoot; localRecord != null; localRecord = localRecord.next)
    {
      String str = (String)localRecord.data[0];
      Result localResult2;
      try
      {
        localResult2 = this.dbCommandInterpreter.execute(str);
      }
      catch (Throwable localThrowable)
      {
        localResult2 = new Result(2);
      }
      if (localResult2.mode == 1)
      {
        arrayOfInt[(i++)] = localResult2.updateCount;
      }
      else if (localResult2.isData())
      {
        arrayOfInt[(i++)] = -2;
      }
      else
      {
        arrayOfInt = ArrayUtil.arraySlice(arrayOfInt, 0, i);
        break;
      }
    }
    Result localResult1 = new Result(65548, arrayOfInt, 0);
    return localResult1;
  }

  private Result sqlExecute(Result paramResult)
  {
    int i = paramResult.getStatementID();
    CompiledStatement localCompiledStatement = this.compiledStatementManager.getStatement(this, i);
    if (localCompiledStatement == null)
      return new Result(Trace.runtimeError(203, null), null);
    Object[] arrayOfObject = paramResult.getParameterData();
    return sqlExecute(localCompiledStatement, arrayOfObject);
  }

  private Result sqlExecute(CompiledStatement paramCompiledStatement, Object[] paramArrayOfObject)
  {
    return sqlExecuteCompiledNoPreChecks(paramCompiledStatement, paramArrayOfObject);
  }

  Date getCurrentDate()
  {
    if (this.currentDateTimeSCN != this.actionTimestamp)
    {
      this.currentDateTimeSCN = this.actionTimestamp;
      this.currentMillis = System.currentTimeMillis();
      this.currentDate = HsqlDateTime.getCurrentDate(this.currentMillis);
      this.currentTime = null;
      this.currentTimestamp = null;
    }
    else if (this.currentDate == null)
    {
      this.currentDate = HsqlDateTime.getCurrentDate(this.currentMillis);
    }
    return this.currentDate;
  }

  Time getCurrentTime()
  {
    if (this.currentDateTimeSCN != this.actionTimestamp)
    {
      this.currentDateTimeSCN = this.actionTimestamp;
      this.currentMillis = System.currentTimeMillis();
      this.currentDate = null;
      this.currentTime = new Time(HsqlDateTime.getNormalisedTime(this.currentMillis));
      this.currentTimestamp = null;
    }
    else if (this.currentTime == null)
    {
      this.currentTime = new Time(HsqlDateTime.getNormalisedTime(this.currentMillis));
    }
    return this.currentTime;
  }

  Timestamp getCurrentTimestamp()
  {
    if (this.currentDateTimeSCN != this.actionTimestamp)
    {
      this.currentDateTimeSCN = this.actionTimestamp;
      this.currentMillis = System.currentTimeMillis();
      this.currentDate = null;
      this.currentTime = null;
      this.currentTimestamp = HsqlDateTime.getTimestamp(this.currentMillis);
    }
    else if (this.currentTimestamp == null)
    {
      this.currentTimestamp = HsqlDateTime.getTimestamp(this.currentMillis);
    }
    return this.currentTimestamp;
  }

  Result getAttributes()
  {
    Result localResult = Result.newSessionAttributesResult();
    Object[] arrayOfObject = { this.database.getURI(), getUsername(), ValuePool.getInt(this.sessionId), ValuePool.getInt(this.isolationMode), ValuePool.getBoolean(this.isAutoCommit), ValuePool.getBoolean(this.database.databaseReadOnly), ValuePool.getBoolean(this.isReadOnly) };
    localResult.add(arrayOfObject);
    return localResult;
  }

  Result setAttributes(Result paramResult)
  {
    Object[] arrayOfObject = paramResult.rRoot.data;
    for (int i = 0; i < arrayOfObject.length; i++)
    {
      Object localObject = arrayOfObject[i];
      if (localObject == null)
        continue;
      try
      {
        switch (i)
        {
        case 4:
          setAutoCommit(((Boolean)localObject).booleanValue());
          break;
        case 6:
          setReadOnly(((Boolean)localObject).booleanValue());
        }
      }
      catch (HsqlException localHsqlException)
      {
        return new Result(localHsqlException, null);
      }
    }
    return emptyUpdateCount;
  }

  public String getInternalConnectionURL()
  {
    return "jdbc:hsqldb:" + this.database.getURI();
  }

  boolean isProcessingScript()
  {
    return this.isProcessingScript;
  }

  boolean isProcessingLog()
  {
    return this.isProcessingLog;
  }

  boolean isSchemaDefintion()
  {
    return this.oldSchema != null;
  }

  void startSchemaDefinition(String paramString)
    throws HsqlException
  {
    if (this.isProcessingScript)
    {
      setSchema(paramString);
      return;
    }
    this.oldSchema = this.currentSchema;
    setSchema(paramString);
  }

  void endSchemaDefinition()
    throws HsqlException
  {
    if (this.oldSchema == null)
      return;
    this.currentSchema = this.oldSchema;
    this.oldSchema = null;
    this.database.logger.writeToLog(this, "SET SCHEMA " + this.currentSchema.statementName);
  }

  public void setSchema(String paramString)
    throws HsqlException
  {
    this.currentSchema = this.database.schemaManager.getSchemaHsqlName(paramString);
  }

  HsqlNameManager.HsqlName getSchemaHsqlName(String paramString)
    throws HsqlException
  {
    return paramString == null ? this.currentSchema : this.database.schemaManager.getSchemaHsqlName(paramString);
  }

  public String getSchemaName(String paramString)
    throws HsqlException
  {
    return paramString == null ? this.currentSchema.name : this.database.schemaManager.getSchemaName(paramString);
  }

  HsqlNameManager.HsqlName getSchemaHsqlNameForWrite(String paramString)
    throws HsqlException
  {
    HsqlNameManager.HsqlName localHsqlName = getSchemaHsqlName(paramString);
    if (this.database.schemaManager.isSystemSchema(localHsqlName))
      throw Trace.error(227);
    return localHsqlName;
  }

  public String getSchemaNameForWrite(String paramString)
    throws HsqlException
  {
    HsqlNameManager.HsqlName localHsqlName = getSchemaHsqlNameForWrite(paramString);
    return localHsqlName.name;
  }

  Node getIndexRoot(HsqlNameManager.HsqlName paramHsqlName, boolean paramBoolean)
  {
    if (paramBoolean)
    {
      if (this.indexArrayKeepMap == null)
        return null;
      return (Node)this.indexArrayKeepMap.get(paramHsqlName.hashCode());
    }
    if (this.indexArrayMap == null)
      return null;
    return (Node)this.indexArrayMap.get(paramHsqlName.hashCode());
  }

  void setIndexRoot(HsqlNameManager.HsqlName paramHsqlName, boolean paramBoolean, Node paramNode)
  {
    if (paramBoolean)
    {
      if (this.indexArrayKeepMap == null)
      {
        if (paramNode == null)
          return;
        this.indexArrayKeepMap = new IntKeyHashMap();
      }
      this.indexArrayKeepMap.put(paramHsqlName.hashCode(), paramNode);
    }
    else
    {
      if (this.indexArrayMap == null)
      {
        if (paramNode == null)
          return;
        this.indexArrayMap = new IntKeyHashMap();
      }
      this.indexArrayMap.put(paramHsqlName.hashCode(), paramNode);
    }
  }

  void dropIndex(HsqlNameManager.HsqlName paramHsqlName, boolean paramBoolean)
  {
    if (paramBoolean)
    {
      if (this.indexArrayKeepMap != null)
        this.indexArrayKeepMap.remove(paramHsqlName.hashCode());
    }
    else if (this.indexArrayMap != null)
      this.indexArrayMap.remove(paramHsqlName.hashCode());
  }

  void clearIndexRoots()
  {
    if (this.indexArrayMap != null)
      this.indexArrayMap.clear();
  }

  void clearIndexRootsKeep()
  {
    if (this.indexArrayKeepMap != null)
      this.indexArrayKeepMap.clear();
  }

  public void addWarning(HsqlException paramHsqlException)
  {
    if (this.sqlWarnings == null)
      this.sqlWarnings = new HsqlArrayList(true);
    this.sqlWarnings.add(paramHsqlException);
  }

  public HsqlException[] getAndClearWarnings()
  {
    if (this.sqlWarnings == null)
      return new HsqlException[0];
    HsqlException[] arrayOfHsqlException = new HsqlException[this.sqlWarnings.size()];
    this.sqlWarnings.toArray(arrayOfHsqlException);
    this.sqlWarnings.clear();
    return arrayOfHsqlException;
  }
}

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.hsqldb.Session
 * JD-Core Version:    0.6.0
 */