/*      */ package org.apache.commons.logging;
/*      */ 
/*      */ import java.io.BufferedReader;
/*      */ import java.io.FileOutputStream;
/*      */ import java.io.IOException;
/*      */ import java.io.InputStream;
/*      */ import java.io.InputStreamReader;
/*      */ import java.io.PrintStream;
/*      */ import java.io.UnsupportedEncodingException;
/*      */ import java.lang.reflect.InvocationTargetException;
/*      */ import java.lang.reflect.Method;
/*      */ import java.net.URL;
/*      */ import java.security.AccessController;
/*      */ import java.security.PrivilegedAction;
/*      */ import java.util.Enumeration;
/*      */ import java.util.Hashtable;
/*      */ import java.util.Properties;
/*      */ 
/*      */ public abstract class LogFactory
/*      */ {
/*      */   public static final String PRIORITY_KEY = "priority";
/*      */   public static final String TCCL_KEY = "use_tccl";
/*      */   public static final String FACTORY_PROPERTY = "org.apache.commons.logging.LogFactory";
/*      */   public static final String FACTORY_DEFAULT = "org.apache.commons.logging.impl.LogFactoryImpl";
/*      */   public static final String FACTORY_PROPERTIES = "commons-logging.properties";
/*      */   protected static final String SERVICE_ID = "META-INF/services/org.apache.commons.logging.LogFactory";
/*      */   public static final String DIAGNOSTICS_DEST_PROPERTY = "org.apache.commons.logging.diagnostics.dest";
/*  123 */   private static PrintStream diagnosticsStream = null;
/*      */   private static String diagnosticPrefix;
/*      */   public static final String HASHTABLE_IMPLEMENTATION_PROPERTY = "org.apache.commons.logging.LogFactory.HashtableImpl";
/*      */   private static final String WEAK_HASHTABLE_CLASSNAME = "org.apache.commons.logging.impl.WeakHashtable";
/*      */   private static ClassLoader thisClassLoader;
/*  284 */   protected static Hashtable factories = null;
/*      */ 
/*  300 */   protected static LogFactory nullClassLoaderFactory = null;
/*      */ 
/*      */   public abstract Object getAttribute(String paramString);
/*      */ 
/*      */   public abstract String[] getAttributeNames();
/*      */ 
/*      */   public abstract Log getInstance(Class paramClass) throws LogConfigurationException;
/*      */ 
/*      */   public abstract Log getInstance(String paramString) throws LogConfigurationException;
/*      */ 
/*      */   public abstract void release();
/*      */ 
/*      */   public abstract void removeAttribute(String paramString);
/*      */ 
/*      */   public abstract void setAttribute(String paramString, Object paramObject);
/*      */ 
/*  318 */   private static final Hashtable createFactoryStore() { Hashtable result = null;
/*  319 */     String storeImplementationClass = System.getProperty("org.apache.commons.logging.LogFactory.HashtableImpl");
/*      */ 
/*  321 */     if (storeImplementationClass == null)
/*  322 */       storeImplementationClass = "org.apache.commons.logging.impl.WeakHashtable";
/*      */     try
/*      */     {
/*  325 */       Class implementationClass = Class.forName(storeImplementationClass);
/*  326 */       result = (Hashtable)implementationClass.newInstance();
/*      */     }
/*      */     catch (Throwable t)
/*      */     {
/*  330 */       if (!"org.apache.commons.logging.impl.WeakHashtable".equals(storeImplementationClass))
/*      */       {
/*  332 */         if (isDiagnosticsEnabled())
/*      */         {
/*  334 */           logDiagnostic("[ERROR] LogFactory: Load of custom hashtable failed");
/*      */         }
/*      */         else
/*      */         {
/*  338 */           System.err.println("[ERROR] LogFactory: Load of custom hashtable failed");
/*      */         }
/*      */       }
/*      */     }
/*  342 */     if (result == null) {
/*  343 */       result = new Hashtable();
/*      */     }
/*  345 */     return result;
/*      */   }
/*      */ 
/*      */   public static LogFactory getFactory()
/*      */     throws LogConfigurationException
/*      */   {
/*  383 */     ClassLoader contextClassLoader = getContextClassLoader();
/*      */ 
/*  385 */     if (contextClassLoader == null)
/*      */     {
/*  389 */       if (isDiagnosticsEnabled()) {
/*  390 */         logDiagnostic("Context classloader is null.");
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/*  395 */     LogFactory factory = getCachedFactory(contextClassLoader);
/*  396 */     if (factory != null) {
/*  397 */       return factory;
/*      */     }
/*      */ 
/*  400 */     if (isDiagnosticsEnabled()) {
/*  401 */       logDiagnostic("[LOOKUP] LogFactory implementation requested for the first time for context classloader " + objectId(contextClassLoader));
/*      */ 
/*  404 */       logHierarchy("[LOOKUP] ", contextClassLoader);
/*      */     }
/*      */ 
/*  417 */     Properties props = getConfigurationFile(contextClassLoader, "commons-logging.properties");
/*      */ 
/*  421 */     ClassLoader baseClassLoader = contextClassLoader;
/*  422 */     if (props != null) {
/*  423 */       String useTCCLStr = props.getProperty("use_tccl");
/*  424 */       if (useTCCLStr != null)
/*      */       {
/*  427 */         if (!Boolean.valueOf(useTCCLStr).booleanValue())
/*      */         {
/*  435 */           baseClassLoader = thisClassLoader;
/*      */         }
/*      */ 
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/*  442 */     if (isDiagnosticsEnabled()) {
/*  443 */       logDiagnostic("[LOOKUP] Looking for system property [org.apache.commons.logging.LogFactory] to define the LogFactory subclass to use...");
/*      */     }
/*      */ 
/*      */     try
/*      */     {
/*  449 */       String factoryClass = System.getProperty("org.apache.commons.logging.LogFactory");
/*  450 */       if (factoryClass != null) {
/*  451 */         if (isDiagnosticsEnabled()) {
/*  452 */           logDiagnostic("[LOOKUP] Creating an instance of LogFactory class '" + factoryClass + "' as specified by system property " + "org.apache.commons.logging.LogFactory");
/*      */         }
/*      */ 
/*  457 */         factory = newFactory(factoryClass, baseClassLoader, contextClassLoader);
/*      */       }
/*  459 */       else if (isDiagnosticsEnabled()) {
/*  460 */         logDiagnostic("[LOOKUP] No system property [org.apache.commons.logging.LogFactory] defined.");
/*      */       }
/*      */ 
/*      */     }
/*      */     catch (SecurityException e)
/*      */     {
/*  466 */       if (isDiagnosticsEnabled()) {
/*  467 */         logDiagnostic("[LOOKUP] A security exception occurred while trying to create an instance of the custom factory class: [" + e.getMessage().trim() + "]. Trying alternative implementations...");
/*      */       }
/*      */ 
/*      */     }
/*      */     catch (RuntimeException e)
/*      */     {
/*  480 */       if (isDiagnosticsEnabled()) {
/*  481 */         logDiagnostic("[LOOKUP] An exception occurred while trying to create an instance of the custom factory class: [" + e.getMessage().trim() + "] as specified by a system property.");
/*      */       }
/*      */ 
/*  487 */       throw e;
/*      */     }
/*      */ 
/*  497 */     if (factory == null) {
/*  498 */       if (isDiagnosticsEnabled()) {
/*  499 */         logDiagnostic("[LOOKUP] Looking for a resource file of name [META-INF/services/org.apache.commons.logging.LogFactory] to define the LogFactory subclass to use...");
/*      */       }
/*      */ 
/*      */       try
/*      */       {
/*  504 */         InputStream is = getResourceAsStream(contextClassLoader, "META-INF/services/org.apache.commons.logging.LogFactory");
/*      */ 
/*  507 */         if (is != null)
/*      */         {
/*      */           BufferedReader rd;
/*      */           try {
/*  512 */             rd = new BufferedReader(new InputStreamReader(is, "UTF-8"));
/*      */           } catch (UnsupportedEncodingException e) {
/*  514 */             rd = new BufferedReader(new InputStreamReader(is));
/*      */           }
/*      */ 
/*  517 */           String factoryClassName = rd.readLine();
/*  518 */           rd.close();
/*      */ 
/*  520 */           if ((factoryClassName != null) && (!"".equals(factoryClassName)))
/*      */           {
/*  522 */             if (isDiagnosticsEnabled()) {
/*  523 */               logDiagnostic("[LOOKUP]  Creating an instance of LogFactory class " + factoryClassName + " as specified by file '" + "META-INF/services/org.apache.commons.logging.LogFactory" + "' which was present in the path of the context" + " classloader.");
/*      */             }
/*      */ 
/*  529 */             factory = newFactory(factoryClassName, baseClassLoader, contextClassLoader);
/*      */           }
/*      */ 
/*      */         }
/*  533 */         else if (isDiagnosticsEnabled()) {
/*  534 */           logDiagnostic("[LOOKUP] No resource file with name 'META-INF/services/org.apache.commons.logging.LogFactory' found.");
/*      */         }
/*      */ 
/*      */       }
/*      */       catch (Exception ex)
/*      */       {
/*  543 */         if (isDiagnosticsEnabled()) {
/*  544 */           logDiagnostic("[LOOKUP] A security exception occurred while trying to create an instance of the custom factory class: [" + ex.getMessage().trim() + "]. Trying alternative implementations...");
/*      */         }
/*      */ 
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/*  557 */     if (factory == null) {
/*  558 */       if (props != null) {
/*  559 */         if (isDiagnosticsEnabled()) {
/*  560 */           logDiagnostic("[LOOKUP] Looking in properties file for entry with key 'org.apache.commons.logging.LogFactory' to define the LogFactory subclass to use...");
/*      */         }
/*      */ 
/*  565 */         String factoryClass = props.getProperty("org.apache.commons.logging.LogFactory");
/*  566 */         if (factoryClass != null) {
/*  567 */           if (isDiagnosticsEnabled()) {
/*  568 */             logDiagnostic("[LOOKUP] Properties file specifies LogFactory subclass '" + factoryClass + "'");
/*      */           }
/*      */ 
/*  572 */           factory = newFactory(factoryClass, baseClassLoader, contextClassLoader);
/*      */         }
/*  576 */         else if (isDiagnosticsEnabled()) {
/*  577 */           logDiagnostic("[LOOKUP] Properties file has no entry specifying LogFactory subclass.");
/*      */         }
/*      */ 
/*      */       }
/*  582 */       else if (isDiagnosticsEnabled()) {
/*  583 */         logDiagnostic("[LOOKUP] No properties file available to determine LogFactory subclass from..");
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/*  593 */     if (factory == null) {
/*  594 */       if (isDiagnosticsEnabled()) {
/*  595 */         logDiagnostic("[LOOKUP] Loading the default LogFactory implementation 'org.apache.commons.logging.impl.LogFactoryImpl' via the same classloader that loaded this LogFactory class (ie not looking in the context classloader).");
/*      */       }
/*      */ 
/*  610 */       factory = newFactory("org.apache.commons.logging.impl.LogFactoryImpl", thisClassLoader, contextClassLoader);
/*      */     }
/*      */ 
/*  613 */     if (factory != null)
/*      */     {
/*  617 */       cacheFactory(contextClassLoader, factory);
/*      */ 
/*  619 */       if (props != null) {
/*  620 */         Enumeration names = props.propertyNames();
/*  621 */         while (names.hasMoreElements()) {
/*  622 */           String name = (String)names.nextElement();
/*  623 */           String value = props.getProperty(name);
/*  624 */           factory.setAttribute(name, value);
/*      */         }
/*      */       }
/*      */     }
/*      */ 
/*  629 */     return factory;
/*      */   }
/*      */ 
/*      */   public static Log getLog(Class clazz)
/*      */     throws LogConfigurationException
/*      */   {
/*  645 */     return getFactory().getInstance(clazz);
/*      */   }
/*      */ 
/*      */   public static Log getLog(String name)
/*      */     throws LogConfigurationException
/*      */   {
/*  664 */     return getFactory().getInstance(name);
/*      */   }
/*      */ 
/*      */   public static void release(ClassLoader classLoader)
/*      */   {
/*  679 */     if (isDiagnosticsEnabled()) {
/*  680 */       logDiagnostic("Releasing factory for classloader " + objectId(classLoader));
/*      */     }
/*  682 */     synchronized (factories) {
/*  683 */       if (classLoader == null) {
/*  684 */         if (nullClassLoaderFactory != null) {
/*  685 */           nullClassLoaderFactory.release();
/*  686 */           nullClassLoaderFactory = null;
/*      */         }
/*      */       } else {
/*  689 */         LogFactory factory = (LogFactory)factories.get(classLoader);
/*  690 */         if (factory != null) {
/*  691 */           factory.release();
/*  692 */           factories.remove(classLoader);
/*      */         }
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   public static void releaseAll()
/*      */   {
/*  710 */     if (isDiagnosticsEnabled()) {
/*  711 */       logDiagnostic("Releasing factory for all classloaders.");
/*      */     }
/*  713 */     synchronized (factories) {
/*  714 */       Enumeration elements = factories.elements();
/*  715 */       while (elements.hasMoreElements()) {
/*  716 */         LogFactory element = (LogFactory)elements.nextElement();
/*  717 */         element.release();
/*      */       }
/*  719 */       factories.clear();
/*      */ 
/*  721 */       if (nullClassLoaderFactory != null) {
/*  722 */         nullClassLoaderFactory.release();
/*  723 */         nullClassLoaderFactory = null;
/*      */       }
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static ClassLoader getClassLoader(Class clazz)
/*      */   {
/*      */     try
/*      */     {
/*  756 */       return clazz.getClassLoader();
/*      */     } catch (SecurityException ex) {
/*  758 */       if (isDiagnosticsEnabled()) {
/*  759 */         logDiagnostic("Unable to get classloader for class '" + clazz + "' due to security restrictions - " + ex.getMessage());
/*      */       }
/*      */     }
/*      */ 
/*  763 */     throw ex;
/*      */   }
/*      */ 
/*      */   protected static ClassLoader getContextClassLoader()
/*      */     throws LogConfigurationException
/*      */   {
/*  788 */     return (ClassLoader)AccessController.doPrivileged(new PrivilegedAction()
/*      */     {
/*      */       public Object run() {
/*  791 */         return LogFactory.directGetContextClassLoader();
/*      */       }
/*      */     });
/*      */   }
/*      */ 
/*      */   protected static ClassLoader directGetContextClassLoader()
/*      */     throws LogConfigurationException
/*      */   {
/*  821 */     ClassLoader classLoader = null;
/*      */     try
/*      */     {
/*  825 */       Method method = Thread.class.getMethod("getContextClassLoader", (Class[])null);
/*      */       try
/*      */       {
/*  830 */         classLoader = (ClassLoader)method.invoke(Thread.currentThread(), (Object[])null);
/*      */       }
/*      */       catch (IllegalAccessException e) {
/*  833 */         throw new LogConfigurationException("Unexpected IllegalAccessException", e);
/*      */       }
/*      */       catch (InvocationTargetException e)
/*      */       {
/*  852 */         if (!(e.getTargetException() instanceof SecurityException))
/*      */         {
/*  857 */           throw new LogConfigurationException("Unexpected InvocationTargetException", e.getTargetException());
/*      */         }
/*      */       }
/*      */     }
/*      */     catch (NoSuchMethodException e)
/*      */     {
/*  863 */       classLoader = getClassLoader(LogFactory.class);
/*      */     }
/*      */ 
/*  880 */     return classLoader;
/*      */   }
/*      */ 
/*      */   private static LogFactory getCachedFactory(ClassLoader contextClassLoader)
/*      */   {
/*  899 */     LogFactory factory = null;
/*      */ 
/*  901 */     if (contextClassLoader == null)
/*      */     {
/*  906 */       factory = nullClassLoaderFactory;
/*      */     }
/*  908 */     else factory = (LogFactory)factories.get(contextClassLoader);
/*      */ 
/*  911 */     return factory;
/*      */   }
/*      */ 
/*      */   private static void cacheFactory(ClassLoader classLoader, LogFactory factory)
/*      */   {
/*  929 */     if (factory != null)
/*  930 */       if (classLoader == null)
/*  931 */         nullClassLoaderFactory = factory;
/*      */       else
/*  933 */         factories.put(classLoader, factory);
/*      */   }
/*      */ 
/*      */   protected static LogFactory newFactory(String factoryClass, ClassLoader classLoader, ClassLoader contextClassLoader)
/*      */     throws LogConfigurationException
/*      */   {
/*  991 */     Object result = AccessController.doPrivileged(new PrivilegedAction(factoryClass, classLoader) { private final String val$factoryClass;
/*      */       private final ClassLoader val$classLoader;
/*      */ 
/*  994 */       public Object run() { return LogFactory.createFactory(this.val$factoryClass, this.val$classLoader);
/*      */       }
/*      */     });
/*  998 */     if ((result instanceof LogConfigurationException)) {
/*  999 */       LogConfigurationException ex = (LogConfigurationException)result;
/* 1000 */       if (isDiagnosticsEnabled()) {
/* 1001 */         logDiagnostic("An error occurred while loading the factory class:" + ex.getMessage());
/*      */       }
/*      */ 
/* 1005 */       throw ex;
/*      */     }
/* 1007 */     if (isDiagnosticsEnabled()) {
/* 1008 */       logDiagnostic("Created object " + objectId(result) + " to manage classloader " + objectId(contextClassLoader));
/*      */     }
/*      */ 
/* 1012 */     return (LogFactory)result;
/*      */   }
/*      */ 
/*      */   protected static LogFactory newFactory(String factoryClass, ClassLoader classLoader)
/*      */   {
/* 1032 */     return newFactory(factoryClass, classLoader, null);
/*      */   }
/*      */ 
/*      */   protected static Object createFactory(String factoryClass, ClassLoader classLoader)
/*      */   {
/* 1052 */     Class logFactoryClass = null;
/*      */     try {
/* 1054 */       if (classLoader != null)
/*      */       {
/*      */         try
/*      */         {
/* 1060 */           logFactoryClass = classLoader.loadClass(factoryClass);
/* 1061 */           if (LogFactory.class.isAssignableFrom(logFactoryClass)) {
/* 1062 */             if (isDiagnosticsEnabled()) {
/* 1063 */               logDiagnostic("Loaded class " + logFactoryClass.getName() + " from classloader " + objectId(classLoader));
/*      */             }
/*      */ 
/*      */           }
/* 1079 */           else if (isDiagnosticsEnabled()) {
/* 1080 */             logDiagnostic("Factory class " + logFactoryClass.getName() + " loaded from classloader " + objectId(logFactoryClass.getClassLoader()) + " does not extend '" + LogFactory.class.getName() + "' as loaded by this classloader.");
/*      */ 
/* 1085 */             logHierarchy("[BAD CL TREE] ", classLoader);
/*      */           }
/*      */ 
/* 1089 */           return (LogFactory)logFactoryClass.newInstance();
/*      */         }
/*      */         catch (ClassNotFoundException ex) {
/* 1092 */           if (classLoader == thisClassLoader)
/*      */           {
/* 1094 */             if (isDiagnosticsEnabled()) {
/* 1095 */               logDiagnostic("Unable to locate any class called '" + factoryClass + "' via classloader " + objectId(classLoader));
/*      */             }
/*      */ 
/* 1099 */             throw ex;
/*      */           }
/*      */         }
/*      */         catch (NoClassDefFoundError e) {
/* 1103 */           if (classLoader == thisClassLoader)
/*      */           {
/* 1105 */             if (isDiagnosticsEnabled()) {
/* 1106 */               logDiagnostic("Class '" + factoryClass + "' cannot be loaded" + " via classloader " + objectId(classLoader) + " - it depends on some other class that cannot" + " be found.");
/*      */             }
/*      */ 
/* 1112 */             throw e;
/*      */           }
/*      */         }
/*      */         catch (ClassCastException e) {
/* 1116 */           if (classLoader == thisClassLoader)
/*      */           {
/* 1122 */             boolean implementsLogFactory = implementsLogFactory(logFactoryClass);
/*      */ 
/* 1129 */             String msg = "The application has specified that a custom LogFactory implementation should be used but Class '" + factoryClass + "' cannot be converted to '" + LogFactory.class.getName() + "'. ";
/*      */ 
/* 1133 */             if (implementsLogFactory) {
/* 1134 */               msg = msg + "The conflict is caused by the presence of multiple LogFactory classes in incompatible classloaders. " + "Background can be found in http://jakarta.apache.org/commons/logging/tech.html. " + "If you have not explicitly specified a custom LogFactory then it is likely that " + "the container has set one without your knowledge. " + "In this case, consider using the commons-logging-adapters.jar file or " + "specifying the standard LogFactory from the command line. ";
/*      */             }
/*      */             else
/*      */             {
/* 1141 */               msg = msg + "Please check the custom implementation. ";
/*      */             }
/* 1143 */             msg = msg + "Help can be found @http://jakarta.apache.org/commons/logging/troubleshooting.html.";
/*      */ 
/* 1145 */             if (isDiagnosticsEnabled()) {
/* 1146 */               logDiagnostic(msg);
/*      */             }
/*      */ 
/* 1149 */             ClassCastException ex = new ClassCastException(msg);
/* 1150 */             throw ex;
/*      */           }
/*      */ 
/*      */         }
/*      */ 
/*      */       }
/*      */ 
/* 1184 */       if (isDiagnosticsEnabled()) {
/* 1185 */         logDiagnostic("Unable to load factory class via classloader " + objectId(classLoader) + " - trying the classloader associated with this LogFactory.");
/*      */       }
/*      */ 
/* 1190 */       logFactoryClass = Class.forName(factoryClass);
/* 1191 */       return (LogFactory)logFactoryClass.newInstance();
/*      */     }
/*      */     catch (Exception e) {
/* 1194 */       if (isDiagnosticsEnabled()) {
/* 1195 */         logDiagnostic("Unable to create LogFactory instance.");
/*      */       }
/* 1197 */       if ((logFactoryClass != null) && (!LogFactory.class.isAssignableFrom(logFactoryClass)))
/*      */       {
/* 1200 */         return new LogConfigurationException("The chosen LogFactory implementation does not extend LogFactory. Please check your configuration.", e);
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/* 1205 */     return new LogConfigurationException(e);
/*      */   }
/*      */ 
/*      */   private static boolean implementsLogFactory(Class logFactoryClass)
/*      */   {
/* 1222 */     boolean implementsLogFactory = false;
/* 1223 */     if (logFactoryClass != null) {
/*      */       try {
/* 1225 */         ClassLoader logFactoryClassLoader = logFactoryClass.getClassLoader();
/* 1226 */         if (logFactoryClassLoader == null) {
/* 1227 */           logDiagnostic("[CUSTOM LOG FACTORY] was loaded by the boot classloader");
/*      */         } else {
/* 1229 */           logHierarchy("[CUSTOM LOG FACTORY] ", logFactoryClassLoader);
/* 1230 */           Class factoryFromCustomLoader = Class.forName("org.apache.commons.logging.LogFactory", false, logFactoryClassLoader);
/*      */ 
/* 1232 */           implementsLogFactory = factoryFromCustomLoader.isAssignableFrom(logFactoryClass);
/* 1233 */           if (implementsLogFactory) {
/* 1234 */             logDiagnostic("[CUSTOM LOG FACTORY] " + logFactoryClass.getName() + " implements LogFactory but was loaded by an incompatible classloader.");
/*      */           }
/*      */           else {
/* 1237 */             logDiagnostic("[CUSTOM LOG FACTORY] " + logFactoryClass.getName() + " does not implement LogFactory.");
/*      */           }
/*      */ 
/*      */         }
/*      */ 
/*      */       }
/*      */       catch (SecurityException e)
/*      */       {
/* 1247 */         logDiagnostic("[CUSTOM LOG FACTORY] SecurityException thrown whilst trying to determine whether the compatibility was caused by a classloader conflict: " + e.getMessage());
/*      */       }
/*      */       catch (LinkageError e)
/*      */       {
/* 1257 */         logDiagnostic("[CUSTOM LOG FACTORY] LinkageError thrown whilst trying to determine whether the compatibility was caused by a classloader conflict: " + e.getMessage());
/*      */       }
/*      */       catch (ClassNotFoundException e)
/*      */       {
/* 1268 */         logDiagnostic("[CUSTOM LOG FACTORY] LogFactory class cannot be loaded by classloader which loaded the custom LogFactory implementation. Is the custom factory in the right classloader?");
/*      */       }
/*      */     }
/*      */ 
/* 1272 */     return implementsLogFactory;
/*      */   }
/*      */ 
/*      */   private static InputStream getResourceAsStream(ClassLoader loader, String name)
/*      */   {
/* 1284 */     return (InputStream)AccessController.doPrivileged(new PrivilegedAction(loader, name) { private final ClassLoader val$loader;
/*      */       private final String val$name;
/*      */ 
/* 1287 */       public Object run() { if (this.val$loader != null) {
/* 1288 */           return this.val$loader.getResourceAsStream(this.val$name);
/*      */         }
/* 1290 */         return ClassLoader.getSystemResourceAsStream(this.val$name);
/*      */       }
/*      */     });
/*      */   }
/*      */ 
/*      */   private static Enumeration getResources(ClassLoader loader, String name)
/*      */   {
/* 1312 */     PrivilegedAction action = new PrivilegedAction(loader, name) { private final ClassLoader val$loader;
/*      */       private final String val$name;
/*      */ 
/*      */       public Object run() { try { if (this.val$loader != null) {
/* 1317 */             return this.val$loader.getResources(this.val$name);
/*      */           }
/* 1319 */           return ClassLoader.getSystemResources(this.val$name);
/*      */         } catch (IOException e)
/*      */         {
/* 1322 */           if (LogFactory.isDiagnosticsEnabled()) {
/* 1323 */             LogFactory.access$000("Exception while trying to find configuration file " + this.val$name + ":" + e.getMessage());
/*      */           }
/*      */ 
/* 1327 */           return null;
/*      */         }
/*      */         catch (NoSuchMethodError e)
/*      */         {
/*      */         }
/* 1332 */         return null;
/*      */       }
/*      */     };
/* 1336 */     Object result = AccessController.doPrivileged(action);
/* 1337 */     return (Enumeration)result;
/*      */   }
/*      */ 
/*      */   private static Properties getProperties(URL url)
/*      */   {
/* 1349 */     PrivilegedAction action = new PrivilegedAction(url) {
/*      */       private final URL val$url;
/*      */ 
/*      */       public Object run() { try { InputStream stream = this.val$url.openStream();
/* 1354 */           if (stream != null) {
/* 1355 */             Properties props = new Properties();
/* 1356 */             props.load(stream);
/* 1357 */             stream.close();
/* 1358 */             return props;
/*      */           }
/*      */         } catch (IOException e) {
/* 1361 */           if (LogFactory.isDiagnosticsEnabled()) {
/* 1362 */             LogFactory.access$000("Unable to read URL " + this.val$url);
/*      */           }
/*      */         }
/*      */ 
/* 1366 */         return null;
/*      */       }
/*      */     };
/* 1369 */     return (Properties)AccessController.doPrivileged(action);
/*      */   }
/*      */ 
/*      */   private static final Properties getConfigurationFile(ClassLoader classLoader, String fileName)
/*      */   {
/* 1394 */     Properties props = null;
/* 1395 */     double priority = 0.0D;
/* 1396 */     URL propsUrl = null;
/*      */     try {
/* 1398 */       Enumeration urls = getResources(classLoader, fileName);
/*      */ 
/* 1400 */       if (urls == null) {
/* 1401 */         return null;
/*      */       }
/*      */ 
/* 1404 */       while (urls.hasMoreElements()) {
/* 1405 */         URL url = (URL)urls.nextElement();
/*      */ 
/* 1407 */         Properties newProps = getProperties(url);
/* 1408 */         if (newProps != null) {
/* 1409 */           if (props == null) {
/* 1410 */             propsUrl = url;
/* 1411 */             props = newProps;
/* 1412 */             String priorityStr = props.getProperty("priority");
/* 1413 */             priority = 0.0D;
/* 1414 */             if (priorityStr != null) {
/* 1415 */               priority = Double.parseDouble(priorityStr);
/*      */             }
/*      */ 
/* 1418 */             if (isDiagnosticsEnabled()) {
/* 1419 */               logDiagnostic("[LOOKUP] Properties file found at '" + url + "'" + " with priority " + priority);
/*      */             }
/*      */           }
/*      */           else
/*      */           {
/* 1424 */             String newPriorityStr = newProps.getProperty("priority");
/* 1425 */             double newPriority = 0.0D;
/* 1426 */             if (newPriorityStr != null) {
/* 1427 */               newPriority = Double.parseDouble(newPriorityStr);
/*      */             }
/*      */ 
/* 1430 */             if (newPriority > priority) {
/* 1431 */               if (isDiagnosticsEnabled()) {
/* 1432 */                 logDiagnostic("[LOOKUP] Properties file at '" + url + "'" + " with priority " + newPriority + " overrides file at '" + propsUrl + "'" + " with priority " + priority);
/*      */               }
/*      */ 
/* 1439 */               propsUrl = url;
/* 1440 */               props = newProps;
/* 1441 */               priority = newPriority;
/*      */             }
/* 1443 */             else if (isDiagnosticsEnabled()) {
/* 1444 */               logDiagnostic("[LOOKUP] Properties file at '" + url + "'" + " with priority " + newPriority + " does not override file at '" + propsUrl + "'" + " with priority " + priority);
/*      */             }
/*      */ 
/*      */           }
/*      */ 
/*      */         }
/*      */ 
/*      */       }
/*      */ 
/*      */     }
/*      */     catch (SecurityException e)
/*      */     {
/* 1456 */       if (isDiagnosticsEnabled()) {
/* 1457 */         logDiagnostic("SecurityException thrown while trying to find/read config files.");
/*      */       }
/*      */     }
/*      */ 
/* 1461 */     if (isDiagnosticsEnabled()) {
/* 1462 */       if (props == null) {
/* 1463 */         logDiagnostic("[LOOKUP] No properties file of name '" + fileName + "' found.");
/*      */       }
/*      */       else
/*      */       {
/* 1467 */         logDiagnostic("[LOOKUP] Properties file of name '" + fileName + "' found at '" + propsUrl + '"');
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/* 1473 */     return props;
/*      */   }
/*      */ 
/*      */   private static void initDiagnostics()
/*      */   {
/*      */     String dest;
/*      */     try
/*      */     {
/* 1485 */       dest = System.getProperty("org.apache.commons.logging.diagnostics.dest");
/* 1486 */       if (dest == null) {
/* 1487 */         return;
/*      */       }
/*      */     }
/*      */     catch (SecurityException ex)
/*      */     {
/* 1492 */       return;
/*      */     }
/*      */ 
/* 1495 */     if (dest.equals("STDOUT"))
/* 1496 */       diagnosticsStream = System.out;
/* 1497 */     else if (dest.equals("STDERR"))
/* 1498 */       diagnosticsStream = System.err;
/*      */     else {
/*      */       try
/*      */       {
/* 1502 */         FileOutputStream fos = new FileOutputStream(dest, true);
/* 1503 */         diagnosticsStream = new PrintStream(fos);
/*      */       }
/*      */       catch (IOException ex) {
/* 1506 */         return;
/*      */       }
/*      */ 
/*      */     }
/*      */ 
/*      */     String classLoaderName;
/*      */     try
/*      */     {
/* 1521 */       ClassLoader classLoader = thisClassLoader;
/*      */       String classLoaderName;
/* 1522 */       if (thisClassLoader == null)
/* 1523 */         classLoaderName = "BOOTLOADER";
/*      */       else
/* 1525 */         classLoaderName = objectId(classLoader);
/*      */     }
/*      */     catch (SecurityException e) {
/* 1528 */       classLoaderName = "UNKNOWN";
/*      */     }
/* 1530 */     diagnosticPrefix = "[LogFactory from " + classLoaderName + "] ";
/*      */   }
/*      */ 
/*      */   protected static boolean isDiagnosticsEnabled()
/*      */   {
/* 1543 */     return diagnosticsStream != null;
/*      */   }
/*      */ 
/*      */   private static final void logDiagnostic(String msg)
/*      */   {
/* 1565 */     if (diagnosticsStream != null) {
/* 1566 */       diagnosticsStream.print(diagnosticPrefix);
/* 1567 */       diagnosticsStream.println(msg);
/* 1568 */       diagnosticsStream.flush();
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static final void logRawDiagnostic(String msg)
/*      */   {
/* 1579 */     if (diagnosticsStream != null) {
/* 1580 */       diagnosticsStream.println(msg);
/* 1581 */       diagnosticsStream.flush();
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static final void logRawDiagnostic(String msg, Throwable t) {
/* 1585 */     if (diagnosticsStream != null) {
/* 1586 */       diagnosticsStream.println(msg);
/* 1587 */       diagnosticsStream.flush();
/* 1588 */       t.printStackTrace(diagnosticsStream);
/*      */     }
/*      */   }
/*      */ 
/*      */   private static void logClassLoaderEnvironment(Class clazz)
/*      */   {
/* 1610 */     if (!isDiagnosticsEnabled()) {
/* 1611 */       return;
/*      */     }
/*      */     try
/*      */     {
/* 1615 */       logDiagnostic("[ENV] Extension directories (java.ext.dir): " + System.getProperty("java.ext.dir"));
/* 1616 */       logDiagnostic("[ENV] Application classpath (java.class.path): " + System.getProperty("java.class.path"));
/*      */     } catch (SecurityException ex) {
/* 1618 */       logDiagnostic("[ENV] Security setting prevent interrogation of system classpaths.");
/* 1621 */     }
/*      */ String className = clazz.getName();
/*      */     ClassLoader classLoader;
/*      */     try {
/* 1625 */       classLoader = getClassLoader(clazz);
/*      */     }
/*      */     catch (SecurityException ex) {
/* 1628 */       logDiagnostic("[ENV] Security forbids determining the classloader for " + className);
/*      */ 
/* 1630 */       return;
/*      */     }
/*      */ 
/* 1633 */     logDiagnostic("[ENV] Class " + className + " was loaded via classloader " + objectId(classLoader));
/*      */ 
/* 1636 */     logHierarchy("[ENV] Ancestry of classloader which loaded " + className + " is ", classLoader);
/*      */   }
/*      */ 
/*      */   private static void logHierarchy(String prefix, ClassLoader classLoader)
/*      */   {
/* 1647 */     if (!isDiagnosticsEnabled()) {
/* 1648 */       return;
/*      */     }
/*      */ 
/* 1651 */     if (classLoader != null) {
/* 1652 */       String classLoaderString = classLoader.toString();
/* 1653 */       logDiagnostic(prefix + objectId(classLoader) + " == '" + classLoaderString + "'");
/*      */     }ClassLoader systemClassLoader;
/*      */     try {
/* 1657 */       systemClassLoader = ClassLoader.getSystemClassLoader();
/*      */     } catch (SecurityException ex) {
/* 1659 */       logDiagnostic(prefix + "Security forbids determining the system classloader.");
/*      */ 
/* 1661 */       return;
/*      */     }
/* 1663 */     if (classLoader != null) {
/* 1664 */       StringBuffer buf = new StringBuffer(prefix + "ClassLoader tree:");
/*      */       while (true) {
/* 1666 */         buf.append(objectId(classLoader));
/* 1667 */         if (classLoader == systemClassLoader) {
/* 1668 */           buf.append(" (SYSTEM) ");
/*      */         }
/*      */         try
/*      */         {
/* 1672 */           classLoader = classLoader.getParent();
/*      */         } catch (SecurityException ex) {
/* 1674 */           buf.append(" --> SECRET");
/* 1675 */           break;
/*      */         }
/*      */ 
/* 1678 */         buf.append(" --> ");
/* 1679 */         if (classLoader == null) {
/* 1680 */           buf.append("BOOT");
/*      */         }
/*      */       }
/*      */ 
/* 1684 */       logDiagnostic(buf.toString());
/*      */     }
/*      */   }
/*      */ 
/*      */   public static String objectId(Object o)
/*      */   {
/* 1701 */     if (o == null) {
/* 1702 */       return "null";
/*      */     }
/* 1704 */     return o.getClass().getName() + "@" + System.identityHashCode(o);
/*      */   }
/*      */ 
/*      */   static
/*      */   {
/* 1729 */     thisClassLoader = getClassLoader(LogFactory.class);
/* 1730 */     initDiagnostics();
/* 1731 */     logClassLoaderEnvironment(LogFactory.class);
/* 1732 */     factories = createFactoryStore();
/* 1733 */     if (isDiagnosticsEnabled())
/* 1734 */       logDiagnostic("BOOTSTRAP COMPLETED");
/*      */   }
/*      */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.apache.commons.logging.LogFactory
 * JD-Core Version:    0.6.0
 */