/*      */ package org.apache.commons.collections.map;
/*      */ 
/*      */ import java.io.IOException;
/*      */ import java.io.ObjectInputStream;
/*      */ import java.io.ObjectOutputStream;
/*      */ import java.util.AbstractCollection;
/*      */ import java.util.AbstractMap;
/*      */ import java.util.AbstractSet;
/*      */ import java.util.Collection;
/*      */ import java.util.ConcurrentModificationException;
/*      */ import java.util.Iterator;
/*      */ import java.util.Map;
/*      */ import java.util.Map.Entry;
/*      */ import java.util.NoSuchElementException;
/*      */ import java.util.Set;
/*      */ import org.apache.commons.collections.IterableMap;
/*      */ import org.apache.commons.collections.KeyValue;
/*      */ import org.apache.commons.collections.MapIterator;
/*      */ import org.apache.commons.collections.iterators.EmptyIterator;
/*      */ import org.apache.commons.collections.iterators.EmptyMapIterator;
/*      */ 
/*      */ public class AbstractHashedMap extends AbstractMap
/*      */   implements IterableMap
/*      */ {
/*      */   protected static final String NO_NEXT_ENTRY = "No next() entry in the iteration";
/*      */   protected static final String NO_PREVIOUS_ENTRY = "No previous() entry in the iteration";
/*      */   protected static final String REMOVE_INVALID = "remove() can only be called once after next()";
/*      */   protected static final String GETKEY_INVALID = "getKey() can only be called after next() and before remove()";
/*      */   protected static final String GETVALUE_INVALID = "getValue() can only be called after next() and before remove()";
/*      */   protected static final String SETVALUE_INVALID = "setValue() can only be called after next() and before remove()";
/*      */   protected static final int DEFAULT_CAPACITY = 16;
/*      */   protected static final int DEFAULT_THRESHOLD = 12;
/*      */   protected static final float DEFAULT_LOAD_FACTOR = 0.75F;
/*      */   protected static final int MAXIMUM_CAPACITY = 1073741824;
/*   78 */   protected static final Object NULL = new Object();
/*      */   protected transient float loadFactor;
/*      */   protected transient int size;
/*      */   protected transient HashEntry[] data;
/*      */   protected transient int threshold;
/*      */   protected transient int modCount;
/*      */   protected transient EntrySet entrySet;
/*      */   protected transient KeySet keySet;
/*      */   protected transient Values values;
/*      */ 
/*      */   protected AbstractHashedMap()
/*      */   {
/*      */   }
/*      */ 
/*      */   protected AbstractHashedMap(int initialCapacity, float loadFactor, int threshold)
/*      */   {
/*  113 */     this.loadFactor = loadFactor;
/*  114 */     this.data = new HashEntry[initialCapacity];
/*  115 */     this.threshold = threshold;
/*  116 */     init();
/*      */   }
/*      */ 
/*      */   protected AbstractHashedMap(int initialCapacity)
/*      */   {
/*  127 */     this(initialCapacity, 0.75F);
/*      */   }
/*      */ 
/*      */   protected AbstractHashedMap(int initialCapacity, float loadFactor)
/*      */   {
/*  141 */     if (initialCapacity < 1) {
/*  142 */       throw new IllegalArgumentException("Initial capacity must be greater than 0");
/*      */     }
/*  144 */     if ((loadFactor <= 0.0F) || (Float.isNaN(loadFactor))) {
/*  145 */       throw new IllegalArgumentException("Load factor must be greater than 0");
/*      */     }
/*  147 */     this.loadFactor = loadFactor;
/*  148 */     this.threshold = calculateThreshold(initialCapacity, loadFactor);
/*  149 */     initialCapacity = calculateNewCapacity(initialCapacity);
/*  150 */     this.data = new HashEntry[initialCapacity];
/*  151 */     init();
/*      */   }
/*      */ 
/*      */   protected AbstractHashedMap(Map map)
/*      */   {
/*  161 */     this(Math.max(2 * map.size(), 16), 0.75F);
/*  162 */     putAll(map);
/*      */   }
/*      */ 
/*      */   protected void init()
/*      */   {
/*      */   }
/*      */ 
/*      */   public Object get(Object key)
/*      */   {
/*  179 */     key = convertKey(key);
/*  180 */     int hashCode = hash(key);
/*  181 */     HashEntry entry = this.data[hashIndex(hashCode, this.data.length)];
/*  182 */     while (entry != null) {
/*  183 */       if ((entry.hashCode == hashCode) && (isEqualKey(key, entry.key))) {
/*  184 */         return entry.getValue();
/*      */       }
/*  186 */       entry = entry.next;
/*      */     }
/*  188 */     return null;
/*      */   }
/*      */ 
/*      */   public int size()
/*      */   {
/*  197 */     return this.size;
/*      */   }
/*      */ 
/*      */   public boolean isEmpty()
/*      */   {
/*  206 */     return this.size == 0;
/*      */   }
/*      */ 
/*      */   public boolean containsKey(Object key)
/*      */   {
/*  217 */     key = convertKey(key);
/*  218 */     int hashCode = hash(key);
/*  219 */     HashEntry entry = this.data[hashIndex(hashCode, this.data.length)];
/*  220 */     while (entry != null) {
/*  221 */       if ((entry.hashCode == hashCode) && (isEqualKey(key, entry.key))) {
/*  222 */         return true;
/*      */       }
/*  224 */       entry = entry.next;
/*      */     }
/*  226 */     return false;
/*      */   }
/*      */ 
/*      */   public boolean containsValue(Object value)
/*      */   {
/*  236 */     if (value == null) {
/*  237 */       int i = 0; for (int isize = this.data.length; i < isize; i++) {
/*  238 */         HashEntry entry = this.data[i];
/*  239 */         while (entry != null) {
/*  240 */           if (entry.getValue() == null) {
/*  241 */             return true;
/*      */           }
/*  243 */           entry = entry.next;
/*      */         }
/*      */       }
/*      */     } else {
/*  247 */       int i = 0; for (int isize = this.data.length; i < isize; i++) {
/*  248 */         HashEntry entry = this.data[i];
/*  249 */         while (entry != null) {
/*  250 */           if (isEqualValue(value, entry.getValue())) {
/*  251 */             return true;
/*      */           }
/*  253 */           entry = entry.next;
/*      */         }
/*      */       }
/*      */     }
/*  257 */     return false;
/*      */   }
/*      */ 
/*      */   public Object put(Object key, Object value)
/*      */   {
/*  269 */     key = convertKey(key);
/*  270 */     int hashCode = hash(key);
/*  271 */     int index = hashIndex(hashCode, this.data.length);
/*  272 */     HashEntry entry = this.data[index];
/*  273 */     while (entry != null) {
/*  274 */       if ((entry.hashCode == hashCode) && (isEqualKey(key, entry.key))) {
/*  275 */         Object oldValue = entry.getValue();
/*  276 */         updateEntry(entry, value);
/*  277 */         return oldValue;
/*      */       }
/*  279 */       entry = entry.next;
/*      */     }
/*      */ 
/*  282 */     addMapping(index, hashCode, key, value);
/*  283 */     return null;
/*      */   }
/*      */ 
/*      */   public void putAll(Map map)
/*      */   {
/*  296 */     int mapSize = map.size();
/*  297 */     if (mapSize == 0) {
/*  298 */       return;
/*      */     }
/*  300 */     int newSize = (int)((this.size + mapSize) / this.loadFactor + 1.0F);
/*  301 */     ensureCapacity(calculateNewCapacity(newSize));
/*  302 */     for (Iterator it = map.entrySet().iterator(); it.hasNext(); ) {
/*  303 */       Map.Entry entry = (Map.Entry)it.next();
/*  304 */       put(entry.getKey(), entry.getValue());
/*      */     }
/*      */   }
/*      */ 
/*      */   public Object remove(Object key)
/*      */   {
/*  315 */     key = convertKey(key);
/*  316 */     int hashCode = hash(key);
/*  317 */     int index = hashIndex(hashCode, this.data.length);
/*  318 */     HashEntry entry = this.data[index];
/*  319 */     HashEntry previous = null;
/*  320 */     while (entry != null) {
/*  321 */       if ((entry.hashCode == hashCode) && (isEqualKey(key, entry.key))) {
/*  322 */         Object oldValue = entry.getValue();
/*  323 */         removeMapping(entry, index, previous);
/*  324 */         return oldValue;
/*      */       }
/*  326 */       previous = entry;
/*  327 */       entry = entry.next;
/*      */     }
/*  329 */     return null;
/*      */   }
/*      */ 
/*      */   public void clear()
/*      */   {
/*  337 */     this.modCount += 1;
/*  338 */     HashEntry[] data = this.data;
/*  339 */     for (int i = data.length - 1; i >= 0; i--) {
/*  340 */       data[i] = null;
/*      */     }
/*  342 */     this.size = 0;
/*      */   }
/*      */ 
/*      */   protected Object convertKey(Object key)
/*      */   {
/*  358 */     return key == null ? NULL : key;
/*      */   }
/*      */ 
/*      */   protected int hash(Object key)
/*      */   {
/*  371 */     int h = key.hashCode();
/*  372 */     h += (h << 9 ^ 0xFFFFFFFF);
/*  373 */     h ^= h >>> 14;
/*  374 */     h += (h << 4);
/*  375 */     h ^= h >>> 10;
/*  376 */     return h;
/*      */   }
/*      */ 
/*      */   protected boolean isEqualKey(Object key1, Object key2)
/*      */   {
/*  389 */     return (key1 == key2) || (key1.equals(key2));
/*      */   }
/*      */ 
/*      */   protected boolean isEqualValue(Object value1, Object value2)
/*      */   {
/*  402 */     return (value1 == value2) || (value1.equals(value2));
/*      */   }
/*      */ 
/*      */   protected int hashIndex(int hashCode, int dataSize)
/*      */   {
/*  415 */     return hashCode & dataSize - 1;
/*      */   }
/*      */ 
/*      */   protected HashEntry getEntry(Object key)
/*      */   {
/*  430 */     key = convertKey(key);
/*  431 */     int hashCode = hash(key);
/*  432 */     HashEntry entry = this.data[hashIndex(hashCode, this.data.length)];
/*  433 */     while (entry != null) {
/*  434 */       if ((entry.hashCode == hashCode) && (isEqualKey(key, entry.key))) {
/*  435 */         return entry;
/*      */       }
/*  437 */       entry = entry.next;
/*      */     }
/*  439 */     return null;
/*      */   }
/*      */ 
/*      */   protected void updateEntry(HashEntry entry, Object newValue)
/*      */   {
/*  453 */     entry.setValue(newValue);
/*      */   }
/*      */ 
/*      */   protected void reuseEntry(HashEntry entry, int hashIndex, int hashCode, Object key, Object value)
/*      */   {
/*  469 */     entry.next = this.data[hashIndex];
/*  470 */     entry.hashCode = hashCode;
/*  471 */     entry.key = key;
/*  472 */     entry.value = value;
/*      */   }
/*      */ 
/*      */   protected void addMapping(int hashIndex, int hashCode, Object key, Object value)
/*      */   {
/*  490 */     this.modCount += 1;
/*  491 */     HashEntry entry = createEntry(this.data[hashIndex], hashCode, key, value);
/*  492 */     addEntry(entry, hashIndex);
/*  493 */     this.size += 1;
/*  494 */     checkCapacity();
/*      */   }
/*      */ 
/*      */   protected HashEntry createEntry(HashEntry next, int hashCode, Object key, Object value)
/*      */   {
/*  511 */     return new HashEntry(next, hashCode, key, value);
/*      */   }
/*      */ 
/*      */   protected void addEntry(HashEntry entry, int hashIndex)
/*      */   {
/*  524 */     this.data[hashIndex] = entry;
/*      */   }
/*      */ 
/*      */   protected void removeMapping(HashEntry entry, int hashIndex, HashEntry previous)
/*      */   {
/*  540 */     this.modCount += 1;
/*  541 */     removeEntry(entry, hashIndex, previous);
/*  542 */     this.size -= 1;
/*  543 */     destroyEntry(entry);
/*      */   }
/*      */ 
/*      */   protected void removeEntry(HashEntry entry, int hashIndex, HashEntry previous)
/*      */   {
/*  558 */     if (previous == null)
/*  559 */       this.data[hashIndex] = entry.next;
/*      */     else
/*  561 */       previous.next = entry.next;
/*      */   }
/*      */ 
/*      */   protected void destroyEntry(HashEntry entry)
/*      */   {
/*  574 */     entry.next = null;
/*  575 */     entry.key = null;
/*  576 */     entry.value = null;
/*      */   }
/*      */ 
/*      */   protected void checkCapacity()
/*      */   {
/*  586 */     if (this.size >= this.threshold) {
/*  587 */       int newCapacity = this.data.length * 2;
/*  588 */       if (newCapacity <= 1073741824)
/*  589 */         ensureCapacity(newCapacity);
/*      */     }
/*      */   }
/*      */ 
/*      */   protected void ensureCapacity(int newCapacity)
/*      */   {
/*  600 */     int oldCapacity = this.data.length;
/*  601 */     if (newCapacity <= oldCapacity) {
/*  602 */       return;
/*      */     }
/*  604 */     if (this.size == 0) {
/*  605 */       this.threshold = calculateThreshold(newCapacity, this.loadFactor);
/*  606 */       this.data = new HashEntry[newCapacity];
/*      */     } else {
/*  608 */       HashEntry[] oldEntries = this.data;
/*  609 */       HashEntry[] newEntries = new HashEntry[newCapacity];
/*      */ 
/*  611 */       this.modCount += 1;
/*  612 */       for (int i = oldCapacity - 1; i >= 0; i--) {
/*  613 */         HashEntry entry = oldEntries[i];
/*  614 */         if (entry != null) {
/*  615 */           oldEntries[i] = null;
/*      */           do {
/*  617 */             HashEntry next = entry.next;
/*  618 */             int index = hashIndex(entry.hashCode, newCapacity);
/*  619 */             entry.next = newEntries[index];
/*  620 */             newEntries[index] = entry;
/*  621 */             entry = next;
/*  622 */           }while (entry != null);
/*      */         }
/*      */       }
/*  625 */       this.threshold = calculateThreshold(newCapacity, this.loadFactor);
/*  626 */       this.data = newEntries;
/*      */     }
/*      */   }
/*      */ 
/*      */   protected int calculateNewCapacity(int proposedCapacity)
/*      */   {
/*  638 */     int newCapacity = 1;
/*  639 */     if (proposedCapacity > 1073741824) {
/*  640 */       newCapacity = 1073741824;
/*      */     } else {
/*  642 */       while (newCapacity < proposedCapacity) {
/*  643 */         newCapacity <<= 1;
/*      */       }
/*  645 */       if (newCapacity > 1073741824) {
/*  646 */         newCapacity = 1073741824;
/*      */       }
/*      */     }
/*  649 */     return newCapacity;
/*      */   }
/*      */ 
/*      */   protected int calculateThreshold(int newCapacity, float factor)
/*      */   {
/*  661 */     return (int)(newCapacity * factor);
/*      */   }
/*      */ 
/*      */   protected HashEntry entryNext(HashEntry entry)
/*      */   {
/*  675 */     return entry.next;
/*      */   }
/*      */ 
/*      */   protected int entryHashCode(HashEntry entry)
/*      */   {
/*  688 */     return entry.hashCode;
/*      */   }
/*      */ 
/*      */   protected Object entryKey(HashEntry entry)
/*      */   {
/*  701 */     return entry.key;
/*      */   }
/*      */ 
/*      */   protected Object entryValue(HashEntry entry)
/*      */   {
/*  714 */     return entry.value;
/*      */   }
/*      */ 
/*      */   public MapIterator mapIterator()
/*      */   {
/*  730 */     if (this.size == 0) {
/*  731 */       return EmptyMapIterator.INSTANCE;
/*      */     }
/*  733 */     return new HashMapIterator(this);
/*      */   }
/*      */ 
/*      */   public Set entrySet()
/*      */   {
/*  783 */     if (this.entrySet == null) {
/*  784 */       this.entrySet = new EntrySet(this);
/*      */     }
/*  786 */     return this.entrySet;
/*      */   }
/*      */ 
/*      */   protected Iterator createEntrySetIterator()
/*      */   {
/*  796 */     if (size() == 0) {
/*  797 */       return EmptyIterator.INSTANCE;
/*      */     }
/*  799 */     return new EntrySetIterator(this);
/*      */   }
/*      */ 
/*      */   public Set keySet()
/*      */   {
/*  872 */     if (this.keySet == null) {
/*  873 */       this.keySet = new KeySet(this);
/*      */     }
/*  875 */     return this.keySet;
/*      */   }
/*      */ 
/*      */   protected Iterator createKeySetIterator()
/*      */   {
/*  885 */     if (size() == 0) {
/*  886 */       return EmptyIterator.INSTANCE;
/*      */     }
/*  888 */     return new KeySetIterator(this);
/*      */   }
/*      */ 
/*      */   public Collection values()
/*      */   {
/*  949 */     if (this.values == null) {
/*  950 */       this.values = new Values(this);
/*      */     }
/*  952 */     return this.values;
/*      */   }
/*      */ 
/*      */   protected Iterator createValuesIterator()
/*      */   {
/*  962 */     if (size() == 0) {
/*  963 */       return EmptyIterator.INSTANCE;
/*      */     }
/*  965 */     return new ValuesIterator(this);
/*      */   }
/*      */ 
/*      */   protected void doWriteObject(ObjectOutputStream out)
/*      */     throws IOException
/*      */   {
/* 1175 */     out.writeFloat(this.loadFactor);
/* 1176 */     out.writeInt(this.data.length);
/* 1177 */     out.writeInt(this.size);
/* 1178 */     for (MapIterator it = mapIterator(); it.hasNext(); ) {
/* 1179 */       out.writeObject(it.next());
/* 1180 */       out.writeObject(it.getValue());
/*      */     }
/*      */   }
/*      */ 
/*      */   protected void doReadObject(ObjectInputStream in)
/*      */     throws IOException, ClassNotFoundException
/*      */   {
/* 1203 */     this.loadFactor = in.readFloat();
/* 1204 */     int capacity = in.readInt();
/* 1205 */     int size = in.readInt();
/* 1206 */     init();
/* 1207 */     this.data = new HashEntry[capacity];
/* 1208 */     for (int i = 0; i < size; i++) {
/* 1209 */       Object key = in.readObject();
/* 1210 */       Object value = in.readObject();
/* 1211 */       put(key, value);
/*      */     }
/* 1213 */     this.threshold = calculateThreshold(this.data.length, this.loadFactor);
/*      */   }
/*      */ 
/*      */   protected Object clone()
/*      */   {
/*      */     try
/*      */     {
/* 1227 */       AbstractHashedMap cloned = (AbstractHashedMap)super.clone();
/* 1228 */       cloned.data = new HashEntry[this.data.length];
/* 1229 */       cloned.entrySet = null;
/* 1230 */       cloned.keySet = null;
/* 1231 */       cloned.values = null;
/* 1232 */       cloned.modCount = 0;
/* 1233 */       cloned.size = 0;
/* 1234 */       cloned.init();
/* 1235 */       cloned.putAll(this);
/* 1236 */       return cloned;
/*      */     } catch (CloneNotSupportedException ex) {
/*      */     }
/* 1239 */     return null;
/*      */   }
/*      */ 
/*      */   public boolean equals(Object obj)
/*      */   {
/* 1250 */     if (obj == this) {
/* 1251 */       return true;
/*      */     }
/* 1253 */     if (!(obj instanceof Map)) {
/* 1254 */       return false;
/*      */     }
/* 1256 */     Map map = (Map)obj;
/* 1257 */     if (map.size() != size()) {
/* 1258 */       return false;
/*      */     }
/* 1260 */     MapIterator it = mapIterator();
/*      */     try {
/* 1262 */       while (it.hasNext()) {
/* 1263 */         Object key = it.next();
/* 1264 */         Object value = it.getValue();
/* 1265 */         if (value == null) {
/* 1266 */           if ((map.get(key) != null) || (!map.containsKey(key))) {
/* 1267 */             return false;
/*      */           }
/*      */         }
/* 1270 */         else if (!value.equals(map.get(key)))
/* 1271 */           return false;
/*      */       }
/*      */     }
/*      */     catch (ClassCastException ignored)
/*      */     {
/* 1276 */       return false;
/*      */     } catch (NullPointerException ignored) {
/* 1278 */       return false;
/*      */     }
/* 1280 */     return true;
/*      */   }
/*      */ 
/*      */   public int hashCode()
/*      */   {
/* 1289 */     int total = 0;
/* 1290 */     Iterator it = createEntrySetIterator();
/* 1291 */     while (it.hasNext()) {
/* 1292 */       total += it.next().hashCode();
/*      */     }
/* 1294 */     return total;
/*      */   }
/*      */ 
/*      */   public String toString()
/*      */   {
/* 1303 */     if (size() == 0) {
/* 1304 */       return "{}";
/*      */     }
/* 1306 */     StringBuffer buf = new StringBuffer(32 * size());
/* 1307 */     buf.append('{');
/*      */ 
/* 1309 */     MapIterator it = mapIterator();
/* 1310 */     boolean hasNext = it.hasNext();
/* 1311 */     while (hasNext) {
/* 1312 */       Object key = it.next();
/* 1313 */       Object value = it.getValue();
/* 1314 */       buf.append(key == this ? "(this Map)" : key).append('=').append(value == this ? "(this Map)" : value);
/*      */ 
/* 1318 */       hasNext = it.hasNext();
/* 1319 */       if (hasNext) {
/* 1320 */         buf.append(',').append(' ');
/*      */       }
/*      */     }
/*      */ 
/* 1324 */     buf.append('}');
/* 1325 */     return buf.toString();
/*      */   }
/*      */ 
/*      */   protected static abstract class HashIterator
/*      */     implements Iterator
/*      */   {
/*      */     protected final AbstractHashedMap parent;
/*      */     protected int hashIndex;
/*      */     protected AbstractHashedMap.HashEntry last;
/*      */     protected AbstractHashedMap.HashEntry next;
/*      */     protected int expectedModCount;
/*      */ 
/*      */     protected HashIterator(AbstractHashedMap parent)
/*      */     {
/* 1093 */       this.parent = parent;
/* 1094 */       AbstractHashedMap.HashEntry[] data = parent.data;
/* 1095 */       int i = data.length;
/* 1096 */       AbstractHashedMap.HashEntry next = null;
/* 1097 */       while ((i > 0) && (next == null)) {
/* 1098 */         i--; next = data[i];
/*      */       }
/* 1100 */       this.next = next;
/* 1101 */       this.hashIndex = i;
/* 1102 */       this.expectedModCount = parent.modCount;
/*      */     }
/*      */ 
/*      */     public boolean hasNext() {
/* 1106 */       return this.next != null;
/*      */     }
/*      */ 
/*      */     protected AbstractHashedMap.HashEntry nextEntry() {
/* 1110 */       if (this.parent.modCount != this.expectedModCount) {
/* 1111 */         throw new ConcurrentModificationException();
/*      */       }
/* 1113 */       AbstractHashedMap.HashEntry newCurrent = this.next;
/* 1114 */       if (newCurrent == null) {
/* 1115 */         throw new NoSuchElementException("No next() entry in the iteration");
/*      */       }
/* 1117 */       AbstractHashedMap.HashEntry[] data = this.parent.data;
/* 1118 */       int i = this.hashIndex;
/* 1119 */       AbstractHashedMap.HashEntry n = newCurrent.next;
/* 1120 */       while ((n == null) && (i > 0)) {
/* 1121 */         i--; n = data[i];
/*      */       }
/* 1123 */       this.next = n;
/* 1124 */       this.hashIndex = i;
/* 1125 */       this.last = newCurrent;
/* 1126 */       return newCurrent;
/*      */     }
/*      */ 
/*      */     protected AbstractHashedMap.HashEntry currentEntry() {
/* 1130 */       return this.last;
/*      */     }
/*      */ 
/*      */     public void remove() {
/* 1134 */       if (this.last == null) {
/* 1135 */         throw new IllegalStateException("remove() can only be called once after next()");
/*      */       }
/* 1137 */       if (this.parent.modCount != this.expectedModCount) {
/* 1138 */         throw new ConcurrentModificationException();
/*      */       }
/* 1140 */       this.parent.remove(this.last.getKey());
/* 1141 */       this.last = null;
/* 1142 */       this.expectedModCount = this.parent.modCount;
/*      */     }
/*      */ 
/*      */     public String toString() {
/* 1146 */       if (this.last != null) {
/* 1147 */         return "Iterator[" + this.last.getKey() + "=" + this.last.getValue() + "]";
/*      */       }
/* 1149 */       return "Iterator[]";
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static class HashEntry
/*      */     implements Map.Entry, KeyValue
/*      */   {
/*      */     protected HashEntry next;
/*      */     protected int hashCode;
/*      */     protected Object key;
/*      */     protected Object value;
/*      */ 
/*      */     protected HashEntry(HashEntry next, int hashCode, Object key, Object value)
/*      */     {
/* 1032 */       this.next = next;
/* 1033 */       this.hashCode = hashCode;
/* 1034 */       this.key = key;
/* 1035 */       this.value = value;
/*      */     }
/*      */ 
/*      */     public Object getKey() {
/* 1039 */       return this.key == AbstractHashedMap.NULL ? null : this.key;
/*      */     }
/*      */ 
/*      */     public Object getValue() {
/* 1043 */       return this.value;
/*      */     }
/*      */ 
/*      */     public Object setValue(Object value) {
/* 1047 */       Object old = this.value;
/* 1048 */       this.value = value;
/* 1049 */       return old;
/*      */     }
/*      */ 
/*      */     public boolean equals(Object obj) {
/* 1053 */       if (obj == this) {
/* 1054 */         return true;
/*      */       }
/* 1056 */       if (!(obj instanceof KeyValue)) {
/* 1057 */         return false;
/*      */       }
/* 1059 */       Map.Entry other = (KeyValue)obj;
/* 1060 */       return (getKey() == null ? other.getKey() == null : getKey().equals(other.getKey())) && (getValue() == null ? other.getValue() == null : getValue().equals(other.getValue()));
/*      */     }
/*      */ 
/*      */     public int hashCode()
/*      */     {
/* 1066 */       return (getKey() == null ? 0 : getKey().hashCode()) ^ (getValue() == null ? 0 : getValue().hashCode());
/*      */     }
/*      */ 
/*      */     public String toString()
/*      */     {
/* 1071 */       return getKey() + '=' + getValue();
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static class ValuesIterator extends AbstractHashedMap.HashIterator
/*      */   {
/*      */     protected ValuesIterator(AbstractHashedMap parent)
/*      */     {
/* 1003 */       super();
/*      */     }
/*      */ 
/*      */     public Object next() {
/* 1007 */       return super.nextEntry().getValue();
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static class Values extends AbstractCollection
/*      */   {
/*      */     protected final AbstractHashedMap parent;
/*      */ 
/*      */     protected Values(AbstractHashedMap parent)
/*      */     {
/*  977 */       this.parent = parent;
/*      */     }
/*      */ 
/*      */     public int size() {
/*  981 */       return this.parent.size();
/*      */     }
/*      */ 
/*      */     public void clear() {
/*  985 */       this.parent.clear();
/*      */     }
/*      */ 
/*      */     public boolean contains(Object value) {
/*  989 */       return this.parent.containsValue(value);
/*      */     }
/*      */ 
/*      */     public Iterator iterator() {
/*  993 */       return this.parent.createValuesIterator();
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static class KeySetIterator extends AbstractHashedMap.EntrySetIterator
/*      */   {
/*      */     protected KeySetIterator(AbstractHashedMap parent)
/*      */     {
/*  932 */       super();
/*      */     }
/*      */ 
/*      */     public Object next() {
/*  936 */       return super.nextEntry().getKey();
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static class KeySet extends AbstractSet
/*      */   {
/*      */     protected final AbstractHashedMap parent;
/*      */ 
/*      */     protected KeySet(AbstractHashedMap parent)
/*      */     {
/*  900 */       this.parent = parent;
/*      */     }
/*      */ 
/*      */     public int size() {
/*  904 */       return this.parent.size();
/*      */     }
/*      */ 
/*      */     public void clear() {
/*  908 */       this.parent.clear();
/*      */     }
/*      */ 
/*      */     public boolean contains(Object key) {
/*  912 */       return this.parent.containsKey(key);
/*      */     }
/*      */ 
/*      */     public boolean remove(Object key) {
/*  916 */       boolean result = this.parent.containsKey(key);
/*  917 */       this.parent.remove(key);
/*  918 */       return result;
/*      */     }
/*      */ 
/*      */     public Iterator iterator() {
/*  922 */       return this.parent.createKeySetIterator();
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static class EntrySetIterator extends AbstractHashedMap.HashIterator
/*      */   {
/*      */     protected EntrySetIterator(AbstractHashedMap parent)
/*      */     {
/*  855 */       super();
/*      */     }
/*      */ 
/*      */     public Object next() {
/*  859 */       return super.nextEntry();
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static class EntrySet extends AbstractSet
/*      */   {
/*      */     protected final AbstractHashedMap parent;
/*      */ 
/*      */     protected EntrySet(AbstractHashedMap parent)
/*      */     {
/*  811 */       this.parent = parent;
/*      */     }
/*      */ 
/*      */     public int size() {
/*  815 */       return this.parent.size();
/*      */     }
/*      */ 
/*      */     public void clear() {
/*  819 */       this.parent.clear();
/*      */     }
/*      */ 
/*      */     public boolean contains(Object entry) {
/*  823 */       if ((entry instanceof Map.Entry)) {
/*  824 */         Map.Entry e = (Map.Entry)entry;
/*  825 */         Map.Entry match = this.parent.getEntry(e.getKey());
/*  826 */         return (match != null) && (match.equals(e));
/*      */       }
/*  828 */       return false;
/*      */     }
/*      */ 
/*      */     public boolean remove(Object obj) {
/*  832 */       if (!(obj instanceof Map.Entry)) {
/*  833 */         return false;
/*      */       }
/*  835 */       if (!contains(obj)) {
/*  836 */         return false;
/*      */       }
/*  838 */       Map.Entry entry = (Map.Entry)obj;
/*  839 */       Object key = entry.getKey();
/*  840 */       this.parent.remove(key);
/*  841 */       return true;
/*      */     }
/*      */ 
/*      */     public Iterator iterator() {
/*  845 */       return this.parent.createEntrySetIterator();
/*      */     }
/*      */   }
/*      */ 
/*      */   protected static class HashMapIterator extends AbstractHashedMap.HashIterator
/*      */     implements MapIterator
/*      */   {
/*      */     protected HashMapIterator(AbstractHashedMap parent)
/*      */     {
/*  742 */       super();
/*      */     }
/*      */ 
/*      */     public Object next() {
/*  746 */       return super.nextEntry().getKey();
/*      */     }
/*      */ 
/*      */     public Object getKey() {
/*  750 */       AbstractHashedMap.HashEntry current = currentEntry();
/*  751 */       if (current == null) {
/*  752 */         throw new IllegalStateException("getKey() can only be called after next() and before remove()");
/*      */       }
/*  754 */       return current.getKey();
/*      */     }
/*      */ 
/*      */     public Object getValue() {
/*  758 */       AbstractHashedMap.HashEntry current = currentEntry();
/*  759 */       if (current == null) {
/*  760 */         throw new IllegalStateException("getValue() can only be called after next() and before remove()");
/*      */       }
/*  762 */       return current.getValue();
/*      */     }
/*      */ 
/*      */     public Object setValue(Object value) {
/*  766 */       AbstractHashedMap.HashEntry current = currentEntry();
/*  767 */       if (current == null) {
/*  768 */         throw new IllegalStateException("setValue() can only be called after next() and before remove()");
/*      */       }
/*  770 */       return current.setValue(value);
/*      */     }
/*      */   }
/*      */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     org.apache.commons.collections.map.AbstractHashedMap
 * JD-Core Version:    0.6.0
 */