/*     */ package javassist.bytecode;
/*     */ 
/*     */ import java.io.DataInputStream;
/*     */ import java.io.DataOutputStream;
/*     */ import java.io.IOException;
/*     */ import java.util.ArrayList;
/*     */ import java.util.LinkedList;
/*     */ import java.util.List;
/*     */ import java.util.ListIterator;
/*     */ import java.util.Map;
/*     */ import javassist.CannotCompileException;
/*     */ 
/*     */ public final class ClassFile
/*     */ {
/*     */   int major;
/*     */   int minor;
/*     */   ConstPool constPool;
/*     */   int thisClass;
/*     */   int accessFlags;
/*     */   int superClass;
/*     */   int[] interfaces;
/*     */   ArrayList fields;
/*     */   ArrayList methods;
/*     */   LinkedList attributes;
/*     */   String thisclassname;
/*     */   String[] cachedInterfaces;
/*     */   String cachedSuperclass;
/*     */   public static final int JAVA_1 = 45;
/*     */   public static final int JAVA_2 = 46;
/*     */   public static final int JAVA_3 = 47;
/*     */   public static final int JAVA_4 = 48;
/*     */   public static final int JAVA_5 = 49;
/*     */   public static final int JAVA_6 = 50;
/*     */   public static final int JAVA_7 = 51;
/*  95 */   public static int MAJOR_VERSION = 47;
/*     */ 
/*     */   public ClassFile(DataInputStream in)
/*     */     throws IOException
/*     */   {
/* 109 */     read(in);
/*     */   }
/*     */ 
/*     */   public ClassFile(boolean isInterface, String classname, String superclass)
/*     */   {
/* 123 */     this.major = MAJOR_VERSION;
/* 124 */     this.minor = 0;
/* 125 */     this.constPool = new ConstPool(classname);
/* 126 */     this.thisClass = this.constPool.getThisClassInfo();
/* 127 */     if (isInterface)
/* 128 */       this.accessFlags = 1536;
/*     */     else {
/* 130 */       this.accessFlags = 32;
/*     */     }
/* 132 */     initSuperclass(superclass);
/* 133 */     this.interfaces = null;
/* 134 */     this.fields = new ArrayList();
/* 135 */     this.methods = new ArrayList();
/* 136 */     this.thisclassname = classname;
/*     */ 
/* 138 */     this.attributes = new LinkedList();
/* 139 */     this.attributes.add(new SourceFileAttribute(this.constPool, getSourcefileName(this.thisclassname)));
/*     */   }
/*     */ 
/*     */   private void initSuperclass(String superclass)
/*     */   {
/* 144 */     if (superclass != null) {
/* 145 */       this.superClass = this.constPool.addClassInfo(superclass);
/* 146 */       this.cachedSuperclass = superclass;
/*     */     }
/*     */     else {
/* 149 */       this.superClass = this.constPool.addClassInfo("java.lang.Object");
/* 150 */       this.cachedSuperclass = "java.lang.Object";
/*     */     }
/*     */   }
/*     */ 
/*     */   private static String getSourcefileName(String qname) {
/* 155 */     int index = qname.lastIndexOf('.');
/* 156 */     if (index >= 0) {
/* 157 */       qname = qname.substring(index + 1);
/*     */     }
/* 159 */     return qname + ".java";
/*     */   }
/*     */ 
/*     */   public void compact()
/*     */   {
/* 168 */     ConstPool cp = compact0();
/* 169 */     ArrayList list = this.methods;
/* 170 */     int n = list.size();
/* 171 */     for (int i = 0; i < n; i++) {
/* 172 */       MethodInfo minfo = (MethodInfo)list.get(i);
/* 173 */       minfo.compact(cp);
/*     */     }
/*     */ 
/* 176 */     list = this.fields;
/* 177 */     n = list.size();
/* 178 */     for (int i = 0; i < n; i++) {
/* 179 */       FieldInfo finfo = (FieldInfo)list.get(i);
/* 180 */       finfo.compact(cp);
/*     */     }
/*     */ 
/* 183 */     this.attributes = AttributeInfo.copyAll(this.attributes, cp);
/* 184 */     this.constPool = cp;
/*     */   }
/*     */ 
/*     */   private ConstPool compact0() {
/* 188 */     ConstPool cp = new ConstPool(this.thisclassname);
/* 189 */     this.thisClass = cp.getThisClassInfo();
/* 190 */     String sc = getSuperclass();
/* 191 */     if (sc != null) {
/* 192 */       this.superClass = cp.addClassInfo(getSuperclass());
/*     */     }
/* 194 */     if (this.interfaces != null) {
/* 195 */       int n = this.interfaces.length;
/* 196 */       for (int i = 0; i < n; i++) {
/* 197 */         this.interfaces[i] = cp.addClassInfo(this.constPool.getClassInfo(this.interfaces[i]));
/*     */       }
/*     */     }
/*     */ 
/* 201 */     return cp;
/*     */   }
/*     */ 
/*     */   public void prune()
/*     */   {
/* 211 */     ConstPool cp = compact0();
/* 212 */     LinkedList newAttributes = new LinkedList();
/* 213 */     AttributeInfo invisibleAnnotations = getAttribute("RuntimeInvisibleAnnotations");
/*     */ 
/* 215 */     if (invisibleAnnotations != null) {
/* 216 */       invisibleAnnotations = invisibleAnnotations.copy(cp, null);
/* 217 */       newAttributes.add(invisibleAnnotations);
/*     */     }
/*     */ 
/* 220 */     AttributeInfo visibleAnnotations = getAttribute("RuntimeVisibleAnnotations");
/*     */ 
/* 222 */     if (visibleAnnotations != null) {
/* 223 */       visibleAnnotations = visibleAnnotations.copy(cp, null);
/* 224 */       newAttributes.add(visibleAnnotations);
/*     */     }
/*     */ 
/* 227 */     AttributeInfo signature = getAttribute("Signature");
/*     */ 
/* 229 */     if (signature != null) {
/* 230 */       signature = signature.copy(cp, null);
/* 231 */       newAttributes.add(signature);
/*     */     }
/*     */ 
/* 234 */     ArrayList list = this.methods;
/* 235 */     int n = list.size();
/* 236 */     for (int i = 0; i < n; i++) {
/* 237 */       MethodInfo minfo = (MethodInfo)list.get(i);
/* 238 */       minfo.prune(cp);
/*     */     }
/*     */ 
/* 241 */     list = this.fields;
/* 242 */     n = list.size();
/* 243 */     for (int i = 0; i < n; i++) {
/* 244 */       FieldInfo finfo = (FieldInfo)list.get(i);
/* 245 */       finfo.prune(cp);
/*     */     }
/*     */ 
/* 248 */     this.attributes = newAttributes;
/* 249 */     this.constPool = cp;
/*     */   }
/*     */ 
/*     */   public ConstPool getConstPool()
/*     */   {
/* 256 */     return this.constPool;
/*     */   }
/*     */ 
/*     */   public boolean isInterface()
/*     */   {
/* 263 */     return (this.accessFlags & 0x200) != 0;
/*     */   }
/*     */ 
/*     */   public boolean isFinal()
/*     */   {
/* 270 */     return (this.accessFlags & 0x10) != 0;
/*     */   }
/*     */ 
/*     */   public boolean isAbstract()
/*     */   {
/* 277 */     return (this.accessFlags & 0x400) != 0;
/*     */   }
/*     */ 
/*     */   public int getAccessFlags()
/*     */   {
/* 286 */     return this.accessFlags;
/*     */   }
/*     */ 
/*     */   public void setAccessFlags(int acc)
/*     */   {
/* 295 */     if ((acc & 0x200) == 0) {
/* 296 */       acc |= 32;
/*     */     }
/* 298 */     this.accessFlags = acc;
/*     */   }
/*     */ 
/*     */   public int getInnerAccessFlags()
/*     */   {
/* 310 */     InnerClassesAttribute ica = (InnerClassesAttribute)getAttribute("InnerClasses");
/*     */ 
/* 312 */     if (ica == null) {
/* 313 */       return -1;
/*     */     }
/* 315 */     String name = getName();
/* 316 */     int n = ica.tableLength();
/* 317 */     for (int i = 0; i < n; i++) {
/* 318 */       if (name.equals(ica.innerClass(i)))
/* 319 */         return ica.accessFlags(i);
/*     */     }
/* 321 */     return -1;
/*     */   }
/*     */ 
/*     */   public String getName()
/*     */   {
/* 328 */     return this.thisclassname;
/*     */   }
/*     */ 
/*     */   public void setName(String name)
/*     */   {
/* 336 */     renameClass(this.thisclassname, name);
/*     */   }
/*     */ 
/*     */   public String getSuperclass()
/*     */   {
/* 343 */     if (this.cachedSuperclass == null) {
/* 344 */       this.cachedSuperclass = this.constPool.getClassInfo(this.superClass);
/*     */     }
/* 346 */     return this.cachedSuperclass;
/*     */   }
/*     */ 
/*     */   public int getSuperclassId()
/*     */   {
/* 354 */     return this.superClass;
/*     */   }
/*     */ 
/*     */   public void setSuperclass(String superclass)
/*     */     throws CannotCompileException
/*     */   {
/* 366 */     if (superclass == null)
/* 367 */       superclass = "java.lang.Object";
/*     */     try
/*     */     {
/* 370 */       this.superClass = this.constPool.addClassInfo(superclass);
/* 371 */       ArrayList list = this.methods;
/* 372 */       int n = list.size();
/* 373 */       for (int i = 0; i < n; i++) {
/* 374 */         MethodInfo minfo = (MethodInfo)list.get(i);
/* 375 */         minfo.setSuperclass(superclass);
/*     */       }
/*     */     }
/*     */     catch (BadBytecode e) {
/* 379 */       throw new CannotCompileException(e);
/*     */     }
/* 381 */     this.cachedSuperclass = superclass;
/*     */   }
/*     */ 
/*     */   public final void renameClass(String oldname, String newname)
/*     */   {
/* 402 */     if (oldname.equals(newname)) {
/* 403 */       return;
/*     */     }
/* 405 */     if (oldname.equals(this.thisclassname)) {
/* 406 */       this.thisclassname = newname;
/*     */     }
/* 408 */     oldname = Descriptor.toJvmName(oldname);
/* 409 */     newname = Descriptor.toJvmName(newname);
/* 410 */     this.constPool.renameClass(oldname, newname);
/*     */ 
/* 412 */     AttributeInfo.renameClass(this.attributes, oldname, newname);
/* 413 */     ArrayList list = this.methods;
/* 414 */     int n = list.size();
/* 415 */     for (int i = 0; i < n; i++) {
/* 416 */       MethodInfo minfo = (MethodInfo)list.get(i);
/* 417 */       String desc = minfo.getDescriptor();
/* 418 */       minfo.setDescriptor(Descriptor.rename(desc, oldname, newname));
/* 419 */       AttributeInfo.renameClass(minfo.getAttributes(), oldname, newname);
/*     */     }
/*     */ 
/* 422 */     list = this.fields;
/* 423 */     n = list.size();
/* 424 */     for (int i = 0; i < n; i++) {
/* 425 */       FieldInfo finfo = (FieldInfo)list.get(i);
/* 426 */       String desc = finfo.getDescriptor();
/* 427 */       finfo.setDescriptor(Descriptor.rename(desc, oldname, newname));
/* 428 */       AttributeInfo.renameClass(finfo.getAttributes(), oldname, newname);
/*     */     }
/*     */   }
/*     */ 
/*     */   public final void renameClass(Map classnames)
/*     */   {
/* 442 */     String jvmNewThisName = (String)classnames.get(Descriptor.toJvmName(this.thisclassname));
/*     */ 
/* 444 */     if (jvmNewThisName != null) {
/* 445 */       this.thisclassname = Descriptor.toJavaName(jvmNewThisName);
/*     */     }
/* 447 */     this.constPool.renameClass(classnames);
/*     */ 
/* 449 */     AttributeInfo.renameClass(this.attributes, classnames);
/* 450 */     ArrayList list = this.methods;
/* 451 */     int n = list.size();
/* 452 */     for (int i = 0; i < n; i++) {
/* 453 */       MethodInfo minfo = (MethodInfo)list.get(i);
/* 454 */       String desc = minfo.getDescriptor();
/* 455 */       minfo.setDescriptor(Descriptor.rename(desc, classnames));
/* 456 */       AttributeInfo.renameClass(minfo.getAttributes(), classnames);
/*     */     }
/*     */ 
/* 459 */     list = this.fields;
/* 460 */     n = list.size();
/* 461 */     for (int i = 0; i < n; i++) {
/* 462 */       FieldInfo finfo = (FieldInfo)list.get(i);
/* 463 */       String desc = finfo.getDescriptor();
/* 464 */       finfo.setDescriptor(Descriptor.rename(desc, classnames));
/* 465 */       AttributeInfo.renameClass(finfo.getAttributes(), classnames);
/*     */     }
/*     */   }
/*     */ 
/*     */   public String[] getInterfaces()
/*     */   {
/* 474 */     if (this.cachedInterfaces != null) {
/* 475 */       return this.cachedInterfaces;
/*     */     }
/* 477 */     String[] rtn = null;
/* 478 */     if (this.interfaces == null) {
/* 479 */       rtn = new String[0];
/*     */     } else {
/* 481 */       int n = this.interfaces.length;
/* 482 */       String[] list = new String[n];
/* 483 */       for (int i = 0; i < n; i++) {
/* 484 */         list[i] = this.constPool.getClassInfo(this.interfaces[i]);
/*     */       }
/* 486 */       rtn = list;
/*     */     }
/*     */ 
/* 489 */     this.cachedInterfaces = rtn;
/* 490 */     return rtn;
/*     */   }
/*     */ 
/*     */   public void setInterfaces(String[] nameList)
/*     */   {
/* 500 */     this.cachedInterfaces = null;
/* 501 */     if (nameList != null) {
/* 502 */       int n = nameList.length;
/* 503 */       this.interfaces = new int[n];
/* 504 */       for (int i = 0; i < n; i++)
/* 505 */         this.interfaces[i] = this.constPool.addClassInfo(nameList[i]);
/*     */     }
/*     */   }
/*     */ 
/*     */   public void addInterface(String name)
/*     */   {
/* 513 */     this.cachedInterfaces = null;
/* 514 */     int info = this.constPool.addClassInfo(name);
/* 515 */     if (this.interfaces == null) {
/* 516 */       this.interfaces = new int[1];
/* 517 */       this.interfaces[0] = info;
/*     */     }
/*     */     else {
/* 520 */       int n = this.interfaces.length;
/* 521 */       int[] newarray = new int[n + 1];
/* 522 */       System.arraycopy(this.interfaces, 0, newarray, 0, n);
/* 523 */       newarray[n] = info;
/* 524 */       this.interfaces = newarray;
/*     */     }
/*     */   }
/*     */ 
/*     */   public List getFields()
/*     */   {
/* 535 */     return this.fields;
/*     */   }
/*     */ 
/*     */   public void addField(FieldInfo finfo)
/*     */     throws DuplicateMemberException
/*     */   {
/* 544 */     testExistingField(finfo.getName(), finfo.getDescriptor());
/* 545 */     this.fields.add(finfo);
/*     */   }
/*     */ 
/*     */   private void addField0(FieldInfo finfo) {
/* 549 */     this.fields.add(finfo);
/*     */   }
/*     */ 
/*     */   private void testExistingField(String name, String descriptor) throws DuplicateMemberException
/*     */   {
/* 554 */     ListIterator it = this.fields.listIterator(0);
/* 555 */     while (it.hasNext()) {
/* 556 */       FieldInfo minfo = (FieldInfo)it.next();
/* 557 */       if (minfo.getName().equals(name))
/* 558 */         throw new DuplicateMemberException("duplicate field: " + name);
/*     */     }
/*     */   }
/*     */ 
/*     */   public List getMethods()
/*     */   {
/* 569 */     return this.methods;
/*     */   }
/*     */ 
/*     */   public MethodInfo getMethod(String name)
/*     */   {
/* 579 */     ArrayList list = this.methods;
/* 580 */     int n = list.size();
/* 581 */     for (int i = 0; i < n; i++) {
/* 582 */       MethodInfo minfo = (MethodInfo)list.get(i);
/* 583 */       if (minfo.getName().equals(name)) {
/* 584 */         return minfo;
/*     */       }
/*     */     }
/* 587 */     return null;
/*     */   }
/*     */ 
/*     */   public MethodInfo getStaticInitializer()
/*     */   {
/* 595 */     return getMethod("<clinit>");
/*     */   }
/*     */ 
/*     */   public void addMethod(MethodInfo minfo)
/*     */     throws DuplicateMemberException
/*     */   {
/* 606 */     testExistingMethod(minfo);
/* 607 */     this.methods.add(minfo);
/*     */   }
/*     */ 
/*     */   private void addMethod0(MethodInfo minfo) {
/* 611 */     this.methods.add(minfo);
/*     */   }
/*     */ 
/*     */   private void testExistingMethod(MethodInfo newMinfo)
/*     */     throws DuplicateMemberException
/*     */   {
/* 617 */     String name = newMinfo.getName();
/* 618 */     String descriptor = newMinfo.getDescriptor();
/* 619 */     ListIterator it = this.methods.listIterator(0);
/* 620 */     while (it.hasNext())
/* 621 */       if (isDuplicated(newMinfo, name, descriptor, (MethodInfo)it.next(), it))
/* 622 */         throw new DuplicateMemberException("duplicate method: " + name + " in " + getName());
/*     */   }
/*     */ 
/*     */   private static boolean isDuplicated(MethodInfo newMethod, String newName, String newDesc, MethodInfo minfo, ListIterator it)
/*     */   {
/* 630 */     if (!minfo.getName().equals(newName)) {
/* 631 */       return false;
/*     */     }
/* 633 */     String desc = minfo.getDescriptor();
/* 634 */     if (!Descriptor.eqParamTypes(desc, newDesc)) {
/* 635 */       return false;
/*     */     }
/* 637 */     if (desc.equals(newDesc)) {
/* 638 */       if (notBridgeMethod(minfo)) {
/* 639 */         return true;
/*     */       }
/* 641 */       it.remove();
/* 642 */       return false;
/*     */     }
/*     */ 
/* 646 */     return (notBridgeMethod(minfo)) && (notBridgeMethod(newMethod));
/*     */   }
/*     */ 
/*     */   private static boolean notBridgeMethod(MethodInfo minfo)
/*     */   {
/* 652 */     return (minfo.getAccessFlags() & 0x40) == 0;
/*     */   }
/*     */ 
/*     */   public List getAttributes()
/*     */   {
/* 666 */     return this.attributes;
/*     */   }
/*     */ 
/*     */   public AttributeInfo getAttribute(String name)
/*     */   {
/* 678 */     LinkedList list = this.attributes;
/* 679 */     int n = list.size();
/* 680 */     for (int i = 0; i < n; i++) {
/* 681 */       AttributeInfo ai = (AttributeInfo)list.get(i);
/* 682 */       if (ai.getName().equals(name)) {
/* 683 */         return ai;
/*     */       }
/*     */     }
/* 686 */     return null;
/*     */   }
/*     */ 
/*     */   public void addAttribute(AttributeInfo info)
/*     */   {
/* 696 */     AttributeInfo.remove(this.attributes, info.getName());
/* 697 */     this.attributes.add(info);
/*     */   }
/*     */ 
/*     */   public String getSourceFile()
/*     */   {
/* 706 */     SourceFileAttribute sf = (SourceFileAttribute)getAttribute("SourceFile");
/*     */ 
/* 708 */     if (sf == null) {
/* 709 */       return null;
/*     */     }
/* 711 */     return sf.getFileName();
/*     */   }
/*     */ 
/*     */   private void read(DataInputStream in) throws IOException
/*     */   {
/* 716 */     int magic = in.readInt();
/* 717 */     if (magic != -889275714) {
/* 718 */       throw new IOException("bad magic number: " + Integer.toHexString(magic));
/*     */     }
/* 720 */     this.minor = in.readUnsignedShort();
/* 721 */     this.major = in.readUnsignedShort();
/* 722 */     this.constPool = new ConstPool(in);
/* 723 */     this.accessFlags = in.readUnsignedShort();
/* 724 */     this.thisClass = in.readUnsignedShort();
/* 725 */     this.constPool.setThisClassInfo(this.thisClass);
/* 726 */     this.superClass = in.readUnsignedShort();
/* 727 */     int n = in.readUnsignedShort();
/* 728 */     if (n == 0) {
/* 729 */       this.interfaces = null;
/*     */     } else {
/* 731 */       this.interfaces = new int[n];
/* 732 */       for (int i = 0; i < n; i++) {
/* 733 */         this.interfaces[i] = in.readUnsignedShort();
/*     */       }
/*     */     }
/* 736 */     ConstPool cp = this.constPool;
/* 737 */     n = in.readUnsignedShort();
/* 738 */     this.fields = new ArrayList();
/* 739 */     for (int i = 0; i < n; i++) {
/* 740 */       addField0(new FieldInfo(cp, in));
/*     */     }
/* 742 */     n = in.readUnsignedShort();
/* 743 */     this.methods = new ArrayList();
/* 744 */     for (i = 0; i < n; i++) {
/* 745 */       addMethod0(new MethodInfo(cp, in));
/*     */     }
/* 747 */     this.attributes = new LinkedList();
/* 748 */     n = in.readUnsignedShort();
/* 749 */     for (i = 0; i < n; i++) {
/* 750 */       addAttribute(AttributeInfo.read(cp, in));
/*     */     }
/* 752 */     this.thisclassname = this.constPool.getClassInfo(this.thisClass);
/*     */   }
/*     */ 
/*     */   public void write(DataOutputStream out)
/*     */     throws IOException
/*     */   {
/* 761 */     out.writeInt(-889275714);
/* 762 */     out.writeShort(this.minor);
/* 763 */     out.writeShort(this.major);
/* 764 */     this.constPool.write(out);
/* 765 */     out.writeShort(this.accessFlags);
/* 766 */     out.writeShort(this.thisClass);
/* 767 */     out.writeShort(this.superClass);
/*     */     int n;
/* 769 */     if (this.interfaces == null)
/* 770 */       n = 0;
/*     */     else {
/* 772 */       n = this.interfaces.length;
/*     */     }
/* 774 */     out.writeShort(n);
/* 775 */     for (int i = 0; i < n; i++) {
/* 776 */       out.writeShort(this.interfaces[i]);
/*     */     }
/* 778 */     ArrayList list = this.fields;
/* 779 */     int n = list.size();
/* 780 */     out.writeShort(n);
/* 781 */     for (i = 0; i < n; i++) {
/* 782 */       FieldInfo finfo = (FieldInfo)list.get(i);
/* 783 */       finfo.write(out);
/*     */     }
/*     */ 
/* 786 */     list = this.methods;
/* 787 */     n = list.size();
/* 788 */     out.writeShort(n);
/* 789 */     for (i = 0; i < n; i++) {
/* 790 */       MethodInfo minfo = (MethodInfo)list.get(i);
/* 791 */       minfo.write(out);
/*     */     }
/*     */ 
/* 794 */     out.writeShort(this.attributes.size());
/* 795 */     AttributeInfo.writeAll(this.attributes, out);
/*     */   }
/*     */ 
/*     */   public int getMajorVersion()
/*     */   {
/* 804 */     return this.major;
/*     */   }
/*     */ 
/*     */   public void setMajorVersion(int major)
/*     */   {
/* 814 */     this.major = major;
/*     */   }
/*     */ 
/*     */   public int getMinorVersion()
/*     */   {
/* 823 */     return this.minor;
/*     */   }
/*     */ 
/*     */   public void setMinorVersion(int minor)
/*     */   {
/* 833 */     this.minor = minor;
/*     */   }
/*     */ 
/*     */   public void setVersionToJava5()
/*     */   {
/* 844 */     this.major = 49;
/* 845 */     this.minor = 0;
/*     */   }
/*     */ 
/*     */   static
/*     */   {
/*     */     try
/*     */     {
/*  99 */       Class.forName("java.lang.StringBuilder");
/* 100 */       MAJOR_VERSION = 49;
/*     */     }
/*     */     catch (Throwable t)
/*     */     {
/*     */     }
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     javassist.bytecode.ClassFile
 * JD-Core Version:    0.6.0
 */