/*     */ package gnu.trove;
/*     */ 
/*     */ import java.io.IOException;
/*     */ import java.io.ObjectInputStream;
/*     */ import java.io.ObjectOutputStream;
/*     */ import java.io.Serializable;
/*     */ import java.util.Arrays;
/*     */ import java.util.Random;
/*     */ 
/*     */ public class TIntArrayList
/*     */   implements Serializable, Cloneable
/*     */ {
/*     */   protected transient int[] _data;
/*     */   protected transient int _pos;
/*     */   protected static final int DEFAULT_CAPACITY = 10;
/*     */ 
/*     */   public TIntArrayList()
/*     */   {
/*  53 */     this(10);
/*     */   }
/*     */ 
/*     */   public TIntArrayList(int capacity)
/*     */   {
/*  63 */     this._data = new int[capacity];
/*  64 */     this._pos = 0;
/*     */   }
/*     */ 
/*     */   public TIntArrayList(int[] values)
/*     */   {
/*  76 */     this(Math.max(values.length, 10));
/*  77 */     add(values);
/*     */   }
/*     */ 
/*     */   public void ensureCapacity(int capacity)
/*     */   {
/*  91 */     if (capacity > this._data.length) {
/*  92 */       int newCap = Math.max(this._data.length << 1, capacity);
/*  93 */       int[] tmp = new int[newCap];
/*  94 */       System.arraycopy(this._data, 0, tmp, 0, this._data.length);
/*  95 */       this._data = tmp;
/*     */     }
/*     */   }
/*     */ 
/*     */   public int size()
/*     */   {
/* 105 */     return this._pos;
/*     */   }
/*     */ 
/*     */   public boolean isEmpty()
/*     */   {
/* 114 */     return this._pos == 0;
/*     */   }
/*     */ 
/*     */   public void trimToSize()
/*     */   {
/* 122 */     if (this._data.length > size()) {
/* 123 */       int[] tmp = new int[size()];
/* 124 */       toNativeArray(tmp, 0, tmp.length);
/* 125 */       this._data = tmp;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void add(int val)
/*     */   {
/* 137 */     ensureCapacity(this._pos + 1);
/* 138 */     this._data[(this._pos++)] = val;
/*     */   }
/*     */ 
/*     */   public void add(int[] vals)
/*     */   {
/* 148 */     add(vals, 0, vals.length);
/*     */   }
/*     */ 
/*     */   public void add(int[] vals, int offset, int length)
/*     */   {
/* 160 */     ensureCapacity(this._pos + length);
/* 161 */     System.arraycopy(vals, offset, this._data, this._pos, length);
/* 162 */     this._pos += length;
/*     */   }
/*     */ 
/*     */   public void insert(int offset, int value)
/*     */   {
/* 174 */     if (offset == this._pos) {
/* 175 */       add(value);
/* 176 */       return;
/*     */     }
/* 178 */     ensureCapacity(this._pos + 1);
/*     */ 
/* 180 */     System.arraycopy(this._data, offset, this._data, offset + 1, this._pos - offset);
/*     */ 
/* 182 */     this._data[offset] = value;
/* 183 */     this._pos += 1;
/*     */   }
/*     */ 
/*     */   public void insert(int offset, int[] values)
/*     */   {
/* 195 */     insert(offset, values, 0, values.length);
/*     */   }
/*     */ 
/*     */   public void insert(int offset, int[] values, int valOffset, int len)
/*     */   {
/* 210 */     if (offset == this._pos) {
/* 211 */       add(values, valOffset, len);
/* 212 */       return;
/*     */     }
/*     */ 
/* 215 */     ensureCapacity(this._pos + len);
/*     */ 
/* 217 */     System.arraycopy(this._data, offset, this._data, offset + len, this._pos - offset);
/*     */ 
/* 219 */     System.arraycopy(values, valOffset, this._data, offset, len);
/* 220 */     this._pos += len;
/*     */   }
/*     */ 
/*     */   public int get(int offset)
/*     */   {
/* 230 */     if (offset >= this._pos) {
/* 231 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 233 */     return this._data[offset];
/*     */   }
/*     */ 
/*     */   public int getQuick(int offset)
/*     */   {
/* 244 */     return this._data[offset];
/*     */   }
/*     */ 
/*     */   public void set(int offset, int val)
/*     */   {
/* 254 */     if (offset >= this._pos) {
/* 255 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 257 */     this._data[offset] = val;
/*     */   }
/*     */ 
/*     */   public int getSet(int offset, int val)
/*     */   {
/* 269 */     if (offset >= this._pos) {
/* 270 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 272 */     int old = this._data[offset];
/* 273 */     this._data[offset] = val;
/* 274 */     return old;
/*     */   }
/*     */ 
/*     */   public void set(int offset, int[] values)
/*     */   {
/* 285 */     set(offset, values, 0, values.length);
/*     */   }
/*     */ 
/*     */   public void set(int offset, int[] values, int valOffset, int length)
/*     */   {
/* 299 */     if ((offset < 0) || (offset + length >= this._pos)) {
/* 300 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 302 */     System.arraycopy(this._data, offset, values, valOffset, length);
/*     */   }
/*     */ 
/*     */   public void setQuick(int offset, int val)
/*     */   {
/* 313 */     this._data[offset] = val;
/*     */   }
/*     */ 
/*     */   public void clear()
/*     */   {
/* 321 */     clear(10);
/*     */   }
/*     */ 
/*     */   public void clear(int capacity)
/*     */   {
/* 331 */     this._data = new int[capacity];
/* 332 */     this._pos = 0;
/*     */   }
/*     */ 
/*     */   public void reset()
/*     */   {
/* 344 */     this._pos = 0;
/* 345 */     fill(0);
/*     */   }
/*     */ 
/*     */   public void resetQuick()
/*     */   {
/* 363 */     this._pos = 0;
/*     */   }
/*     */ 
/*     */   public int remove(int offset)
/*     */   {
/* 373 */     int old = get(offset);
/* 374 */     remove(offset, 1);
/* 375 */     return old;
/*     */   }
/*     */ 
/*     */   public void remove(int offset, int length)
/*     */   {
/* 386 */     if ((offset < 0) || (offset >= this._pos)) {
/* 387 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/*     */ 
/* 390 */     if (offset == 0)
/*     */     {
/* 392 */       System.arraycopy(this._data, length, this._data, 0, this._pos - length);
/* 393 */     } else if (this._pos - length != offset)
/*     */     {
/* 398 */       System.arraycopy(this._data, offset + length, this._data, offset, this._pos - (offset + length));
/*     */     }
/*     */ 
/* 401 */     this._pos -= length;
/*     */   }
/*     */ 
/*     */   public void transformValues(TIntFunction function)
/*     */   {
/* 413 */     for (int i = this._pos; i-- > 0; )
/* 414 */       this._data[i] = function.execute(this._data[i]);
/*     */   }
/*     */ 
/*     */   public void reverse()
/*     */   {
/* 422 */     reverse(0, this._pos);
/*     */   }
/*     */ 
/*     */   public void reverse(int from, int to)
/*     */   {
/* 432 */     if (from == to) {
/* 433 */       return;
/*     */     }
/* 435 */     if (from > to) {
/* 436 */       throw new IllegalArgumentException("from cannot be greater than to");
/*     */     }
/* 438 */     int i = from; for (int j = to - 1; i < j; j--) {
/* 439 */       swap(i, j);
/*     */ 
/* 438 */       i++;
/*     */     }
/*     */   }
/*     */ 
/*     */   public void shuffle(Random rand)
/*     */   {
/* 450 */     for (int i = this._pos; i-- > 1; )
/* 451 */       swap(i, rand.nextInt(i));
/*     */   }
/*     */ 
/*     */   private final void swap(int i, int j)
/*     */   {
/* 462 */     int tmp = this._data[i];
/* 463 */     this._data[i] = this._data[j];
/* 464 */     this._data[j] = tmp;
/*     */   }
/*     */ 
/*     */   public Object clone()
/*     */   {
/* 476 */     TIntArrayList clone = null;
/*     */     try {
/* 478 */       clone = (TIntArrayList)super.clone();
/* 479 */       clone._data = ((int[])this._data.clone());
/*     */     }
/*     */     catch (CloneNotSupportedException e) {
/*     */     }
/* 483 */     return clone;
/*     */   }
/*     */ 
/*     */   public int[] toNativeArray()
/*     */   {
/* 492 */     return toNativeArray(0, this._pos);
/*     */   }
/*     */ 
/*     */   public int[] toNativeArray(int offset, int len)
/*     */   {
/* 503 */     int[] rv = new int[len];
/* 504 */     toNativeArray(rv, offset, len);
/* 505 */     return rv;
/*     */   }
/*     */ 
/*     */   public void toNativeArray(int[] dest, int offset, int len)
/*     */   {
/* 516 */     if (len == 0) {
/* 517 */       return;
/*     */     }
/* 519 */     if ((offset < 0) || (offset >= this._pos)) {
/* 520 */       throw new ArrayIndexOutOfBoundsException(offset);
/*     */     }
/* 522 */     System.arraycopy(this._data, offset, dest, 0, len);
/*     */   }
/*     */ 
/*     */   public boolean equals(Object other)
/*     */   {
/* 535 */     if (other == this)
/* 536 */       return true;
/* 537 */     if ((other instanceof TIntArrayList)) {
/* 538 */       TIntArrayList that = (TIntArrayList)other;
/* 539 */       if (that.size() != size()) {
/* 540 */         return false;
/*     */       }
/* 542 */       for (int i = this._pos; i-- > 0; ) {
/* 543 */         if (this._data[i] != that._data[i]) {
/* 544 */           return false;
/*     */         }
/*     */       }
/* 547 */       return true;
/*     */     }
/*     */ 
/* 550 */     return false;
/*     */   }
/*     */ 
/*     */   public int hashCode()
/*     */   {
/* 555 */     int h = 0;
/* 556 */     for (int i = this._pos; i-- > 0; ) {
/* 557 */       h += this._data[i];
/*     */     }
/* 559 */     return h;
/*     */   }
/*     */ 
/*     */   public boolean forEach(TIntProcedure procedure)
/*     */   {
/* 572 */     for (int i = 0; i < this._pos; i++) {
/* 573 */       if (!procedure.execute(this._data[i])) {
/* 574 */         return false;
/*     */       }
/*     */     }
/* 577 */     return true;
/*     */   }
/*     */ 
/*     */   public boolean forEachDescending(TIntProcedure procedure)
/*     */   {
/* 588 */     for (int i = this._pos; i-- > 0; ) {
/* 589 */       if (!procedure.execute(this._data[i])) {
/* 590 */         return false;
/*     */       }
/*     */     }
/* 593 */     return true;
/*     */   }
/*     */ 
/*     */   public void sort()
/*     */   {
/* 605 */     Arrays.sort(this._data, 0, this._pos);
/*     */   }
/*     */ 
/*     */   public void sort(int fromIndex, int toIndex)
/*     */   {
/* 617 */     Arrays.sort(this._data, fromIndex, toIndex);
/*     */   }
/*     */ 
/*     */   public void fill(int val)
/*     */   {
/* 628 */     Arrays.fill(this._data, 0, this._pos, val);
/*     */   }
/*     */ 
/*     */   public void fill(int fromIndex, int toIndex, int val)
/*     */   {
/* 639 */     if (toIndex > this._pos) {
/* 640 */       ensureCapacity(toIndex);
/* 641 */       this._pos = toIndex;
/*     */     }
/* 643 */     Arrays.fill(this._data, fromIndex, toIndex, val);
/*     */   }
/*     */ 
/*     */   public int binarySearch(int value)
/*     */   {
/* 658 */     return binarySearch(value, 0, this._pos);
/*     */   }
/*     */ 
/*     */   public int binarySearch(int value, int fromIndex, int toIndex)
/*     */   {
/* 673 */     if (fromIndex < 0) {
/* 674 */       throw new ArrayIndexOutOfBoundsException(fromIndex);
/*     */     }
/* 676 */     if (toIndex > this._pos) {
/* 677 */       throw new ArrayIndexOutOfBoundsException(toIndex);
/*     */     }
/*     */ 
/* 680 */     int low = fromIndex;
/* 681 */     int high = toIndex - 1;
/*     */ 
/* 683 */     while (low <= high) {
/* 684 */       int mid = low + high >> 1;
/* 685 */       int midVal = this._data[mid];
/*     */ 
/* 687 */       if (midVal < value)
/* 688 */         low = mid + 1;
/* 689 */       else if (midVal > value)
/* 690 */         high = mid - 1;
/*     */       else {
/* 692 */         return mid;
/*     */       }
/*     */     }
/* 695 */     return -(low + 1);
/*     */   }
/*     */ 
/*     */   public int indexOf(int value)
/*     */   {
/* 708 */     return indexOf(0, value);
/*     */   }
/*     */ 
/*     */   public int indexOf(int offset, int value)
/*     */   {
/* 723 */     for (int i = offset; i < this._pos; i++) {
/* 724 */       if (this._data[i] == value) {
/* 725 */         return i;
/*     */       }
/*     */     }
/* 728 */     return -1;
/*     */   }
/*     */ 
/*     */   public int lastIndexOf(int value)
/*     */   {
/* 741 */     return lastIndexOf(this._pos, value);
/*     */   }
/*     */ 
/*     */   public int lastIndexOf(int offset, int value)
/*     */   {
/* 756 */     for (int i = offset; i-- > 0; ) {
/* 757 */       if (this._data[i] == value) {
/* 758 */         return i;
/*     */       }
/*     */     }
/* 761 */     return -1;
/*     */   }
/*     */ 
/*     */   public boolean contains(int value)
/*     */   {
/* 771 */     return lastIndexOf(value) >= 0;
/*     */   }
/*     */ 
/*     */   public TIntArrayList grep(TIntProcedure condition)
/*     */   {
/* 782 */     TIntArrayList list = new TIntArrayList();
/* 783 */     for (int i = 0; i < this._pos; i++) {
/* 784 */       if (condition.execute(this._data[i])) {
/* 785 */         list.add(this._data[i]);
/*     */       }
/*     */     }
/* 788 */     return list;
/*     */   }
/*     */ 
/*     */   public TIntArrayList inverseGrep(TIntProcedure condition)
/*     */   {
/* 799 */     TIntArrayList list = new TIntArrayList();
/* 800 */     for (int i = 0; i < this._pos; i++) {
/* 801 */       if (!condition.execute(this._data[i])) {
/* 802 */         list.add(this._data[i]);
/*     */       }
/*     */     }
/* 805 */     return list;
/*     */   }
/*     */ 
/*     */   public int max()
/*     */   {
/* 815 */     if (size() == 0) {
/* 816 */       throw new IllegalStateException("cannot find maximum of an empty list");
/*     */     }
/* 818 */     int max = this._data[(this._pos - 1)];
/* 819 */     for (int i = this._pos - 1; i-- > 0; ) {
/* 820 */       max = Math.max(max, this._data[this._pos]);
/*     */     }
/* 822 */     return max;
/*     */   }
/*     */ 
/*     */   public int min()
/*     */   {
/* 832 */     if (size() == 0) {
/* 833 */       throw new IllegalStateException("cannot find minimum of an empty list");
/*     */     }
/* 835 */     int min = this._data[(this._pos - 1)];
/* 836 */     for (int i = this._pos - 1; i-- > 0; ) {
/* 837 */       min = Math.min(min, this._data[this._pos]);
/*     */     }
/* 839 */     return min;
/*     */   }
/*     */ 
/*     */   public String toString()
/*     */   {
/* 850 */     StringBuffer buf = new StringBuffer("{");
/* 851 */     forEach(new TIntProcedure(buf) { private final StringBuffer val$buf;
/*     */ 
/* 853 */       public boolean execute(int val) { this.val$buf.append(val);
/* 854 */         this.val$buf.append(", ");
/* 855 */         return true;
/*     */       }
/*     */     });
/* 858 */     buf.append("}");
/* 859 */     return buf.toString();
/*     */   }
/*     */ 
/*     */   private void writeObject(ObjectOutputStream stream) throws IOException
/*     */   {
/* 864 */     stream.defaultWriteObject();
/*     */ 
/* 867 */     stream.writeInt(size());
/*     */ 
/* 869 */     SerializationProcedure writeProcedure = new SerializationProcedure(stream);
/* 870 */     if (!forEach(writeProcedure))
/* 871 */       throw writeProcedure.exception;
/*     */   }
/*     */ 
/*     */   private void readObject(ObjectInputStream stream)
/*     */     throws IOException, ClassNotFoundException
/*     */   {
/* 877 */     stream.defaultReadObject();
/*     */ 
/* 879 */     int size = stream.readInt();
/* 880 */     this._data = new int[size];
/* 881 */     while (size-- > 0) {
/* 882 */       int val = stream.readInt();
/* 883 */       add(val);
/*     */     }
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/thirdparty-all.jar
 * Qualified Name:     gnu.trove.TIntArrayList
 * JD-Core Version:    0.6.0
 */