/*     */ package org.jboss.ws.extensions.security;
/*     */ 
/*     */ import java.io.BufferedReader;
/*     */ import java.io.File;
/*     */ import java.io.InputStream;
/*     */ import java.io.InputStreamReader;
/*     */ import java.lang.reflect.Constructor;
/*     */ import java.lang.reflect.Method;
/*     */ import java.math.BigInteger;
/*     */ import java.net.MalformedURLException;
/*     */ import java.net.URL;
/*     */ import java.security.InvalidAlgorithmParameterException;
/*     */ import java.security.KeyStore;
/*     */ import java.security.KeyStoreException;
/*     */ import java.security.PrivateKey;
/*     */ import java.security.cert.CertPath;
/*     */ import java.security.cert.CertPathValidator;
/*     */ import java.security.cert.CertPathValidatorException;
/*     */ import java.security.cert.Certificate;
/*     */ import java.security.cert.CertificateFactory;
/*     */ import java.security.cert.PKIXParameters;
/*     */ import java.security.cert.X509Certificate;
/*     */ import java.util.ArrayList;
/*     */ import java.util.Arrays;
/*     */ import java.util.Enumeration;
/*     */ import java.util.HashMap;
/*     */ import java.util.List;
/*     */ import java.util.StringTokenizer;
/*     */ import org.jboss.logging.Logger;
/*     */ 
/*     */ public class SecurityStore
/*     */ {
/*  63 */   private static Logger log = Logger.getLogger(SecurityStore.class);
/*     */   private KeyStore keyStore;
/*     */   private String keyStorePassword;
/*     */   private KeyStore trustStore;
/*     */   private String trustStorePassword;
/*     */   private HashMap<String, String> keyPasswords;
/*     */ 
/*     */   public SecurityStore()
/*     */     throws WSSecurityException
/*     */   {
/*  77 */     this(null, null, null, null, null, null, null);
/*     */   }
/*     */ 
/*     */   public SecurityStore(URL keyStoreURL, String keyStoreType, String keyStorePassword, HashMap<String, String> keyPasswords) throws WSSecurityException
/*     */   {
/*  82 */     loadKeyStore(keyStoreURL, keyStoreType, keyStorePassword);
/*  83 */     loadTrustStore(keyStoreURL, keyStoreType, keyStorePassword);
/*  84 */     this.keyPasswords = keyPasswords;
/*     */   }
/*     */ 
/*     */   public SecurityStore(URL keyStoreURL, String keyStoreType, String keyStorePassword, HashMap<String, String> keyPasswords, URL trustStoreURL, String trustStoreType, String trustStorePassword)
/*     */     throws WSSecurityException
/*     */   {
/*  90 */     loadKeyStore(keyStoreURL, keyStoreType, keyStorePassword);
/*  91 */     loadTrustStore(trustStoreURL, trustStoreType, trustStorePassword);
/*  92 */     this.keyPasswords = keyPasswords;
/*     */   }
/*     */ 
/*     */   private void loadKeyStore(URL keyStoreURL, String keyStoreType, String keyStorePassword) throws WSSecurityException
/*     */   {
/*  97 */     if (keyStorePassword == null) {
/*  98 */       keyStorePassword = System.getProperty("org.jboss.ws.wsse.keyStorePassword");
/*     */     }
/* 100 */     this.keyStore = loadStore("org.jboss.ws.wsse.keyStore", "Keystore", keyStoreURL, keyStoreType, keyStorePassword);
/* 101 */     this.keyStorePassword = keyStorePassword;
/*     */   }
/*     */ 
/*     */   private void loadTrustStore(URL trustStoreURL, String trustStoreType, String trustStorePassword) throws WSSecurityException
/*     */   {
/* 106 */     if (trustStorePassword == null) {
/* 107 */       trustStorePassword = System.getProperty("org.jboss.ws.wsse.trustStorePassword");
/*     */     }
/* 109 */     this.trustStore = loadStore("org.jboss.ws.wsse.trustStore", "Truststore", trustStoreURL, trustStoreType, trustStorePassword);
/* 110 */     this.trustStorePassword = trustStorePassword;
/*     */   }
/*     */ 
/*     */   private KeyStore loadStore(String property, String type, URL storeURL, String storeType, String storePassword) throws WSSecurityException
/*     */   {
/* 115 */     if (storeURL == null)
/*     */     {
/* 117 */       String defaultStore = System.getProperty(property);
/* 118 */       if (defaultStore == null)
/*     */       {
/* 120 */         return null;
/*     */       }
/*     */ 
/* 123 */       File storeFile = new File(defaultStore);
/*     */       try
/*     */       {
/* 126 */         storeURL = storeFile.toURL();
/*     */       }
/*     */       catch (MalformedURLException e)
/*     */       {
/* 130 */         throw new WSSecurityException("Problems loading " + type + ": " + e.getMessage(), e);
/*     */       }
/*     */     }
/*     */ 
/* 134 */     if (storeType == null)
/* 135 */       storeType = System.getProperty(property + "Type");
/* 136 */     if (storeType == null) {
/* 137 */       storeType = "jks";
/*     */     }
/* 139 */     KeyStore keyStore = null;
/*     */     try
/*     */     {
/* 142 */       log.debug("loadStore: " + storeURL);
/* 143 */       InputStream stream = storeURL.openStream();
/* 144 */       if (stream == null) {
/* 145 */         throw new WSSecurityException("Cannot load store from: " + storeURL);
/*     */       }
/* 147 */       keyStore = KeyStore.getInstance(storeType);
/* 148 */       if (keyStore == null) {
/* 149 */         throw new WSSecurityException("Cannot get keystore for type: " + storeType);
/*     */       }
/* 151 */       String decryptedPassword = decryptPassword(storePassword);
/* 152 */       if (decryptedPassword == null) {
/* 153 */         throw new WSSecurityException("Cannot decrypt store password");
/*     */       }
/* 155 */       keyStore.load(stream, decryptedPassword.toCharArray());
/*     */     }
/*     */     catch (RuntimeException rte)
/*     */     {
/* 159 */       throw rte;
/*     */     }
/*     */     catch (WSSecurityException ex)
/*     */     {
/* 163 */       throw ex;
/*     */     }
/*     */     catch (Exception ex)
/*     */     {
/* 167 */       throw new WSSecurityException("Problems loading " + type + ": " + ex.getMessage(), ex);
/*     */     }
/*     */ 
/* 170 */     return keyStore;
/*     */   }
/*     */ 
/*     */   private String decryptPassword(String password)
/*     */     throws WSSecurityException
/*     */   {
/* 179 */     log.trace("decrypt password: " + password);
/*     */ 
/* 181 */     if (password == null) {
/* 182 */       throw new WSSecurityException("Invalid null password for security store");
/*     */     }
/* 184 */     if (password.charAt(0) == '{')
/*     */     {
/* 186 */       StringTokenizer tokenizer = new StringTokenizer(password, "{}");
/* 187 */       String keyStorePasswordCmdType = tokenizer.nextToken();
/* 188 */       String keyStorePasswordCmd = tokenizer.nextToken();
/* 189 */       if (keyStorePasswordCmdType.equals("EXT"))
/*     */       {
/* 191 */         password = execPasswordCmd(keyStorePasswordCmd);
/*     */       }
/* 193 */       else if (keyStorePasswordCmdType.equals("CLASS"))
/*     */       {
/* 195 */         password = invokePasswordClass(keyStorePasswordCmd);
/*     */       }
/*     */       else
/*     */       {
/* 199 */         throw new WSSecurityException("Unknown keyStorePasswordCmdType: " + keyStorePasswordCmdType);
/*     */       }
/*     */     }
/* 202 */     if (password == null) {
/* 203 */       throw new WSSecurityException("Cannot decrypt password, result is null");
/*     */     }
/* 205 */     log.trace("decrypted password: " + password);
/* 206 */     return password;
/*     */   }
/*     */ 
/*     */   private String execPasswordCmd(String keyStorePasswordCmd) throws WSSecurityException
/*     */   {
/* 211 */     log.debug("Executing cmd: " + keyStorePasswordCmd);
/*     */     try
/*     */     {
/* 214 */       String password = null;
/* 215 */       Runtime rt = Runtime.getRuntime();
/* 216 */       Process p = rt.exec(keyStorePasswordCmd);
/* 217 */       int status = p.waitFor();
/* 218 */       if (status == 0)
/*     */       {
/* 220 */         InputStream stdin = p.getInputStream();
/* 221 */         BufferedReader reader = new BufferedReader(new InputStreamReader(stdin));
/* 222 */         password = reader.readLine();
/* 223 */         stdin.close();
/*     */       }
/*     */       else
/*     */       {
/* 227 */         InputStream stderr = p.getErrorStream();
/* 228 */         BufferedReader reader = new BufferedReader(new InputStreamReader(stderr));
/* 229 */         String line = reader.readLine();
/* 230 */         while (line != null)
/*     */         {
/* 232 */           log.error(line);
/* 233 */           line = reader.readLine();
/*     */         }
/* 235 */         stderr.close();
/*     */       }
/* 237 */       log.debug("Command exited with: " + status);
/* 238 */       return password;
/*     */     }
/*     */     catch (Exception e) {
/*     */     }
/* 242 */     throw new WSSecurityException("Problems executing password cmd: " + keyStorePasswordCmd, e);
/*     */   }
/*     */ 
/*     */   private String invokePasswordClass(String keyStorePasswordCmd)
/*     */     throws WSSecurityException
/*     */   {
/* 248 */     String password = null;
/* 249 */     String classname = keyStorePasswordCmd;
/* 250 */     String ctorArg = null;
/* 251 */     int colon = keyStorePasswordCmd.indexOf(':');
/* 252 */     if (colon > 0)
/*     */     {
/* 254 */       classname = keyStorePasswordCmd.substring(0, colon);
/* 255 */       ctorArg = keyStorePasswordCmd.substring(colon + 1);
/*     */     }
/* 257 */     log.debug("Loading class: " + classname + ", ctorArg=" + ctorArg);
/*     */     try
/*     */     {
/* 260 */       ClassLoader loader = Thread.currentThread().getContextClassLoader();
/* 261 */       Class c = loader.loadClass(classname);
/* 262 */       Object instance = null;
/* 263 */       if (ctorArg != null)
/*     */       {
/* 265 */         Class[] sig = { String.class };
/* 266 */         Constructor ctor = c.getConstructor(sig);
/* 267 */         Object[] args = { ctorArg };
/* 268 */         instance = ctor.newInstance(args);
/*     */       }
/*     */       else
/*     */       {
/* 272 */         instance = c.newInstance();
/*     */       }
/*     */       try
/*     */       {
/* 276 */         log.debug("Checking for toCharArray");
/* 277 */         Class[] sig = new Class[0];
/* 278 */         Method toCharArray = c.getMethod("toCharArray", sig);
/* 279 */         Object[] args = new Object[0];
/* 280 */         log.debug("Invoking toCharArray");
/* 281 */         password = new String((char[])(char[])toCharArray.invoke(instance, args));
/*     */       }
/*     */       catch (NoSuchMethodException e)
/*     */       {
/* 285 */         log.debug("No toCharArray found, invoking toString");
/* 286 */         password = instance.toString();
/*     */       }
/*     */     }
/*     */     catch (Exception e)
/*     */     {
/* 291 */       throw new WSSecurityException("Problems loading or invoking Password class : " + classname, e);
/*     */     }
/* 293 */     return password;
/*     */   }
/*     */ 
/*     */   public static byte[] getSubjectKeyIdentifier(X509Certificate cert)
/*     */   {
/* 299 */     byte[] encoded = cert.getExtensionValue("2.5.29.14");
/* 300 */     if (encoded == null) {
/* 301 */       return null;
/*     */     }
/*     */ 
/* 304 */     int trunc = encoded.length - 4;
/*     */ 
/* 306 */     byte[] identifier = new byte[trunc];
/* 307 */     System.arraycopy(encoded, 4, identifier, 0, trunc);
/*     */ 
/* 309 */     return identifier;
/*     */   }
/*     */ 
/*     */   public X509Certificate getCertificate(String alias) throws WSSecurityException
/*     */   {
/* 314 */     if (this.keyStore == null)
/*     */     {
/* 316 */       throw new WSSecurityException("KeyStore not set.");
/*     */     }
/*     */     X509Certificate cert;
/*     */     try
/*     */     {
/* 322 */       cert = (X509Certificate)this.keyStore.getCertificate(alias);
/*     */     }
/*     */     catch (Exception e)
/*     */     {
/* 326 */       throw new WSSecurityException("Problems retrieving cert: " + e.getMessage(), e);
/*     */     }
/*     */ 
/* 329 */     if (cert == null) {
/* 330 */       throw new WSSecurityException("Certificate (" + alias + ") not in keystore");
/*     */     }
/* 332 */     return cert;
/*     */   }
/*     */ 
/*     */   public X509Certificate getCertificateBySubjectKeyIdentifier(byte[] identifier) throws WSSecurityException
/*     */   {
/* 337 */     if (identifier == null) {
/* 338 */       return null;
/*     */     }
/* 340 */     if (this.keyStore == null)
/*     */     {
/* 342 */       throw new WSSecurityException("KeyStore not set.");
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 347 */       Enumeration i = this.keyStore.aliases();
/*     */ 
/* 349 */       while (i.hasMoreElements())
/*     */       {
/* 351 */         String alias = (String)i.nextElement();
/* 352 */         Certificate cert = this.keyStore.getCertificate(alias);
/* 353 */         if (!(cert instanceof X509Certificate)) {
/*     */           continue;
/*     */         }
/* 356 */         byte[] subjectKeyIdentifier = getSubjectKeyIdentifier((X509Certificate)cert);
/* 357 */         if (subjectKeyIdentifier == null) {
/*     */           continue;
/*     */         }
/* 360 */         if (Arrays.equals(identifier, subjectKeyIdentifier))
/* 361 */           return (X509Certificate)cert;
/*     */       }
/*     */     }
/*     */     catch (KeyStoreException e)
/*     */     {
/* 366 */       throw new WSSecurityException("Problems retrieving cert: " + e.getMessage(), e);
/*     */     }
/*     */ 
/* 369 */     return null;
/*     */   }
/*     */ 
/*     */   public X509Certificate getCertificateByIssuerSerial(String issuer, String serial) throws WSSecurityException
/*     */   {
/* 374 */     if (this.keyStore == null)
/*     */     {
/* 376 */       throw new WSSecurityException("KeyStore not set.");
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 381 */       Enumeration i = this.keyStore.aliases();
/*     */ 
/* 383 */       while (i.hasMoreElements())
/*     */       {
/* 385 */         String alias = (String)i.nextElement();
/* 386 */         Certificate cert = this.keyStore.getCertificate(alias);
/* 387 */         if (!(cert instanceof X509Certificate)) {
/*     */           continue;
/*     */         }
/* 390 */         X509Certificate x509 = (X509Certificate)cert;
/* 391 */         if ((issuer.equals(x509.getIssuerDN().toString())) && (serial.equals(x509.getSerialNumber().toString())))
/* 392 */           return x509;
/*     */       }
/*     */     }
/*     */     catch (KeyStoreException e)
/*     */     {
/* 397 */       throw new WSSecurityException("Problems retrieving cert: " + e.getMessage(), e);
/*     */     }
/*     */ 
/* 400 */     return null;
/*     */   }
/*     */ 
/*     */   public PrivateKey getPrivateKey(String alias) throws WSSecurityException
/*     */   {
/* 405 */     if (this.keyStore == null)
/*     */     {
/* 407 */       throw new WSSecurityException("KeyStore not set.");
/*     */     }
/*     */     PrivateKey key;
/*     */     try
/*     */     {
/* 413 */       String password = this.keyStorePassword;
/* 414 */       if ((this.keyPasswords != null) && (this.keyPasswords.containsKey(alias)))
/* 415 */         password = (String)this.keyPasswords.get(alias);
/* 416 */       key = (PrivateKey)this.keyStore.getKey(alias, decryptPassword(password).toCharArray());
/*     */     }
/*     */     catch (Exception e)
/*     */     {
/* 420 */       throw new WSSecurityException("Problems retrieving private key: " + e.getMessage(), e);
/*     */     }
/*     */ 
/* 423 */     if (key == null) {
/* 424 */       throw new WSSecurityException("Private key (" + alias + ") not in keystore");
/*     */     }
/* 426 */     return key;
/*     */   }
/*     */ 
/*     */   public PrivateKey getPrivateKey(X509Certificate cert) throws WSSecurityException
/*     */   {
/* 431 */     if (this.keyStore == null)
/*     */     {
/* 433 */       throw new WSSecurityException("KeyStore not set.");
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 438 */       String alias = this.keyStore.getCertificateAlias(cert);
/* 439 */       return getPrivateKey(alias);
/*     */     }
/*     */     catch (Exception e) {
/*     */     }
/* 443 */     throw new WSSecurityException("Problems retrieving private key: " + e.getMessage(), e);
/*     */   }
/*     */ 
/*     */   public void validateCertificate(X509Certificate cert)
/*     */     throws WSSecurityException
/*     */   {
/*     */     try
/*     */     {
/* 451 */       cert.checkValidity();
/*     */     }
/*     */     catch (Exception e)
/*     */     {
/* 455 */       log.debug("Certificate is invalid", e);
/* 456 */       throw new FailedAuthenticationException();
/*     */     }
/*     */ 
/* 459 */     if (this.keyStore == null)
/*     */     {
/* 461 */       throw new WSSecurityException("TrustStore not set.");
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 467 */       if (this.trustStore.getCertificateAlias(cert) != null)
/*     */       {
/* 469 */         return;
/*     */       }
/*     */     }
/*     */     catch (KeyStoreException e)
/*     */     {
/* 474 */       throw new WSSecurityException("Problems searching truststore", e);
/* 477 */     }
/*     */ List list = new ArrayList(1);
/* 478 */     list.add(cert);
/*     */     CertPath cp;
/*     */     CertPathValidator cpv;
/*     */     PKIXParameters parameters;
/*     */     try { cp = CertificateFactory.getInstance("X.509").generateCertPath(list);
/* 487 */       cpv = CertPathValidator.getInstance("PKIX");
/* 488 */       parameters = new PKIXParameters(this.trustStore);
/*     */ 
/* 491 */       parameters.setRevocationEnabled(false);
/*     */     }
/*     */     catch (Exception e)
/*     */     {
/* 495 */       throw new WSSecurityException("Problems setting up certificate validation", e);
/*     */     }
/*     */ 
/*     */     try
/*     */     {
/* 500 */       cpv.validate(cp, parameters);
/*     */     }
/*     */     catch (CertPathValidatorException cpve)
/*     */     {
/* 504 */       log.debug("Certificate is invalid:", cpve);
/* 505 */       throw new FailedAuthenticationException();
/*     */     }
/*     */     catch (InvalidAlgorithmParameterException e)
/*     */     {
/* 509 */       throw new WSSecurityException("Problems setting up certificate validation", e);
/*     */     }
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.ws.extensions.security.SecurityStore
 * JD-Core Version:    0.6.0
 */