/*     */ package org.jboss.ws.extensions.security;
/*     */ 
/*     */ import java.security.PrivateKey;
/*     */ import java.security.PublicKey;
/*     */ import java.security.cert.X509Certificate;
/*     */ import java.util.HashMap;
/*     */ import org.apache.xml.security.keys.KeyInfo;
/*     */ import org.jboss.util.NotImplementedException;
/*     */ import org.jboss.ws.extensions.security.element.BinarySecurityToken;
/*     */ import org.jboss.ws.extensions.security.element.DirectReference;
/*     */ import org.jboss.ws.extensions.security.element.KeyIdentifier;
/*     */ import org.jboss.ws.extensions.security.element.Reference;
/*     */ import org.jboss.ws.extensions.security.element.SecurityTokenReference;
/*     */ import org.jboss.ws.extensions.security.element.X509IssuerSerial;
/*     */ import org.jboss.ws.extensions.security.element.X509Token;
/*     */ import org.w3c.dom.Element;
/*     */ 
/*     */ public class KeyResolver
/*     */ {
/*  50 */   private HashMap<String, BinarySecurityToken> tokenCache = new HashMap();
/*     */   private SecurityStore store;
/*     */ 
/*     */   public KeyResolver(SecurityStore store)
/*     */   {
/*  56 */     this.store = store;
/*     */   }
/*     */ 
/*     */   private SecurityTokenReference extractSecurityTokenReference(KeyInfo info) throws WSSecurityException
/*     */   {
/*  61 */     Element child = Util.getFirstChildElement(info.getElement());
/*  62 */     if (child == null) {
/*  63 */       throw new WSSecurityException("Empty KeyInfo");
/*     */     }
/*  65 */     if (!child.getLocalName().equals("SecurityTokenReference")) {
/*  66 */       throw new WSSecurityException("KeyInfo did not contain expected SecurityTokenReference, instead got: " + child.getLocalName());
/*     */     }
/*  68 */     return new SecurityTokenReference(child);
/*     */   }
/*     */ 
/*     */   public void cacheToken(BinarySecurityToken token)
/*     */   {
/*  73 */     this.tokenCache.put(token.getId(), token);
/*     */   }
/*     */ 
/*     */   public BinarySecurityToken resolve(SecurityTokenReference reference) throws WSSecurityException
/*     */   {
/*  78 */     Reference ref = reference.getReference();
/*  79 */     if ((ref instanceof DirectReference))
/*     */     {
/*  81 */       DirectReference direct = (DirectReference)ref;
/*  82 */       return resolveDirectReference(direct);
/*     */     }
/*  84 */     if ((ref instanceof KeyIdentifier))
/*     */     {
/*  86 */       KeyIdentifier identifier = (KeyIdentifier)ref;
/*  87 */       return resolveKeyIdentifier(identifier);
/*     */     }
/*  89 */     if ((ref instanceof X509IssuerSerial))
/*     */     {
/*  91 */       X509IssuerSerial issuerSerial = (X509IssuerSerial)ref;
/*  92 */       return resolveX509IssuerSerial(issuerSerial);
/*     */     }
/*     */ 
/*  95 */     throw new NotImplementedException("Currently only DirectReference is supported!");
/*     */   }
/*     */ 
/*     */   private BinarySecurityToken resolveDirectReference(DirectReference direct) throws WSSecurityException
/*     */   {
/* 100 */     String id = direct.getUri().substring(1);
/*     */ 
/* 102 */     BinarySecurityToken token = (BinarySecurityToken)this.tokenCache.get(id);
/* 103 */     if (token == null) {
/* 104 */       throw new SecurityTokenUnavailableException("Could not resolve token id: " + id);
/*     */     }
/* 106 */     return token;
/*     */   }
/*     */ 
/*     */   private BinarySecurityToken resolveKeyIdentifier(KeyIdentifier identifier)
/*     */     throws WSSecurityException
/*     */   {
/* 112 */     X509Certificate cert = this.store.getCertificateBySubjectKeyIdentifier(identifier.getIdentifier());
/* 113 */     if (cert == null)
/* 114 */       throw new SecurityTokenUnavailableException("Could not locate certificate by key identifier");
/* 115 */     return new X509Token(cert, identifier.getDocument());
/*     */   }
/*     */ 
/*     */   private BinarySecurityToken resolveX509IssuerSerial(X509IssuerSerial issuerSerial) throws WSSecurityException
/*     */   {
/* 120 */     X509Certificate cert = this.store.getCertificateByIssuerSerial(issuerSerial.getIssuer(), issuerSerial.getSerial());
/* 121 */     if (cert == null) {
/* 122 */       throw new SecurityTokenUnavailableException("Could not locate certificate by issuer and serial number");
/*     */     }
/* 124 */     return new X509Token(cert, issuerSerial.getDocument());
/*     */   }
/*     */ 
/*     */   public X509Certificate resolveCertificate(SecurityTokenReference reference) throws WSSecurityException
/*     */   {
/* 129 */     BinarySecurityToken token = resolve(reference);
/*     */ 
/* 131 */     if (!(token instanceof X509Token)) {
/* 132 */       throw new WSSecurityException("Expected X509Token, cache contained: " + token.getClass().getName());
/*     */     }
/* 134 */     return ((X509Token)token).getCert();
/*     */   }
/*     */ 
/*     */   public PublicKey resolvePublicKey(SecurityTokenReference reference) throws WSSecurityException
/*     */   {
/* 139 */     return resolveCertificate(reference).getPublicKey();
/*     */   }
/*     */ 
/*     */   public PrivateKey resolvePrivateKey(SecurityTokenReference reference) throws WSSecurityException
/*     */   {
/* 144 */     return this.store.getPrivateKey(resolveCertificate(reference));
/*     */   }
/*     */ 
/*     */   public BinarySecurityToken resolve(KeyInfo info) throws WSSecurityException
/*     */   {
/* 149 */     return resolve(extractSecurityTokenReference(info));
/*     */   }
/*     */ 
/*     */   public X509Certificate resolveCertificate(KeyInfo info) throws WSSecurityException
/*     */   {
/* 154 */     return resolveCertificate(extractSecurityTokenReference(info));
/*     */   }
/*     */ 
/*     */   public PublicKey resolvePublicKey(KeyInfo info) throws WSSecurityException
/*     */   {
/* 159 */     return resolvePublicKey(extractSecurityTokenReference(info));
/*     */   }
/*     */ 
/*     */   public PrivateKey resolvePrivateKey(KeyInfo info) throws WSSecurityException
/*     */   {
/* 164 */     return resolvePrivateKey(extractSecurityTokenReference(info));
/*     */   }
/*     */ }

/* Location:           /home/mnovotny/projects/EMBEDDED_JBOSS_BETA3_COMMUNITY/embedded/output/lib/embedded-jboss/lib/jboss-embedded-all.jar
 * Qualified Name:     org.jboss.ws.extensions.security.KeyResolver
 * JD-Core Version:    0.6.0
 */