/*
 * Copyright (c) 2022, Peter Abeles. All Rights Reserved.
 *
 * This file is part of Efficient Java Matrix Library (EJML).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ejml.data;

import org.ejml.MatrixDimensionException;
import org.ejml.UtilEjml;

import static org.ejml.UtilEjml.checkValidMatrixShapeComplex;

/**
 * A generic abstract class for matrices whose data is stored in a single 1D array of doubles. The
 * format of the elements in this array is not specified. For example row major, column major,
 * and block row major are all common formats.
 *
 * @author Peter Abeles
 */
public abstract class ZMatrixD1 implements ZMatrix, ReshapeMatrix {
    /**
     * Where the raw data for the matrix is stored. The format is type dependent.
     */
    public double[] data = UtilEjml.ZERO_LENGTH_F64;

    /**
     * Number of rows in the matrix.
     */
    public int numRows;
    /**
     * Number of columns in the matrix.
     */
    public int numCols;

    /** Sets the matrix's shape while checking that it's valid */
    protected void assignShape(int rows, int cols) {
        checkValidMatrixShapeComplex(rows, cols);
        this.numRows = rows;
        this.numCols = cols;
    }

    /**
     * Used to get a reference to the internal data.
     *
     * @return Reference to the matrix's data.
     */
    public double[] getData() {
        return data;
    }

    /**
     * Changes the internal array reference.
     */
    public void setData( double[] data ) {
        this.data = data;
    }

    /**
     * Returns the internal array index for the specified row and column.
     *
     * @param row Row index.
     * @param col Column index.
     * @return Internal array index.
     */
    public abstract int getIndex( int row, int col );

    /**
     * Sets the value of this matrix to be the same as the value of the provided matrix. Both
     * matrices must have the same shape:<br>
     * <br>
     * a<sub>ij</sub> = b<sub>ij</sub><br>
     * <br>
     *
     * @param b The matrix that this matrix is to be set equal to.
     */
    public void setTo( ZMatrixD1 b ) {
        if (numRows != b.numRows || numCols != b.numCols) {
            throw new MatrixDimensionException("The two matrices do not have compatible shapes.");
        }

        int dataLength = b.getDataLength();

        System.arraycopy(b.data, 0, this.data, 0, dataLength);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getNumRows() {
        return numRows;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getNumCols() {
        return numCols;
    }

    /**
     * Sets the number of rows.
     *
     * @param numRows Number of rows
     */
    public void setNumRows( int numRows ) {
        this.numRows = numRows;
    }

    /**
     * Sets the number of columns.
     *
     * @param numCols Number of columns
     */
    public void setNumCols( int numCols ) {
        this.numCols = numCols;
    }

    public int getNumElements() {
        return numRows*numCols;
    }
}
