/*
 * #%L
 * Pollen :: Vote Counting
 * 
 * $Id: ImportExportService.java 3257 2012-04-13 11:28:02Z fdesbois $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1.1/pollen-votecounting/src/main/java/org/chorem/pollen/votecounting/services/ImportExportService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting.services;

import com.google.common.collect.Lists;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.chorem.pollen.votecounting.dto.ChoiceDTO;
import org.chorem.pollen.votecounting.dto.CommentDTO;
import org.chorem.pollen.votecounting.dto.PollChoiceDTO;
import org.chorem.pollen.votecounting.dto.PollDTO;
import org.chorem.pollen.votecounting.dto.PollExportDTO;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.chorem.pollen.votecounting.dto.VoteToChoiceDTO;
import org.chorem.pollen.votecounting.dto.VotingGroupDTO;
import org.chorem.pollen.votecounting.dto.VotingPersonDTO;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import java.io.File;
import java.io.IOException;
import java.util.List;

/**
 * Service to import / export a poll to or from his xml representation.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class ImportExportService {

    public static final String POLL = "poll";

    public static final String TITLE1 = "title";

    public static final String TITLE = TITLE1;

    public static final String CHOICE = "choice";

    public static final String ID_CHOICE = "idChoice";

    public static final String VALUE = "value";

    public static final String PERCENTAGE = "percentage";

    public static final String NB_VOTES = "nbVotes";

    public static final String IS_BY_GROUP = "isByGroup";

    public static final String CHOICES = "choices";

    public static final String CHOICE_RESULT = "choiceResult";

    public static final String ID_POLL = "idPoll";

    public static final String VOTE_COUNTING_RESULT = "voteCountingResult";

    public static final String DESCRIPTION = "description";

    public static final String NAME = "name";

    public static final String POLL_CHOICE = "pollChoice";

    public static final String VOTING_PERSON = "votingPerson";

    public static final String VOTING_ID = "votingId";

    public static final String WEIGHT = "weight";

    public static final String EMAIL = "email";

    public static final String COMMENT = "comment";

    public static final String VOTING_PERSONS = "votingPersons";

    public static final String ID_GROUP = "idGroup";

    public static final String GROUP = "group";

    public static final String POLL_CHOICES = "pollChoices";

    public static final String VOTE_COUNTING_RESULTS = "voteCountingResults";

    public static final String GROUPS = "groups";

    public static final String COMMENTS = "comments";

    public static final String CREATOR_ID = "creatorId";

    public static final String CREATOR_EMAIL = "creatorEmail";

    public static final String MAX_CHOICE_NB = "maxChoiceNb";

    public static final String POLL_EXPORT = "pollExport";

    public static final String POLL_ID = "pollId";

    public static final String POLL_TYPE = "pollType";

    public static final String CHOICE_TYPE = "choiceType";

    public static final String VOTE_COUNTING = "voteCounting";

    public static final String CLOSED = "closed";

    public static final String CHOICE_ADD_ALLOWED = "choiceAddAllowed";

    public static final String ANONYMOUS_VOTE_ALLOWED = "anonymousVoteAllowed";

    public static final String ANONYMOUS = "anonymous";

    public static final String PUBLIC_RESULTS = "publicResults";

    public static final String CONTINUOUS_RESULTS = "continuousResults";

    public static final String VALUE_TRUE = "true";

    public static final String VALUE_FALSE = "false";

    /** Logger. */
    private static final Log log = LogFactory.getLog(ImportExportService.class);

    /**
     * Export the given poll to an xml representation.
     *
     * @param exportPoll the poll to export
     * @return the xml representation of the given poll
     */
    public String exportToXml(PollExportDTO exportPoll) {

        Document document = toDOM(exportPoll);

        // Mise en forme classique avec getPrettyFormat()
        XMLOutputter sortie = new XMLOutputter(Format.getPrettyFormat());
        String content = sortie.outputString(document);

        if (log.isDebugEnabled()) {
            log.debug(content);
        }
        return content;
    }

    /**
     * Import into a {@link PollExportDTO} his xml representation from the
     * given file.
     *
     * @param file file to load
     * @return the loaded dto from the xml representation.
     * @throws JDOMException if could not load xml from file
     * @throws IOException   if could not load the file
     */
    public PollExportDTO importFromXml(File file) throws JDOMException, IOException {

        // Création d'un nouveau document JDOM avec le fichier XML en argument
        Document document = new SAXBuilder().build(file);

        // Initialisation d'un nouvel élément racine
        Element racine = document.getRootElement();

        PollExportDTO result = toDTO(racine);

        return result;
    }

    protected PollExportDTO toDTO(Element racine) {

        // Création du sondage
        Element pollElm = racine.getChild(POLL);

        PollDTO poll = new PollDTO(pollElm.getAttributeValue(POLL_ID));
        poll.setTitle(pollElm.getChild(TITLE).getText());
        poll.setDescription(pollElm.getChild(DESCRIPTION).getText());
        poll.setCreatorId(pollElm.getChild(CREATOR_ID).getText());
        poll.setCreatorEmail(pollElm.getChild(CREATOR_EMAIL).getText());
        poll.setMaxChoiceNb(Integer.parseInt(
                pollElm.getChild(MAX_CHOICE_NB).getText()));

        poll.setPollType(
                PollType.valueOf(pollElm.getAttributeValue(POLL_TYPE)));
        poll.setChoiceType(ChoiceType.valueOf(
                pollElm.getAttributeValue(CHOICE_TYPE)));
        poll.setVoteCounting(VoteCountingType.valueOf(
                pollElm.getAttributeValue(VOTE_COUNTING)));
        poll.setClosed(Boolean.valueOf(pollElm.getAttributeValue(CLOSED)));
        poll.setChoiceAddAllowed(Boolean.valueOf(
                pollElm.getAttributeValue(CHOICE_ADD_ALLOWED)));
        poll.setAnonymousVoteAllowed(Boolean.valueOf(
                pollElm.getAttributeValue(ANONYMOUS_VOTE_ALLOWED)));
        poll.setAnonymous(Boolean.valueOf(
                pollElm.getAttributeValue(ANONYMOUS)));
        poll.setPublicResults(Boolean.valueOf(pollElm.getAttributeValue(PUBLIC_RESULTS)));
        poll.setContinuousResults(Boolean.valueOf(
                pollElm.getAttributeValue(CONTINUOUS_RESULTS)));

        // Ajout des commentaires
        Element commentsElm = pollElm.getChild(COMMENTS);
        List<Element> listComments = commentsElm.getChildren(COMMENT);
        List vectorComments = toComments(listComments);

        poll.setComments(vectorComments);

        // Ajout des groupes
        Element groupsElm = pollElm.getChild(GROUPS);
        List<Element> listGroups = groupsElm.getChildren(GROUP);
        List vectorGroups = toVotingGroups(listGroups);

        poll.setVotingGroups(vectorGroups);

        // Ajout des choix
        Element pollChoicesElm = pollElm.getChild(POLL_CHOICES);
        List<Element> listPollChoices = pollChoicesElm.getChildren(POLL_CHOICE);
        List<PollChoiceDTO> vectorPollChoices = toPollChoices(listPollChoices);

        poll.setChoices(vectorPollChoices);

        // Ajout des résultats
        Element voteCountingResultsElm = racine.getChild(VOTE_COUNTING_RESULTS);
        List<Element> listVoteCountingResults =
                voteCountingResultsElm.getChildren(VOTE_COUNTING_RESULT);
        List<VoteCountingResultDTO> vectorVoteCountingResults =
                toVoteCountingResults(listVoteCountingResults);

        // Création du sondage PollExportDTO
        PollExportDTO pollExport = new PollExportDTO();
        pollExport.setPollId(poll.getPollId());
        pollExport.setPoll(poll);
        pollExport.setVoteCountingResults(vectorVoteCountingResults);

        return pollExport;
    }

    protected List<CommentDTO> toComments(List<Element> listComments) {
        List<CommentDTO> vectorComments = Lists.newArrayList();

        for (Element commentElm : listComments) {
            CommentDTO comment = new CommentDTO(
                    commentElm.getAttributeValue(VOTING_ID),
                    commentElm.getText());
            vectorComments.add(comment);
        }

        return vectorComments;
    }

    protected List<VotingGroupDTO> toVotingGroups(List<Element> listGroups) {
        List<VotingGroupDTO> vectorGroups = Lists.newArrayList();

        for (Element groupElm : listGroups) {
            VotingGroupDTO group = new VotingGroupDTO(
                    groupElm.getAttributeValue(ID_GROUP),
                    Double.parseDouble((groupElm.getAttributeValue(WEIGHT))));

            group.setName(groupElm.getAttributeValue(NAME));

            Element votingPersonsElm = groupElm.getChild(VOTING_PERSONS);
            List<Element> listVotingPersons = votingPersonsElm
                    .getChildren(VOTING_PERSON);
            List vectorVotingPersons = toVotingPersons(listVotingPersons);

            group.setVotingPersons(vectorVotingPersons);

            vectorGroups.add(group);
        }

        return vectorGroups;
    }

    protected List<VotingPersonDTO> toVotingPersons(List<Element> listVotingPersons) {
        List<VotingPersonDTO> vectorVotingPersons = Lists.newArrayList();

        for (Element votingPersonElm : listVotingPersons) {
            VotingPersonDTO votingPerson = new VotingPersonDTO(
                    votingPersonElm.getAttributeValue(VOTING_ID),
                    Double.parseDouble(votingPersonElm.getAttributeValue(WEIGHT)));

            votingPerson.setEmail(votingPersonElm.getChild(EMAIL).getValue());
            votingPerson.setComment(votingPersonElm.getChild(COMMENT)
                                            .getValue());

            Element choicesElm = votingPersonElm.getChild(CHOICES);

            List<Element> listVoteToChoices = choicesElm.getChildren(CHOICE);
            List vectorVoteToChoices = toVoteToChoices(listVoteToChoices);

            votingPerson.setChoices(vectorVoteToChoices);
            vectorVotingPersons.add(votingPerson);
        }

        return vectorVotingPersons;
    }

    protected List<VoteToChoiceDTO> toVoteToChoices(List<Element> listVoteToChoices) {
        List<VoteToChoiceDTO> vectorVoteToChoices = Lists.newArrayList();

        for (Element voteToChoiceElm : listVoteToChoices) {
            VoteToChoiceDTO voteToChoice = new VoteToChoiceDTO(
                    voteToChoiceElm.getAttributeValue(ID_CHOICE),
                    Double.parseDouble(voteToChoiceElm.getAttributeValue(VALUE)));
            vectorVoteToChoices.add(voteToChoice);
        }

        return vectorVoteToChoices;
    }

    protected List<PollChoiceDTO> toPollChoices(List<Element> listPollChoices) {
        List<PollChoiceDTO> vectorPollChoices = Lists.newArrayList();

        for (Element pollChoiceElm : listPollChoices) {
            PollChoiceDTO pollChoice = new PollChoiceDTO(
                    pollChoiceElm.getAttributeValue(ID_CHOICE));

            String pollChoiceName = pollChoiceElm.getChild(NAME).getValue();
            pollChoice.setName(pollChoiceName);
            pollChoice.setHidden(pollChoiceName != null
                                 && pollChoiceName.startsWith(NumberMethod.HIDDEN_PREFIX));

            pollChoice.setDescription(pollChoiceElm.getChild(DESCRIPTION)
                                              .getValue());

            vectorPollChoices.add(pollChoice);
        }

        return vectorPollChoices;
    }

    protected List<VoteCountingResultDTO> toVoteCountingResults(List<Element> listVoteCountingResults) {
        List<VoteCountingResultDTO> vectorVoteCountingResults = Lists.newArrayList();

        for (Element voteCountingResultElm : listVoteCountingResults) {
            VoteCountingResultDTO voteCountingResult = new VoteCountingResultDTO();
            voteCountingResult.setIdPoll(
                    voteCountingResultElm.getAttributeValue(ID_POLL));
            voteCountingResult.setNbVotes(
                    Integer.parseInt(voteCountingResultElm.getChild(NB_VOTES)
                                             .getValue()));
            voteCountingResult.setChoiceResult(voteCountingResultElm.getChild(
                    CHOICE_RESULT).getValue());
            voteCountingResult.setByGroup(
                    Boolean.parseBoolean(
                            voteCountingResultElm.getChild(IS_BY_GROUP).getValue()));

            Element choicesElm = voteCountingResultElm.getChild(CHOICES);
            List<Element> listChoices = choicesElm.getChildren(CHOICE);
            List vectorChoices = toChoices(listChoices);
            voteCountingResult.setChoices(vectorChoices);
            vectorVoteCountingResults.add(voteCountingResult);
        }

        return vectorVoteCountingResults;
    }

    protected List<ChoiceDTO> toChoices(List<Element> listChoices) {
        List<ChoiceDTO> vectorChoices = Lists.newArrayList();

        for (Element choiceElm : listChoices) {
            ChoiceDTO choice = new ChoiceDTO(
                    choiceElm.getAttribute(ID_CHOICE).getName(),
                    Double.parseDouble(choiceElm.getAttribute(VALUE).getValue()),
                    Double.parseDouble(choiceElm.getChild(PERCENTAGE).getValue()),
                    Integer.parseInt((choiceElm.getChild(NB_VOTES).getValue())));
            vectorChoices.add(choice);
        }

        return vectorChoices;
    }

    protected Document toDOM(PollExportDTO exportPoll) {

        PollDTO poll = exportPoll.getPoll();

        List<VoteCountingResultDTO> voteCountingResults =
                exportPoll.getVoteCountingResults();

        // Création de la racine et du document
        Element racine = new Element(POLL_EXPORT);
        Document document = new Document(racine);

        // Création du sondage
        Element pollElm = new Element(POLL);
        pollElm.setAttribute(POLL_ID, poll.getPollId());
        pollElm.setAttribute(POLL_TYPE, poll.getPollType().name());
        pollElm.setAttribute(CHOICE_TYPE, poll.getChoiceType().name());
        pollElm.setAttribute(VOTE_COUNTING, poll.getVoteCounting().name());
        pollElm.setAttribute(CLOSED, poll.isClosed() ? VALUE_TRUE : VALUE_FALSE);
        pollElm.setAttribute(CHOICE_ADD_ALLOWED,
                             poll.isChoiceAddAllowed() ? VALUE_TRUE : VALUE_FALSE);
        pollElm.setAttribute(ANONYMOUS_VOTE_ALLOWED,
                             poll.isAnonymousVoteAllowed() ? VALUE_TRUE : VALUE_FALSE);
        pollElm.setAttribute(ANONYMOUS,
                             poll.isAnonymous() ? VALUE_TRUE : VALUE_FALSE);
        pollElm.setAttribute(PUBLIC_RESULTS,
                             poll.isPublicResults() ? VALUE_TRUE : VALUE_FALSE);
        pollElm.setAttribute(CONTINUOUS_RESULTS,
                             poll.isContinuousResults() ? VALUE_TRUE : VALUE_FALSE);

        Element titleElm = new Element(TITLE1);
        titleElm.setText(poll.getTitle());
        pollElm.addContent(titleElm);

        Element descriptionElm = new Element(DESCRIPTION);
        descriptionElm.setText(poll.getDescription());
        pollElm.addContent(descriptionElm);

        Element creatorIdElm = new Element(CREATOR_ID);
        creatorIdElm.setText(poll.getCreatorId());
        pollElm.addContent(creatorIdElm);

        Element creatorEmailElm = new Element(CREATOR_EMAIL);
        creatorEmailElm.setText(poll.getCreatorEmail());
        pollElm.addContent(creatorEmailElm);

        Element maxChoiceNbElm = new Element(MAX_CHOICE_NB);
        maxChoiceNbElm.setText(Integer.toString(poll.getMaxChoiceNb()));
        pollElm.addContent(maxChoiceNbElm);

        // Création des commentaires
        Element commentsElm = new Element(COMMENTS);

        for (CommentDTO commentDTO : poll.getComments()) {
            Element commentElm = toDOM(commentDTO);
            commentsElm.addContent(commentElm);
        }

        pollElm.addContent(commentsElm);

        // Création des groupes de votants
        Element groupsElm = new Element(GROUPS);

        for (VotingGroupDTO votingGroupDTO : poll.getVotingGroups()) {
            Element groupElm = toDOM(votingGroupDTO);
            groupsElm.addContent(groupElm);
        }

        pollElm.addContent(groupsElm);

        // Création des choix
        Element pollChoicesElm = new Element(POLL_CHOICES);

        for (PollChoiceDTO pollChoiceDTO : poll.getChoices()) {
            Element pollChoiceElm = toDOM(pollChoiceDTO);
            pollChoicesElm.addContent(pollChoiceElm);
        }
        pollElm.addContent(pollChoicesElm);

        // Création des résultats
        Element voteCountingResultsElm = new Element(VOTE_COUNTING_RESULTS);

        for (VoteCountingResultDTO voteCountingResult1 : voteCountingResults) {
            Element voteCountingResult = toDOM(voteCountingResult1);
            voteCountingResultsElm.addContent(voteCountingResult);
        }

        // Ajout du sondage et des résultats à la racine
        racine.addContent(pollElm);

        racine.addContent(voteCountingResultsElm);

        return document;
    }

    protected Element toDOM(CommentDTO comment) {
        Element commentElm = new Element(COMMENT);

        commentElm.setAttribute(VOTING_ID, comment.getVotingID());
        commentElm.setText(comment.getText());

        return commentElm;
    }

    protected Element toDOM(VotingGroupDTO group) {
        Element groupElm = new Element(GROUP);

        groupElm.setAttribute(ID_GROUP, group.getIdGroup());
        groupElm.setAttribute(NAME, group.getName());
        groupElm.setAttribute(WEIGHT, Double.toString(group.getWeight()));

        Element votingPersonsElm = new Element(VOTING_PERSONS);

        for (VotingPersonDTO votingPersonDTO : group.getVotingPersons()) {
            Element votingPersonElm = toDOM(votingPersonDTO);
            votingPersonsElm.addContent(votingPersonElm);
        }
        groupElm.addContent(votingPersonsElm);

        return groupElm;
    }

    protected Element toDOM(VotingPersonDTO votingPerson) {
        Element votingPersonElm = new Element(VOTING_PERSON);

        votingPersonElm.setAttribute(VOTING_ID, votingPerson.getVotingId());
        votingPersonElm.setAttribute(WEIGHT, Double.toString(votingPerson
                                                                     .getWeight()));

        Element emailElm = new Element(EMAIL);
        emailElm.setText(votingPerson.getEmail());
        votingPersonElm.addContent(emailElm);

        Element commentElm = new Element(COMMENT);
        commentElm.setText(votingPerson.getComment());
        votingPersonElm.addContent(commentElm);

        Element choicesElm = new Element(CHOICES);

        for (VoteToChoiceDTO voteToChoice : votingPerson.getChoices()) {
            Element choiceElm = new Element(CHOICE);
            choiceElm.setAttribute(ID_CHOICE, voteToChoice.getIdChoice());
            choiceElm.setAttribute(VALUE, Double.toString(voteToChoice
                                                                  .getValue()));
            choicesElm.addContent(choiceElm);
        }

        votingPersonElm.addContent(choicesElm);

        return votingPersonElm;
    }

    protected Element toDOM(PollChoiceDTO pollChoice) {
        Element pollChoiceElm = new Element(POLL_CHOICE);

        pollChoiceElm.setAttribute(ID_CHOICE, pollChoice.getIdChoice());

        Element nameElm = new Element(NAME);
        nameElm.setText(pollChoice.getName());
        pollChoiceElm.addContent(nameElm);

        Element descriptionElm = new Element(DESCRIPTION);
        descriptionElm.setText(pollChoice.getDescription());
        pollChoiceElm.addContent(descriptionElm);

        return pollChoiceElm;
    }

    protected Element toDOM(VoteCountingResultDTO voteCountingResults) {
        Element voteCountingResultElm = new Element(VOTE_COUNTING_RESULT);

        voteCountingResultElm.setAttribute(ID_POLL, voteCountingResults
                .getIdPoll());

        Element nbVotesElm = new Element(NB_VOTES);
        nbVotesElm.setText(Integer.toString(voteCountingResults.getNbVotes()));
        voteCountingResultElm.addContent(nbVotesElm);

        Element choiceResultElm = new Element(CHOICE_RESULT);
        choiceResultElm.setText(voteCountingResults.getChoiceResult());
        voteCountingResultElm.addContent(choiceResultElm);

        Element isByGroupElm = new Element(IS_BY_GROUP);
        isByGroupElm.setText(Boolean.toString(voteCountingResults.isByGroup()));
        voteCountingResultElm.addContent(isByGroupElm);

        Element choicesElm = new Element(CHOICES);

        for (ChoiceDTO choiceDTO : voteCountingResults.getChoices()) {
            Element choiceElm = toDOM(choiceDTO);
            choicesElm.addContent(choiceElm);
        }

        voteCountingResultElm.addContent(choicesElm);

        return voteCountingResultElm;
    }

    protected Element toDOM(ChoiceDTO choice) {
        Element choiceElm = new Element(CHOICE);

        choiceElm.setAttribute(ID_CHOICE, choice.getIdChoice());
        choiceElm.setAttribute(VALUE, Double.toString(choice.getValue()));

        Element percentageElm = new Element(PERCENTAGE);
        percentageElm.setText(Double.toString(choice.getPercentage()));
        choiceElm.addContent(percentageElm);

        Element nbVotesElm = new Element(NB_VOTES);
        nbVotesElm.setText(Integer.toString(choice.getNbVotes()));
        choiceElm.addContent(nbVotesElm);

        return choiceElm;
    }

}
