/*
 * #%L
 * Pollen :: VoteCounting strategy :: Condorcet
 * $Id: CondorcetStrategy.java 3593 2012-08-12 11:01:15Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5.1/pollen-votecounting-strategy-condorcet/src/main/java/org/chorem/pollen/votecounting/strategy/CondorcetStrategy.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting.strategy;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.votecounting.model.ChoiceIdAble;
import org.chorem.pollen.votecounting.model.ChoiceScore;
import org.chorem.pollen.votecounting.model.ChoiceToVoteRenderType;
import org.chorem.pollen.votecounting.model.VoteCountingResult;
import org.chorem.pollen.votecounting.model.VoteForChoice;
import org.chorem.pollen.votecounting.model.Voter;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n.l_;
import static org.nuiton.i18n.I18n.n_;

/**
 * Condorcet.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5
 */
public class CondorcetStrategy extends AbstractVoteCountingStrategy {

    public static final int ID = 2;

    /** Logger. */
    private static final Log log = LogFactory.getLog(CondorcetStrategy.class);

    @Override
    public int getId() {
        return ID;
    }

    @Override
    public String getI18nName() {
        return n_("pollen.voteCountingType.condorcet");
    }

    @Override
    public String getI18nHelp() {
        return n_("pollen.voteCountingType.condorcet.help");
    }

    @Override
    public String getTotalVoteValueNotValidMessage(Locale locale) {
        // no validation on total value, so no message
        return null;
    }

    @Override
    public String getVoteValueNotValidMessage(Locale locale) {
        return l_(locale, "pollen.error.vote.invalidCondorcetVoteValue");
    }

    @Override
    public String getDisplayVoteValue(Integer voteValue) {
        return voteValue == null ? "" : String.valueOf(voteValue);
    }

    @Override
    public ChoiceToVoteRenderType getVoteValueEditorType() {
        return ChoiceToVoteRenderType.TEXTFIELD;
    }

    @Override
    public boolean isChoiceInVote(Integer voteValue) {
        return voteValue != null && voteValue > 0 && voteValue < 100;
    }

    @Override
    public boolean isVoteValueNull(Integer voteValue) {
        return voteValue == null;
    }

    @Override
    public boolean isDisplayResultsByChoice() {
        return false;
    }

    @Override
    public boolean isVoteValueValid(Integer voteValue) {
        return voteValue != null && voteValue > 0;
    }

    @Override
    public boolean isTotalVoteValueValid(int totalValues) {
        // no validation on total value
        return true;
    }

    @Override
    public VoteCountingResult votecount(Set<Voter> voters) {

        // get empty result by choice
        Map<String, ChoiceScore> resultByChoice = votersToResult(voters);

        // get order over choices (needed to get coordinates over matrix)
        List<String> choiceIds = Lists.newArrayList(resultByChoice.keySet());

        // nb of choices
        int nbChoices = choiceIds.size();

        //matrix of pairwise
        double[][] matrix = new double[nbChoices][nbChoices];

        // compute pairwise battle matrix and try to have a direct winner(s)
        Set<String> winners = computePairWiseMatrix(choiceIds, voters, matrix);

        // compute nb battles wins for each choice
        // and store it as choice score value
        computeNbBattlesByChoice(resultByChoice, choiceIds, matrix);

        // get choice score ordered by their number of win battles
        List<ChoiceScore> choiceScores = toChoiceScore(resultByChoice);

        if (winners != null) {

            // there is winner(s), re-add them to

            if (log.isDebugEnabled()) {
                log.debug("Direct winners : " + winners);
            }
            for (String choiceId : winners) {
                ChoiceScore choiceScore = resultByChoice.get(choiceId);
                choiceScores.remove(choiceScore);
                choiceScores.add(0, choiceScore);
            }
        } else {

            // no direct winner, must resolv conflicts

            resolvConflicts(choiceScores, matrix);
        }

        // transform map of result to list of them (and sort them)
        VoteCountingResult result = resultToList(resultByChoice);
        return result;
    }

    protected void resolvConflicts(List<ChoiceScore> choiceScores,
                                   double[][] matrix) {

        // for the moment we use only number of win battles
        // so nothing more to do here...
    }

    protected void computeNbBattlesByChoice(Map<String, ChoiceScore> resultByChoice,
                                            List<String> choiceIds,
                                            double[][] matrix) {

        int nbChoices = choiceIds.size();

        for (String choiceId : choiceIds) {

            int i = choiceIds.indexOf(choiceId);

            double nbBattles = 0;
            for (int j = 0; j < nbChoices; j++) {
                double aRow = matrix[i][j];
                nbBattles += aRow;
            }

            resultByChoice.get(choiceId).setScoreValue(
                    BigDecimal.valueOf(nbBattles));

            if (log.isDebugEnabled()) {
                log.debug("Nb battle wins for choice " + choiceId + ": " + nbBattles);
            }
        }
    }

    private Set<String> computePairWiseMatrix(List<String> choiceIds,
                                              Set<Voter> voters,
                                              double[][] matrix) {

        int nbChoices = choiceIds.size();

        Set<String> winners = null;

        VoteForChoiceComparator comparator = new VoteForChoiceComparator();

        double[] currentVoteWinner = new double[choiceIds.size()];

        boolean firstVoter = true;
        for (Voter voter : voters) {

            if (log.isDebugEnabled()) {
                log.debug("Start count for voter " + voter.getVoterId());
            }
            double voterWeight = voter.getWeight();

            Arrays.fill(currentVoteWinner, 0);

            Map<String, VoteForChoice> voteByChoiceIds =
                    Maps.uniqueIndex(voter.getVoteForChoices(),
                                     new ChoiceIdAble.ChoiceIdAbleById());

            double maxBattleWins = 0;

            for (String choiceId : choiceIds) {

                // get vote for this choice
                VoteForChoice voteForChoice = voteByChoiceIds.get(choiceId);

                int x = choiceIds.indexOf(choiceId);

                for (VoteForChoice voteForChoice1 : voter.getVoteForChoices()) {

                    String choiceId1 = voteForChoice1.getChoiceId();
                    if (choiceId.equals(choiceId1)) {

                        // can not battle same choice
                        continue;
                    }

                    int y = choiceIds.indexOf(choiceId1);

                    int compare = comparator.compare(voteForChoice,
                                                     voteForChoice1);

                    if (compare < 0) {

                        // voteForChoice wins his battle

                        int pos = x * nbChoices + y;

                        matrix[x][y] = matrix[x][y] + voterWeight;
                        if (log.isTraceEnabled()) {
                            log.trace("battle [" + choiceId + "<" +
                                      voteForChoice.getVoteValue() + "> - " +
                                      choiceId1 + "<" +
                                      voteForChoice1.getVoteValue() +
                                      ">] wins (store to " + pos +
                                      ", new value=" + matrix[x][y] + ")");
                        }
                        maxBattleWins = Math.max(
                                maxBattleWins,
                                currentVoteWinner[x] =
                                        currentVoteWinner[x] + voterWeight);
                    }
                }
            }

            if (firstVoter || winners != null) {

                // still have a winner, le'ts compute current winner
                Set<String> currentVoteWinners = Sets.newHashSet();

                // find winners of this vote
                for (int i = 0; i < currentVoteWinner.length; i++) {
                    if (maxBattleWins == currentVoteWinner[i]) {

                        // this is a winner for this vote
                        String choiceId = choiceIds.get(i);
                        currentVoteWinners.add(choiceId);
                    }
                }
                if (log.isDebugEnabled()) {
                    log.debug("Winners of this vote : " + currentVoteWinners);
                }


                if (firstVoter) {

                    // keep this first result
                    winners = currentVoteWinners;
                } else if (!winners.equals(currentVoteWinners)) {

                    // current vote has not same winners than previous,
                    // so will need resolution
                    winners = null;

                    if (log.isDebugEnabled()) {
                        log.debug("Dismatch winners (will need resolv...)");
                    }
                }
            }

            firstVoter = false;
        }
        return winners;
    }

}
