/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: ResultForPoll.java 3719 2012-10-01 11:32:38Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.1/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/ResultForPoll.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.opensymphony.xwork2.inject.Inject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.views.util.UrlHelper;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.ChoiceType;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.services.impl.PollVoteCountingService;
import org.chorem.pollen.ui.PollenUIUtils;
import org.chorem.pollen.ui.actions.PageSkin;
import org.chorem.pollen.ui.actions.PollUriAware;
import org.chorem.pollen.ui.actions.PollenActionSupport;
import org.chorem.pollen.ui.actions.PollenUserSecurityAware;
import org.chorem.pollen.ui.converters.DateConverter;
import org.chorem.pollen.votecounting.VoteCounting;
import org.chorem.pollen.votecounting.model.ChoiceScore;
import org.chorem.pollen.votecounting.model.GroupVoteCountingResult;
import org.chorem.pollen.votecounting.model.VoteCountingResult;
import org.nuiton.topia.persistence.TopiaId;

import java.net.URL;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Display results of a poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class ResultForPoll extends PollenActionSupport implements PollenUserSecurityAware, PollUriAware {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(ResultForPoll.class);

    /**
     * Is feed exists for this poll ?
     *
     * @since 1.4
     */
    private boolean feedFileExisting;

    /**
     * Loaded poll.
     *
     * @since 1.3
     */
    private Poll poll;

    /** Poll main votecounting result. */
    private VoteCountingResult pollResult;

    /** Group result (if poll is group and {]code byGroup} flag is on). */
    private GroupVoteCountingResult groupResult;

    /** choices of the poll indexed by ther id. */
    private Map<String, Choice> choicesById;

    /** group of the poll indexed by ther id. */
    private Map<String, VotingList> groupsById;

    /**
     * Url helper (to build new url)
     *
     * @since 1.3.1
     */
    @Inject(required = true)
    private transient UrlHelper urlHelper;

    /** Flag to votecount by group (only used if a poll is group). */
    private Boolean byGroup;

    public ResultForPoll() {
        super(PageSkin.RESULT);
    }

    public Poll getPoll() {
        return poll;
    }

    public boolean isCreatorOrAdmin() {
        return getUserSecurityContext().isCreator() || isUserAdmin();
    }

    public boolean isFeedFileExisting() {
        return feedFileExisting;
    }

    public Boolean getByGroup() {
        return byGroup;
    }

    public void setByGroup(Boolean byGroup) {
        this.byGroup = byGroup;
    }

    public String getVoteCountingTypeName() {
        VoteCounting voteCounting = getVoteCounting(getPoll());
        String result = voteCounting.getName(getLocale());
        return result;
    }

    public String getVoteCountingTypeHelp() {
        VoteCounting voteCounting = getVoteCounting(getPoll());
        String result = voteCounting.getShortHelp(getLocale());
        return result;
    }

    public String getSummaryUrl() {
        PollUrl url = getPollUrlService().getPollSummaryUrl(poll);
        url.getPollUri().setAccountId(getUserSecurityContext().getAccountId());
        return url.getUrl();
    }

    public String getResultUrl(boolean byGroup) {
        PollUrl url = getPollUrlService().getPollResultUrl(poll);
        url.getPollUri().setAccountId(getUserSecurityContext().getAccountId());
        String result = url.getUrl();
        if (!byGroup) {
            result += "?byGroup=false";
        }
        return result;
    }

    public String getVoteUrl() {
        PollUrl url = getPollUrlService().getPollVoteUrl(poll);
        url.getPollUri().setAccountId(getUserSecurityContext().getAccountId());
        if (poll.isPollFree()) {

            // can removed accountId only for free poll
            //FIXME Should found out in ohter case the accountId (if exists for the connected id) if no accountId is given
            getUserSecurityContext().removeAccountIdWhenConnected(url);
        }
        return url.getUrl();
    }

    public String getPollCreatorName() {
        PollAccount creator = poll.getCreator();
        String result = creator.getVotingId();
        if (StringUtils.isBlank(result)) {
            result = creator.getEmail();
        }
        if (StringUtils.isBlank(result)) {
            result = _("pollen.common.undefined");
        }
        return result;
    }

    public String getPollBeginDate() {
        Date date = poll.getBeginDate();
        String result = date == null ? _("pollen.common.undefined") :
                        getPollService().decorateDate(date);
        return result;
    }

    public String getPollEndDate() {
        Date date = poll.getEndDate();
        String result = date == null ? _("pollen.common.undefined") :
                        getPollService().decorateDate(date);
        return result;
    }

    public Set<String> getGroupIds() {
        return groupResult.getGroupIds();
    }

    public GroupVoteCountingResult getGroupResult() {
        return groupResult;
    }

    public VoteCountingResult getPollResult() {
        return pollResult;
    }

    public Date getResultAsDate(String choiceName) {
        Date date = new Date(Long.valueOf(choiceName));
        return date;
    }

    public String getVoteSizeMessage() {
        String result = PollenUIUtils.getVoteSizeMessage(getPoll(),
                                                         getLocale());
        return result;
    }

    public String getVictoryMessage() {

        Collection<ChoiceScore> topRanking = pollResult.getTopRanking();

        String victoryMessage;
        if (CollectionUtils.isEmpty(topRanking)) {
            victoryMessage = "";
        } else if (topRanking.size() == 1) {
            victoryMessage = _("pollen.common.victory");
        } else {
            victoryMessage = _("pollen.common.victories");
        }
        return victoryMessage;
    }

    @Override
    public String execute() throws Exception {
        poll = getUserSecurityContext().getPoll();

        if (byGroup == null) {

            // nothing defined, use default value
            byGroup = poll.isPollGroup();
        }

        choicesById = Maps.uniqueIndex(poll.getChoice(), TopiaId.GET_TOPIA_ID);

        feedFileExisting =
                getSecurityService().isCanShowFeed(getUserSecurityContext());

        PollVoteCountingService voteCountingService =
                getService(PollVoteCountingService.class);

        if (poll.isPollGroup() && getByGroup()) {

            groupsById = Maps.newHashMap();

            for (VotingList votingList : poll.getVotingList()) {
                groupsById.put(votingList.getTopiaId(), votingList);
            }

            // load group result
            groupResult = voteCountingService.getGroupResult(poll);

            pollResult = groupResult.getMainResult();
        } else {

            // load simple result
            pollResult = voteCountingService.getSimpleResult(poll);
        }

        if (poll.isRunning(serviceContext.getCurrentTime())) {
            addFlashWarning(_("pollen.information.pollRunning"));
        }
        if (log.isInfoEnabled()) {
            log.info("topRanking = " + pollResult.getTopRanking());
        }
        return SUCCESS;
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    public String getNormalPollResultChartUrl(int width, int height) {
        String title;
        if (poll.isPollGroup() && !getByGroup()) {

            // means poll votecount as anot group poll
            title = _("pollen.common.pollResult.forPollAsNotGroupPoll");
        } else {
            title = _("pollen.common.pollResult.forPoll");

        }
        return getPollResultChartUrl(pollResult, title, width, height);
    }

    public String getGroupPollResultChartUrl(String groupId, int width, int height) {
        VoteCountingResult result = groupResult.getGroupResult(groupId);
        String groupName = getGroupName(groupId);
        String title = _("pollen.common.pollResult.forGroup", groupName);
        return getPollResultChartUrl(result, title, width, height);
    }

    public String getChoiceName(String id) {
        String result = choicesById.get(id).getName();
        return result;
    }

    public String getGroupName(String id) {
        String result = groupsById.get(id).getName();
        return result;
    }

    protected String getPollResultChartUrl(VoteCountingResult results,
                                           String title,
                                           int width, int height) {

        List<String> choiceValues = Lists.newArrayList();

        for (ChoiceScore result : results.getScores()) {

            String name = getChoiceName(result.getChoiceId());
            String value = String.valueOf(result.getScoreValue());

            if (isDateType()) {
                // mise en forme de la date
                Date date = new Date(Long.parseLong(name));
                name = DateConverter.convertToString(date);
            }

            choiceValues.add(name);
            choiceValues.add(value);
        }

        URL applicationUrl = getApplicationUrl();
        StringBuilder url = new StringBuilder(applicationUrl.toString());

        url.append("/io/generateChart");

        Map<String, Object> params = Maps.newHashMap();
        params.put("width", width);
        params.put("height", height);
        params.put("title", title);
        params.put("values", choiceValues);

        urlHelper.buildParametersString(params, url, "&");

        String chartUrl = url.toString();
        return chartUrl;
    }

}
