/*
 * #%L
 * Lima :: business
 * %%
 * Copyright (C) 2008 - 2012 CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.web.service;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.LimaTechnicalException;
import org.chorem.lima.beans.DocumentReport;
import org.chorem.lima.beans.FinancialStatementAmounts;
import org.chorem.lima.beans.ReportsDatas;
import org.chorem.lima.business.LimaServiceConfig;
import org.chorem.lima.business.api.AccountService;
import org.chorem.lima.business.api.ClosedPeriodicEntryBookService;
import org.chorem.lima.business.api.EntryService;
import org.chorem.lima.business.api.FinancialPeriodService;
import org.chorem.lima.business.api.FinancialStatementService;
import org.chorem.lima.business.api.IdentityService;
import org.chorem.lima.business.api.ReportService;
import org.chorem.lima.business.api.VatStatementService;
import org.chorem.lima.business.api.report.BalanceReportService;
import org.chorem.lima.business.api.report.GeneralEntryBookReportService;
import org.chorem.lima.business.api.report.LedgerReportService;
import org.chorem.lima.business.api.report.ProvisionalEntryBookReportService;
import org.chorem.lima.business.utils.BigDecimalToString;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.Identity;
import org.chorem.lima.service.LimaServiceFactory;
import org.chorem.lima.web.action.ReportBuilder;
import org.chorem.lima.web.utils.DocumentsEnum;

import javax.swing.*;
import java.io.File;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import static org.nuiton.i18n.I18n.t;


public class DocumentService {

    protected static final Log log = LogFactory.getLog(DocumentService.class);

    private IdentityService identityService;

    private FinancialStatementService financialStatementService;

    private ReportService reportService;

    private VatStatementService vatStatementService;

    protected BalanceReportService balanceReportService;

    protected GeneralEntryBookReportService generalEntryBookReportService;

    protected ProvisionalEntryBookReportService entryBookReportService;

    protected LedgerReportService ledgerReportService;

    protected ClosedPeriodicEntryBookService closedPeriodicEntryBookService;

    protected FinancialPeriodService financialPeriodService;

    protected EntryService entryService;

    protected ReportBuilder reportBuilder;

    protected AccountService accountService;

    protected String path = LimaServiceConfig.getInstance().getReportsModelDir().getAbsolutePath();

    protected final String BALANCE_FILE_PATH = path + File.separator + DocumentsEnum.BALANCE.getFileName() + ".pdf";
    protected final String GENERAL_ENTRY_BOOK_REPORT_PDF_FILE_PATH = path + File.separator + DocumentsEnum.GENERAL_ENTRY_BOOK.getFileName() + ".pdf";
    protected final String ENTRY_BOOKS_REPORT_PDF_FILE_PATH = path + File.separator + DocumentsEnum.ENTRY_BOOKS.getFileName() + ".pdf";
    protected final String LEDGER_REPORT_PDF_FILE_PATH = path + File.separator + DocumentsEnum.LEDGER.getFileName() + ".pdf";

    public DocumentService() {
        identityService = LimaServiceFactory.getService(IdentityService.class);
        financialStatementService = LimaServiceFactory.getService(FinancialStatementService.class);
        reportService = LimaServiceFactory.getService(ReportService.class);
        vatStatementService = LimaServiceFactory.getService(VatStatementService.class);
        closedPeriodicEntryBookService = LimaServiceFactory.getService(ClosedPeriodicEntryBookService.class);
        financialPeriodService = LimaServiceFactory.getService(FinancialPeriodService.class);
        entryService = LimaServiceFactory.getService(EntryService.class);
        accountService = LimaServiceFactory.getService(AccountService.class);

        balanceReportService = LimaServiceFactory.getService(BalanceReportService.class);
        generalEntryBookReportService = LimaServiceFactory.getService(GeneralEntryBookReportService.class);
        entryBookReportService = LimaServiceFactory.getService(ProvisionalEntryBookReportService.class);
        ledgerReportService = LimaServiceFactory.getService(LedgerReportService.class);

        reportBuilder = new ReportBuilder();

        if (log.isDebugEnabled()) {
            log.debug("Path : " + path);
        }

    }

    /**
     * General Leger
     * @param beginDate
     * @param endDate
     * @return
     */
    public String createFinancialStatementsDocuments(Date beginDate,
                                                     Date endDate) {

        List<FinancialStatementAmounts> financialStatementAmounts =
                financialStatementService.financialStatementReport(beginDate, endDate);
        String financialReport = "";

        try {

            financialReport = constructHtmlHeader("");

            //Split list by financialstatement type
            List<List<FinancialStatementAmounts>> listList =
                    new ArrayList<List<FinancialStatementAmounts>>();
            Boolean first = true;
            int min = 0;
            int size = financialStatementAmounts.size();
            for (int i = 0; i < size; i++) {
                FinancialStatementAmounts fStatementAmounts =
                        financialStatementAmounts.get(i);
                if (fStatementAmounts.getLevel() == 1 && !fStatementAmounts.isSubAmount()) {
                    if (first) {
                        first = false;
                    } else {
                        listList.add(financialStatementAmounts.subList(min, i - 1));
                    }
                    min = i;
                }
            }
            listList.add(financialStatementAmounts.subList(min, size));
            int printedType = -1;

            //create pages
            for (List<FinancialStatementAmounts> list : listList) {
                if (!list.isEmpty()){
                    String title = list.get(0).getLabel();
                    int i = 0;
                    int n = list.size();
                    printedType ++;
                    while (i < n) {

                        financialReport += constructHeaderTitle(title, beginDate, endDate);

                        String boldBegin = "<b>";
                        String boldEnd = "</b>";
                        financialReport += "<table border=\"1\" width=\"100%\" cellpadding=\"3\" cellspacing=\"0\">\n" +
                                "<tr align=\"center\">\n";

                        if (printedType == 0) {
                            String[] columnHeaderTable = {boldBegin+t("lima-business.document.label")+boldEnd, boldBegin+t("lima-business.document.grossamount")+boldEnd,
                                    boldBegin+t("lima-business.document.provisiondeprecationamount")+boldEnd, boldBegin+t("lima-business.document.netamount")+boldEnd};
                            financialReport += constructTableLine(columnHeaderTable);
                        } else {
                            String[] columnHeaderTable = {boldBegin+t("lima-business.document.label")+boldEnd, boldBegin+t("lima-business.document.amount")+boldEnd};
                            financialReport += constructTableLine(columnHeaderTable);
                        }

                        for (FinancialStatementAmounts financialStatementAmount : list) {

                            String label = financialStatementAmount.getLabel();
                            int level = financialStatementAmount.getLevel();
                            BigDecimal grossAmount =
                                    financialStatementAmount.getGrossAmount();
                            if (grossAmount == null) {
                                grossAmount = BigDecimal.ZERO;
                            }
                            BigDecimal provisionDeprecationAmount =
                                    financialStatementAmount.getProvisionDeprecationAmount();
                            if (provisionDeprecationAmount == null) {
                                provisionDeprecationAmount =BigDecimal.ZERO;
                            }

                            if (label == null) {
                                if (printedType == 0) {
                                    String [] emptyColumn = {"","","",""};
                                    financialReport += constructTableLine(emptyColumn);
                                } else {
                                    String [] emptyColumn = {"", ""};
                                    financialReport += constructTableLine(emptyColumn);
                                }
                            } else {
                                //cell1
                                StringBuilder tab = new StringBuilder();
                                for (int k = 0; k < level; k++) {
                                    tab.append("\t");
                                }
                                //Phrase phrase;
                                String tabLabel = "";
                                if (financialStatementAmount.isHeader()) {
                                    tabLabel = boldBegin + tab + label + boldEnd;
                                } else {
                                    tabLabel = tab + label;
                                }
                                //cell2
                                String grossAmountStr = "";
                                if (!grossAmount.equals(BigDecimal.ZERO)) {
                                    grossAmountStr = grossAmount.toString();
                                }

                                //cell 3
                                String provisionDeprecationAmountStr = "";
                                if (!provisionDeprecationAmount.equals(BigDecimal.ZERO)) {
                                    provisionDeprecationAmountStr = provisionDeprecationAmount.toString();
                                }

                                //cell 4
                                BigDecimal solde = grossAmount;
                                solde = solde.subtract(provisionDeprecationAmount);
                                String soldeStr = "";
                                if (!solde.equals(BigDecimal.ZERO)) {
                                    soldeStr = solde.toString();
                                }

                                if (printedType == 0) {
                                    String [] columns = {tabLabel, grossAmountStr, provisionDeprecationAmountStr, soldeStr};
                                    financialReport += constructTableLine(columns);
                                } else /*if (printedType == 1)*/ {
                                    String [] columns = {tabLabel, soldeStr};
                                    financialReport += constructTableLine(columns);
                                }
                            }
                        }
                        i = i + n;
                        financialReport += "</table>";
                    }
                }
            }
            financialReport += "</body>" +
                    "</html>";

        } catch (Exception ex) {
            throw new LimaTechnicalException("Can't create document", ex);
        }
        return financialReport;
    }

    //##############     VAT      ##############
//    public void createVatDocuments(Date beginDate,
//                                   Date endDate,
//                                   String autocomplete) {
//
//        String filePath = path + File.separator
//                + DocumentsEnum.VAT.getFileName() + ".pdf";
//
//        String path = LimaServiceConfig.getInstance().getReportsModelDir().getAbsolutePath();
//
//        String filePathDefault = path + File.separator
//                + DocumentsEnum.VAT.getFileName() + "_default.pdf";
//
//        PDDocument doc;
//        InputStream reportsStream;
//
//        String vatPDFUrl = LimaServiceConfig.getInstance().getVatPDFUrl();
//
//        if (vatPDFUrl.equals("default")) {
//            reportsStream = DocumentService.class
//                    .getResourceAsStream("/reports/vat_form_fr.pdf");
//            if (reportsStream == null) {
//                throw new LimaTechnicalException("Could not find such file "
//                        + "/reports/vat_form_fr.pdf");
//            }
//        } else {
//            try {
//                reportsStream = new FileInputStream(filePathDefault);
//            } catch (FileNotFoundException eee) {
//                throw new LimaTechnicalException("Could not find such file "
//                        + filePathDefault, eee);
//            }
//        }
//
//        try {
//
//            // load the document
//            doc = PDDocument.load(reportsStream);
//
//            if (autocomplete != null) {
//                if (log.isDebugEnabled()) {
//                    log.debug("autocomplete: " + autocomplete);
//                }
//                if (autocomplete.equals("true")) {
//
//                    SetField fields = new SetField();
//
//                    //search for all VAT Statements from the report
//                    List<VatStatement> vatStatementsList = vatStatementService.getAllVatStatements();
//                    if (log.isDebugEnabled()) {
//                        log.debug("vatStatementsList.size() : " + vatStatementsList.size());
//                    }
//                    for (VatStatement vatStatement : vatStatementsList) {
//                        //search for amount to display
//                        BigDecimal amount = vatStatementService.vatStatementAmounts(vatStatement, beginDate, endDate).getAmount();
//                        //display amount only if it is a child and has a BoxName
//                        if (vatStatement.getBoxName() != null && !vatStatement.isHeader()) {
//                            if (log.isDebugEnabled()) {
//                                log.debug("Set field...");
//                            }
//                            fields.setField(doc, vatStatement.getBoxName(), amount.toString());
//                        }
//                    }
//                }
//            }
//
//            // save the updated document to the new file and close
//            doc.save(filePath);
//            doc.close();
//
//        } catch (Exception ex) {
//            throw new LimaTechnicalException("Can't create document", ex);
//        }
//    }


    public String createAccountDocument(Date beginDate, Date endDate, String account) {

        String accountReport = null;

        try {

            Account accountFormat = accountService.findAccountById(account);
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("dd MMMMM yyyy");

            accountReport = constructHtmlHeader(t("lima.reports.accounts"));

            ReportsDatas results;

            if (beginDate != null && endDate != null && accountFormat != null) {

                String subTitleFirstpart =  t("lima.fiscalperiod.fiscalperiod") + " : " + simpleDateFormat.format(beginDate) + " - " + simpleDateFormat.format(endDate);
                String subTitleSecPart = t("lima.financialtransaction.account") + " : " + accountFormat.getAccountNumber() + " - " + accountFormat.getLabel();
                accountReport += constructSubTitleHtml(subTitleFirstpart, subTitleSecPart);

                results = reportService.generateAccountsReports(accountFormat, true,
                        beginDate, endDate);
                List<Entry> entries = results.getListEntry();

                String[] columnNames = {t("lima.table.number"), t("lima.table.date"), t("lima.table.entrybook"),
                        t("lima.table.voucher"), t("lima.table.description"), t("lima.table.letter"),
                        t("lima.table.debit"), t("lima.table.credit")};

                accountReport += "\t<table border=\"1\" width=\"100%\" cellpadding=\"3\" cellspacing=\"0\">\n";

                accountReport += constructTableHeader(columnNames);
                accountReport += "\t\t<tbody>\n";

                boolean even = true;
                for(Entry entry : entries) {

                    String accountNumber = entry.getAccount().getAccountNumber();
                    String transactionDate = simpleDateFormat.format(entry.getFinancialTransaction().getTransactionDate());
                    String code = "";
                    if (entry.getFinancialTransaction().getEntryBook() != null) {
                        code = entry.getFinancialTransaction().getEntryBook().getCode();
                    }
                    String voucher = entry.getVoucher();
                    String description = entry.getDescription();
                    String lettering = entry.getLettering();


                    String[] columnData = {(StringUtils.isBlank(accountNumber)?"":accountNumber), (StringUtils.isBlank(transactionDate)?"":transactionDate),
                            (StringUtils.isBlank(code)?"":code), (StringUtils.isBlank(voucher)?"":voucher),
                            (StringUtils.isBlank(description)?"":description), (StringUtils.isBlank(lettering)?"":lettering),
                            (entry.isDebit() ? entry.getAmount() : BigDecimal.ZERO).toString(),
                            (entry.isDebit() ? BigDecimal.ZERO : entry.getAmount()).toString()};

                    accountReport += constructTableLine(columnData, even);
                    even = !even;
                }

                accountReport += "\t\t</tbody>\n\t</table>\n" +
                        "</body>\n";

            } else {
                JOptionPane.showMessageDialog(null, t("lima.reports.account.noaccount"), t("lima.reports.account.noaccounttitle"), JOptionPane.INFORMATION_MESSAGE);
            }

            accountReport += "</html>";

        }
        catch (Exception e) {
            throw new LimaTechnicalException("Can't create document", e);
        }

        return accountReport;
    }

    protected String constructHtmlHeader(String title) {
        String head = "<!DOCTYPE html>\n" +
                "<html>\n" +
                "<head>\n" +
                "\t<meta charset=\"UTF-8\" />\n" +
                "\t<title>" + title + "</title>\n" +
                "</head>\n" +
                "<body>\n" +
                "\t<h1>" + title + "</h1>\n";
        return head;
    }

    protected String constructSubTitleHtml(String subTitle) {
        return "\t<h2>" + subTitle + "</h2>\n";
    }

    protected String constructSubTitleHtml(String subTitleFirstpart, String subTitleSecPart) {
        String subTitle = "\t<h2>\n" +
                subTitleFirstpart +
                "<br/>\n" +
                subTitleSecPart +
                "</h2>\n";

        return subTitle;
    }

    protected String constructTableHeader(String[] columnsNames) {
        String header = "\t\t<thead>\n\t\t\t<tr>\n";
        for (String name : columnsNames) {
            header += "\t\t\t\t<th>" +  name + "</th>\n";
        }
        header += "\t\t\t</tr>\t\t\n</thead>\n";
        return header;
    }

    protected String constructTableLine(String[] cells, boolean even) {
        String style = even ? "line_even" : "line_odd";
        String line = "\t\t<tr class=\"" + style + "\">\n";
        for (String cell : cells) {
            line += "\t\t\t<td>" + cell + "</td>\n";
        }
        line += "\t\t</tr>\n";
        return line;
    }

    protected String constructTableLine(String[] cells) {
        String line = "\t\t<tr>\n";
        for (String cell : cells) {
            line += "\t\t\t<td>" + cell + "</td>\n";
        }
        line += "\t\t</tr>\n";
        return line;
    }

    protected String constructHeaderTitle(String title, Date beginDate, Date endDate) {
        String headerTitle;
        Identity identity = identityService.getIdentity();

        headerTitle = "<table>" +
                "<thead> " +
                "<tr><th>" + title +
                "</th></tr>" +
                "</thead>" +
                "<tr> " +
                "<td>" +
                "<table align=\"left\" border=\"1\" cellpadding=\"3\" cellspacing=\"0\" style=\"font-size:13px;\" >\n" +
                "<tr>\n";

        String boldItalicBegin = "<b>" + "<i>";
        String boldItalicEnd = "</i>" + "</b>";

        if (identity != null) {
            String [] columnsNameSociety = {boldItalicBegin + t("lima-business.document.society") + boldItalicEnd,
                    "<i>" + (StringUtils.isNotEmpty(identity.getName()) ? identity.getName() : " - ") + "</i>", };
            headerTitle += constructTableLine(columnsNameSociety);

            String [] columnsDescription = {boldItalicBegin + t("lima-business.document.description") + boldItalicEnd,
                    "<i>" + (StringUtils.isNotEmpty(identity.getDescription()) ? identity.getDescription() : " - ") + "</i>", };
            headerTitle += constructTableLine(columnsDescription);

            String [] columnsAdressOne = {boldItalicBegin + t("lima-business.document.adress") + boldItalicEnd,
                    "<i>" + (StringUtils.isNotEmpty(identity.getAddress()) ? identity.getAddress() : " - ") + "</i>", };
            headerTitle += constructTableLine(columnsAdressOne);

            String [] columnsAdressTwo = {boldItalicBegin + t("lima-business.document.adresssuite") + boldItalicEnd,
                    "<i>" + (StringUtils.isNotEmpty(identity.getAddress2()) ? identity.getAddress2() : " - ") + "</i>", };
            headerTitle += constructTableLine(columnsAdressTwo);

            String [] columnsZipCode = {boldItalicBegin + t("lima-business.document.zipcode") + boldItalicEnd,
                    "<i>" + (StringUtils.isNotEmpty(identity.getZipCode()) ? identity.getZipCode() : " - ") + "</i>", };
            headerTitle += constructTableLine(columnsZipCode);

            String [] columnsCity = {boldItalicBegin + t("lima-business.document.city") + boldItalicEnd,
                    "<i>" + (StringUtils.isNotEmpty(identity.getCity()) ? identity.getCity() : " - ") + "</i>", };
            headerTitle += constructTableLine(columnsCity);

            String [] columnsBusinessNumber = {boldItalicBegin + t("lima-business.document.businessnumber") +
                    boldItalicEnd, "<i>" + (StringUtils.isNotEmpty(identity.getBusinessNumber()) ? identity.getBusinessNumber() : " - ") + "</i>"};
            headerTitle += constructTableLine(columnsBusinessNumber);

            String [] columnsClassifCode = {boldItalicBegin + t("lima-business.document.classificationcode") +
                    boldItalicEnd, "<i>" + (StringUtils.isNotEmpty(identity.getClassificationCode()) ? identity.getClassificationCode() : " - ") + "</i>"};
            headerTitle += constructTableLine(columnsClassifCode);

            String [] columnsVatNumber = {boldItalicBegin + t("lima-business.document.vatnumber") + boldItalicEnd,
                    "<i>" + (StringUtils.isNotEmpty(identity.getVatNumber()) ? identity.getVatNumber() : " - ") + "</i>", };
            headerTitle += constructTableLine(columnsVatNumber);
        }

        String [] columnsPeriodOne = {boldItalicBegin + t("lima-business.document.period1") + boldItalicEnd, "<i>"
                + t("lima-business.document.period1format", beginDate)+ "</i>"};
        headerTitle += constructTableLine(columnsPeriodOne);

        headerTitle += "<tr>\n";
        String [] columnsPeriodTwo = {boldItalicBegin + t("lima-business.document.period2") + boldItalicEnd, "<i>"
                + t("lima-business.document.period2format", endDate)+ "</i>"};
        headerTitle += constructTableLine(columnsPeriodTwo);

        headerTitle += "</table>" +
                "</td>" +
                "</table>";

        return headerTitle;
    }

    //##############     balance     ##############

    public DocumentReport createBalanceDocuments(Date beginDate, Date endDate, String fromToAccount) {

        DocumentReport report = balanceReportService.getBalanceDocumentReport(beginDate, endDate, fromToAccount, BigDecimalToString.getDecimalFormat());
        reportBuilder.generatePDFReport(org.chorem.lima.business.utils.DocumentsEnum.BALANCE, BALANCE_FILE_PATH, Lists.newArrayList(report));
        return report;
    }

    //##############     General EntryBook      #############
    public DocumentReport createGeneralEntryBooksDocuments(Date beginDate, Date endDate) {

        DocumentReport report = generalEntryBookReportService.getGeneralEntryBookDocumentReport(beginDate, endDate, BigDecimalToString.getDecimalFormat());
        reportBuilder.generatePDFReport(org.chorem.lima.business.utils.DocumentsEnum.GENERAL_ENTRYBOOK, GENERAL_ENTRY_BOOK_REPORT_PDF_FILE_PATH, Lists.newArrayList(report));
        return report;
    }

    public DocumentReport createEntryBooksDocuments(Date beginDate, Date endDate, List<String> entryBookCodes) {
        DocumentReport report = entryBookReportService.getEntryBookDocumentReport(beginDate, endDate, entryBookCodes, BigDecimalToString.getDecimalFormat());
        reportBuilder.generatePDFReport(org.chorem.lima.business.utils.DocumentsEnum.ENTRYBOOKS, ENTRY_BOOKS_REPORT_PDF_FILE_PATH, Lists.newArrayList(report));
        return report;
    }

    public DocumentReport createLedgerDocuments(Date beginDate, Date endDate) {
        DocumentReport report = ledgerReportService.getLedgerDocumentReport(beginDate, endDate, BigDecimalToString.getDecimalFormat());
        reportBuilder.generatePDFReport(org.chorem.lima.business.utils.DocumentsEnum.LEDGER, LEDGER_REPORT_PDF_FILE_PATH, Lists.newArrayList(report));
        return report;
    }
}
