package fr.ifremer.tutti.ui.swing.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: ImportPsionAction.java 1536 2014-01-29 17:40:19Z tchemit $
 * $HeadURL:$
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.CatchBatch;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.service.TuttiEnumerationFile;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.psionimport.PsionImportResult;
import fr.ifremer.tutti.service.psionimport.PsionImportService;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.ImportPupitriPopupUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchUIModel;

import javax.swing.JOptionPane;
import javax.swing.UIManager;
import java.io.File;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * Created on 1/21/14.
 *
 * @author Tony Chemit <chemit@codelutin.com>
 * @since 3.1
 */
public class ImportPsionAction extends AbstractTuttiAction<SpeciesBatchUIModel, SpeciesBatchUI, SpeciesBatchUIHandler> {

    protected File importedTrunkFile;

    protected ImportPupitriPopupUI importPupitriDialog;

    protected PersistenceService persistenceService;

    protected EditFishingOperationAction editAction;

    protected PsionImportResult importResult;

    public ImportPsionAction(SpeciesBatchUIHandler handler) {
        super(handler, false);
        persistenceService = getContext().getPersistenceService();
        importPupitriDialog = new ImportPupitriPopupUI(handler.getContext());
    }

    public EditFishingOperationAction getEditAction() {
        if (editAction == null) {
            FishingOperationsUI parentContainer = handler.getParentContainer(FishingOperationsUI.class);
            editAction = getContext().getActionFactory().createLogicAction(parentContainer.getHandler(),
                                                                           EditFishingOperationAction.class);
        }
        return editAction;
    }

    @Override
    public boolean prepareAction() throws Exception {
        boolean result = true;

        // must check that sample category model is compatible for pupitri import
        // means need some categories

        TuttiEnumerationFile enumerationFile = persistenceService.getEnumerationFile();
        SampleCategoryModel sampleCategoryModel = getDataContext().getSampleCategoryModel();

        List<String> missingCategories = Lists.newArrayList();
        if (!sampleCategoryModel.containsCategoryId(enumerationFile.PMFM_ID_SEX)) {
            missingCategories.add("<li>" + persistenceService.getSexCaracteristic().getParameterName() + "</li>");
        }
        if (!sampleCategoryModel.containsCategoryId(enumerationFile.PMFM_ID_MATURITY)) {
            missingCategories.add("<li>" + persistenceService.getMaturityCaracteristic().getParameterName() + "</li>");
        }
        if (!missingCategories.isEmpty()) {
            result = false;
            JOptionPane.showMessageDialog(
                    getContext().getActionUI(),
                    _("tutti.editSpeciesBatch.action.importPsion.invalidSampleCategoryModel.message", Joiner.on("").join(missingCategories)),
                    _("tutti.editSpeciesBatch.action.importPsion.invalidSampleCategoryModel.title"),
                    JOptionPane.ERROR_MESSAGE,
                    UIManager.getIcon("error")
            );
        }

        if (result) {
            // choose file to import
            importedTrunkFile = chooseFile(
                    _("tutti.editSpeciesBatch.action.title.choose.importPsionFile"),
                    _("tutti.editSpeciesBatch.action.choosePsionFile.import"),
                    "^.*\\.IWA", _("tutti.common.file.iwa"));

            result = importedTrunkFile != null;
        }

        return result;
    }

    @Override
    public void doAction() throws Exception {
        PsionImportService importService = getContext().getTuttiPsionImportExportService();

        EditCatchesUI parentContainer = handler.getParentContainer(EditCatchesUI.class);
        EditCatchesUIModel model = parentContainer.getModel();

        FishingOperation operation = model.getFishingOperation();
        CatchBatch catchBatch = model.toEntity();

        // import
        importResult = importService.importFile(importedTrunkFile,
                                                operation,
                                                catchBatch);

        if (importResult.isDone()) {

            // reload operation
            getEditAction().loadCatchBatch(operation);
        }
    }

    @Override
    public void releaseAction() {
        super.releaseAction();
        importedTrunkFile = null;
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();

        if (importResult.isDone()) {

            sendMessage(_("tutti.editSpeciesBatch.action.importPsion.success",
                          getModel().getRootNumber(), importResult.getNbImported()));
        } else {

            StringBuilder sb = new StringBuilder();
            for (String s : importResult.getErrors()) {
                sb.append("<li>").append(s).append("</li>");
            }
            displayWarningMessage(
                    _("tutti.editSpeciesBatch.action.importPsion.no.matching.fishingOperation.title"),
                    "<html><body>" +
                    _("tutti.editSpeciesBatch.action.importPsion.no.matching.fishingOperation", sb.toString()) +
                    "</body></html>"
            );
            sendMessage(_("tutti.editSpeciesBatch.action.importPsion.no.matching.data"));
        }
    }
}
