package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosBatchUIHandler.java 900 2013-04-30 15:59:25Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/benthos/BenthosBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.BatchContainer;
import fr.ifremer.tutti.persistence.entities.data.BenthosBatch;
import fr.ifremer.tutti.persistence.entities.data.BenthosBatchFrequency;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocols;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.ValidationService;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SampleCategoryColumnIdentifier;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SampleCategoryComponent;
import fr.ifremer.tutti.ui.swing.content.operation.catches.TableViewMode;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.create.CreateBenthosBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.create.CreateBenthosBatchUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.BenthosFrequencyCellComponent;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.BenthosFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.split.SplitBenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.split.SplitBenthosBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.split.SplitBenthosBatchUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellEditor;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellEditor;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.editor.TuttiComputedOrNotDataTableCell;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUI;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUIModel;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;
import org.nuiton.validator.NuitonValidatorResult;

import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.RowFilter;
import javax.swing.UIManager;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.util.Collection;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class BenthosBatchUIHandler extends AbstractTuttiBatchTableUIHandler<BenthosBatchRowModel, BenthosBatchUIModel, BenthosBatchUI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(BenthosBatchUIHandler.class);

    public static final Set<String> SAMPLING_PROPERTIES = Sets.newHashSet(
            BenthosBatchRowModel.PROPERTY_SAMPLE_CATEGORY,
            BenthosBatchRowModel.PROPERTY_SPECIES,
            BenthosBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
            BenthosBatchRowModel.PROPERTY_SIZE_CATEGORY,
            BenthosBatchRowModel.PROPERTY_SEX_CATEGORY,
            BenthosBatchRowModel.PROPERTY_MATURITY_CATEGORY,
            BenthosBatchRowModel.PROPERTY_AGE_CATEGORY,
            BenthosBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT,
            BenthosBatchRowModel.PROPERTY_SIZE_CATEGORY_WEIGHT,
            BenthosBatchRowModel.PROPERTY_SEX_CATEGORY_WEIGHT,
            BenthosBatchRowModel.PROPERTY_MATURITY_CATEGORY_WEIGHT,
            BenthosBatchRowModel.PROPERTY_AGE_CATEGORY_WEIGHT);

    public static final Set<String> SAMPLING_WEIGHT_PROPERTIES = Sets.newHashSet(
            BenthosBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT,
            BenthosBatchRowModel.PROPERTY_SIZE_CATEGORY_WEIGHT,
            BenthosBatchRowModel.PROPERTY_SEX_CATEGORY_WEIGHT,
            BenthosBatchRowModel.PROPERTY_MATURITY_CATEGORY_WEIGHT,
            BenthosBatchRowModel.PROPERTY_AGE_CATEGORY_WEIGHT);

    private final EnumMap<TableViewMode, RowFilter<BenthosBatchTableModel, Integer>> tableFilters;

    protected ValidationService validationService = getContext().getValidationService();

    public BenthosBatchUIHandler(TuttiUI<?, ?> parentUi,
                                 BenthosBatchUI ui) {
        super(parentUi, ui,
              BenthosBatchRowModel.PROPERTY_SPECIES,
              BenthosBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
              BenthosBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT,
              BenthosBatchRowModel.PROPERTY_SIZE_CATEGORY,
              BenthosBatchRowModel.PROPERTY_SIZE_CATEGORY_WEIGHT,
              BenthosBatchRowModel.PROPERTY_SEX_CATEGORY,
              BenthosBatchRowModel.PROPERTY_SEX_CATEGORY_WEIGHT,
              BenthosBatchRowModel.PROPERTY_MATURITY_CATEGORY,
              BenthosBatchRowModel.PROPERTY_MATURITY_CATEGORY_WEIGHT,
              BenthosBatchRowModel.PROPERTY_AGE_CATEGORY,
              BenthosBatchRowModel.PROPERTY_AGE_CATEGORY_WEIGHT,
              BenthosBatchRowModel.PROPERTY_WEIGHT,
              BenthosBatchRowModel.PROPERTY_NUMBER,
              BenthosBatchRowModel.PROPERTY_COMMENT,
              BenthosBatchRowModel.PROPERTY_ATTACHMENT,
              BenthosBatchRowModel.PROPERTY_FREQUENCY,
              BenthosBatchRowModel.PROPERTY_SPECIES_TO_CONFIRM);
        tableFilters = new EnumMap<TableViewMode, RowFilter<BenthosBatchTableModel, Integer>>(TableViewMode.class);

        tableFilters.put(TableViewMode.ALL, new RowFilter<BenthosBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends BenthosBatchTableModel, ? extends Integer> entry) {
                return true;
            }
        });

        tableFilters.put(TableViewMode.ROOT, new RowFilter<BenthosBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends BenthosBatchTableModel, ? extends Integer> entry) {
                boolean result = false;
                Integer rowIndex = entry.getIdentifier();
                if (rowIndex != null) {
                    BenthosBatchTableModel model = entry.getModel();
                    BenthosBatchRowModel row = model.getEntry(rowIndex);
                    result = row != null && row.isBatchRoot();
                }
                return result;
            }
        });

        tableFilters.put(TableViewMode.LEAF, new RowFilter<BenthosBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends BenthosBatchTableModel, ? extends Integer> entry) {
                boolean result = false;
                Integer rowIndex = entry.getIdentifier();
                if (rowIndex != null) {
                    BenthosBatchTableModel model = entry.getModel();
                    BenthosBatchRowModel row = model.getEntry(rowIndex);
                    result = row != null && row.isBatchLeaf();
                }
                return result;
            }
        });
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        BenthosBatchUIModel model = getModel();

        List<BenthosBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {

            if (log.isInfoEnabled()) {
                log.info("Get species batch for fishingOperation: " +
                         bean.getId());
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {

                // get all batch species root (says the one with only a species sample category)
                BatchContainer<BenthosBatch> rootBenthosBatch =
                        persistenceService.getRootBenthosBatch(bean.getId());

                List<BenthosBatch> catches = rootBenthosBatch.getChildren();

                for (BenthosBatch aBatch : catches) {

                    // root batch sample categroy is species
                    Preconditions.checkState(
                            aBatch.getSampleCategoryType() == SampleCategoryEnum.sortedUnsorted,
                            "Root species batch must be a sortedUnsorted sample " +
                            "category but was:" + aBatch.getSampleCategoryType());

                    BenthosBatchRowModel rootRow =
                            loadBatch(aBatch, null, rows);

                    if (log.isDebugEnabled()) {
                        log.debug("Loaded root batch " +
                                  decorate(rootRow.getSpecies(), DecoratorService.FROM_PROTOCOL) + " - " +
                                  decorate(rootRow.getSortedUnsortedCategoryValue()));
                    }
                }
            }
        }

        model.setRows(rows);
        recomputeBatchActionEnable();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public BenthosBatchTableModel getTableModel() {
        return (BenthosBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(BenthosBatchRowModel row) {
        BenthosBatch batch = convertRowToBean(row, true);
        NuitonValidatorResult validator = validationService.validateBenthosBatch(batch);
        boolean result = !validator.hasErrorMessagess();

        if (result
            && ValidationService.VALIDATION_CONTEXT_VALIDATE.equals(
                getContext().getValidationContext())
            && row.isBatchLeaf()) {
            List<BenthosBatchFrequency> frequencies =
                    BenthosFrequencyRowModel.toBeans(row.getFrequency(), batch);
            result = TuttiProtocols.isBenthosBatchValid(getDataContext().getProtocol(),
                                                        batch,
                                                        frequencies);
        }

        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 BenthosBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        recomputeRowValidState(row);

        if (SAMPLING_WEIGHT_PROPERTIES.contains(propertyName)) {

            // sampling category weight has changed, must then save the top
            // ancestor row

            BenthosBatchTableModel tableModel = getTableModel();
            SampleCategoryColumnIdentifier<BenthosBatchRowModel> sampleCategoryColumnIdentifier = tableModel.getCategoryIdentifierForWeightProperty(propertyName);

            SampleCategoryEnum sampleCategoryType = sampleCategoryColumnIdentifier.getSampleCategoryType();
            SampleCategory<?> sampleCategory = row.getSampleCategory(sampleCategoryType);
            BenthosBatchRowModel firstAncestorRow = row.getFirstAncestor(sampleCategory);
            int firstAncestorIndex = tableModel.getRowIndex(firstAncestorRow);
            if (rowIndex != firstAncestorIndex) {

                // ancestor is not this row
                // then only save ancestor

                if (log.isInfoEnabled()) {
                    log.info("Sample category " + sampleCategoryType +
                             " weight was modified, First ancestor row: " +
                             firstAncestorIndex + " will save it");
                }
                saveRow(firstAncestorRow);

                // when row valid state has changed, recompute action enabled states
                recomputeBatchActionEnable();

                cleanrRowMonitor();

                return;
            }

            // modified sample weight is a leaf
            // will save it after
        }

        saveSelectedRowIfNeeded();

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<BenthosBatchRowModel> rowMonitor,
                                             BenthosBatchRowModel row) {
        // there is a valid bean attached to the monitor
        if (rowMonitor.wasModified()) {

            // monitored bean was modified, save it
            if (log.isInfoEnabled()) {
                log.info("Row " + row + " was modified, will save it");
            }

            showInformationMessage(
                    "[ Captures - Benthos ] " +
                    "Sauvegarde des modifications de " + row + '.');

            rowMonitor.setBean(null);
            saveRow(row);
            rowMonitor.setBean(row);

            // clear modified flag on the monitor
            rowMonitor.clearModified();
        }
    }

    @Override
    protected void onModelRowsChanged(List<BenthosBatchRowModel> rows) {
        super.onModelRowsChanged(rows);

        BenthosBatchUIModel model = getModel();
        model.setRootNumber(0);

        for (BenthosBatchRowModel row : rows) {
            updateTotalFromFrequencies(row);
            if (row.isBatchRoot()) {

                // update speciesUsed
                addToSpeciesUsed(row);
            }
        }
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          BenthosBatchRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             BenthosBatchRowModel oldRow,
                                             int newRowIndex,
                                             BenthosBatchRowModel newRow) {
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);

        // when selected row has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void addHighlighters(JXTable table) {

        super.addHighlighters(table);

        Color toConfirmColor = getConfig().getColorRowToConfirm();

        // paint the cell in orange if the row is to confirm
        Highlighter confirmHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate() {

                    public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                        BenthosBatchRowModel row = getTableModel().getEntry(adapter.row);
                        return row.isSpeciesToConfirm();
                    }

                }, toConfirmColor);
        table.addHighlighter(confirmHighlighter);

        // highlight only the species column if the row is selected
        Highlighter selectedHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.IS_SELECTED,
                        new HighlightPredicate.IdentifierHighlightPredicate(BenthosBatchTableModel.SPECIES)),
                UIManager.getColor("Table[Enabled+Selected].textBackground"));

        table.addHighlighter(selectedHighlighter);

        // paint the cell in dark orange if the row is to confirm and the cell is not editable
        Highlighter confirmNotEditableHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate() {

                    public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                        BenthosBatchRowModel row = getTableModel().getEntry(adapter.row);
                        return row.isSpeciesToConfirm() && !adapter.isEditable();
                    }

                }, toConfirmColor.darker());
        table.addHighlighter(confirmNotEditableHighlighter);

        // paint in a special color for comment cell (with not null value)
        Color cellWithValueColor = getConfig().getColorCellWithValue();

        Highlighter commentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        new HighlightPredicate.IdentifierHighlightPredicate(BenthosBatchTableModel.COMMENT),
                        // for not null value
                        new HighlightPredicate() {
                            @Override
                            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                                String value = (String) adapter.getValue();
                                return StringUtils.isNotBlank(value);
                            }
                        }), cellWithValueColor);
        table.addHighlighter(commentHighlighter);

        // paint in a special color for attachment cell (when some attachments)

        Highlighter attachmentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        new HighlightPredicate.IdentifierHighlightPredicate(BenthosBatchTableModel.ATTACHMENT),
                        // for not null value
                        new HighlightPredicate() {
                            @Override
                            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                                Collection attachments = (Collection) adapter.getValue();
                                return CollectionUtils.isNotEmpty(attachments);
                            }
                        }
                ), cellWithValueColor);
        table.addHighlighter(attachmentHighlighter);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<BenthosBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("beforeInit: " + ui);
        }

        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);

        BenthosBatchUIModel model = new BenthosBatchUIModel(catchesUIModel);
        model.setTableViewMode(TableViewMode.ALL);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("afterInit: " + ui);
        }

        initUI(ui);

        Map<Integer, SampleCategoryEnum> categoryEnumMap =
                SampleCategoryEnum.toIdMapping();

        List<SampleCategoryEnum> samplingOrder = Lists.newArrayList();

        List<Integer> samplingOrderIds = getConfig().getServiceConfig().getSamplingOrderIds();
        for (Integer id : samplingOrderIds) {
            samplingOrder.add(categoryEnumMap.get(id));
        }
        if (log.isInfoEnabled()) {
            log.info("Will use sampling order: " + samplingOrder);
        }

        JXTable table = getTable();

        // can show / hide some columns in model
        table.setColumnControlVisible(true);

        // create table column model
        TableCellRenderer defaultRenderer =
                table.getDefaultRenderer(Object.class);

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        Decorator<CaracteristicQualitativeValue> caracteristicDecorator =
                getDecorator(CaracteristicQualitativeValue.class, null);

        Color computedDataColor = getConfig().getColorComputedWeights();

        { // Species column

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Species.class, DecoratorService.FROM_PROTOCOL),
                             BenthosBatchTableModel.SPECIES);
        }

        { // SortedUnsortedCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           BenthosBatchTableModel.SORTED_UNSORTED_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        for (SampleCategoryEnum sampleCategoryType : samplingOrder) {
            switch (sampleCategoryType) {

                case size:

                { // SizeCategory column

                    addSampleCategoryColumnToModel(columnModel,
                                                   BenthosBatchTableModel.SIZE_CATEGORY,
                                                   caracteristicDecorator,
                                                   defaultRenderer);
                }
                break;
                case sex:

                { // SexCategory column

                    addSampleCategoryColumnToModel(columnModel,
                                                   BenthosBatchTableModel.SEX_CATEGORY,
                                                   caracteristicDecorator,
                                                   defaultRenderer);
                }
                break;
                case maturity:

                { // MaturityCategory column

                    addSampleCategoryColumnToModel(columnModel,
                                                   BenthosBatchTableModel.MATURITY_CATEGORY,
                                                   caracteristicDecorator,
                                                   defaultRenderer);
                }

                break;
                case age:


                { // AgeCategory column

                    addSampleCategoryColumnToModel(columnModel,
                                                   BenthosBatchTableModel.AGE_CATEGORY,
                                                   getDecorator(Float.class, null),
                                                   defaultRenderer);
                }

                break;
            }
        }


        { // Weight column

            addColumnToModel(columnModel,
                             TuttiComputedOrNotDataTableCell.newEditor(
                                     Float.class, false, true, 3, computedDataColor),
                             TuttiComputedOrNotDataTableCell.newRender(
                                     defaultRenderer, true, 3, computedDataColor),
                             BenthosBatchTableModel.WEIGHT);
        }

        { // Number column (from frequencies)

            addColumnToModel(columnModel,
                             BenthosFrequencyCellComponent.newEditor(ui, computedDataColor),
                             BenthosFrequencyCellComponent.newRender(computedDataColor),
                             BenthosBatchTableModel.COMPUTED_NUMBER);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             CommentCellEditor.newEditor(ui),
                             CommentCellRenderer.newRender(),
                             BenthosBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellEditor.newEditor(ui),
                             AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                             BenthosBatchTableModel.ATTACHMENT);
        }

        { // Species to confirm column

            addBooleanColumnToModel(columnModel,
                                    BenthosBatchTableModel.SPECIES_TO_CONFIRM,
                                    getTable());
        }

        // create table model
        BenthosBatchTableModel tableModel =
                new BenthosBatchTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);

        getModel().addPropertyChangeListener(BenthosBatchUIModel.PROPERTY_TABLE_VIEW_MODE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                TableViewMode tableViewMode = (TableViewMode) evt.getNewValue();

                if (tableViewMode == null) {
                    tableViewMode = TableViewMode.ALL;
                }

                if (log.isDebugEnabled()) {
                    log.debug("Will use rowfilter for viewMode: " + tableViewMode);
                }
                RowFilter<BenthosBatchTableModel, Integer> filter = tableFilters.get(tableViewMode);
                getTable().setRowFilter(filter);
            }
        });
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getTable();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("Closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void createBatch() {

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        CreateBenthosBatchUI createBatchEditor = parent.getBenthosTabCreateBatch();

        createBatchEditor.getHandler().openUI(getModel());
        parent.getHandler().setBenthosSelectedCard(EditCatchesUIHandler.CREATE_BATCH_CARD);
    }

    public void addBatch(CreateBenthosBatchUIModel bethosBatchRootRowModel) {
        if (bethosBatchRootRowModel.isValid()) {

            BenthosBatchTableModel tableModel = getTableModel();

            BenthosBatchRowModel newRow = tableModel.createNewRow();
            Species species = bethosBatchRootRowModel.getSpecies();
            newRow.setSpecies(species);

            CaracteristicQualitativeValue sortedUnsortedCategory = bethosBatchRootRowModel.getSortedUnsortedCategory();
            SampleCategory<CaracteristicQualitativeValue> category = newRow.getSortedUnsortedCategory();
            category.setCategoryValue(sortedUnsortedCategory);
            category.setCategoryWeight(bethosBatchRootRowModel.getBatchWeight());
            newRow.setSampleCategory(category);

            recomputeRowValidState(newRow);

            saveRow(newRow);

            tableModel.addNewRow(newRow);
            AbstractSelectTableAction.doSelectCell(getTable(), tableModel.getRowCount() - 1, 0);

            // update speciesUsed
            addToSpeciesUsed(newRow);
        }

        recomputeBatchActionEnable();
    }

    public void splitBatch() {

        JXTable table = getTable();

        // get selected row
        int rowIndex = table.getSelectedRow();

        Preconditions.checkState(rowIndex != -1,
                                 "Cant split batch if no batch selected");

        BenthosBatchTableModel tableModel = getTableModel();

        BenthosBatchRowModel parentBatch = tableModel.getEntry(rowIndex);

        boolean split = true;
        if (parentBatch.getWeight() != null) {
            String htmlMessage = String.format(
                    CONFIRMATION_FORMAT,
                    _("tutti.editBenthosBatch.split.weightNotNull.message"),
                    _("tutti.editBenthosBatch.split.weightNotNull.help"));
            int i = JOptionPane.showConfirmDialog(
                    getTopestUI(),
                    htmlMessage,
                    _("tutti.editBenthosBatch.split.weightNotNull.title"),
                    JOptionPane.OK_CANCEL_OPTION);

            if (i == JOptionPane.OK_OPTION) {
                parentBatch.setWeight(null);

            } else {
                split = false;
            }
        }

        if (split) {
            if (log.isInfoEnabled()) {
                log.info("Open split batch ui for row [" + rowIndex + ']');
            }

            EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
            SplitBenthosBatchUI splitBatchEditor = parent.getBenthosTabSplitBatch();

            splitBatchEditor.getHandler().editBatch(parentBatch);
            parent.getHandler().setBenthosSelectedCard(EditCatchesUIHandler.SPLIT_BATCH_CARD);
        }
    }

    public void splitBatch(SplitBenthosBatchUIModel splitModel) {
        if (splitModel.isValid()) {

            JXTable table = getTable();

            // get selected row
            int insertRow = table.getSelectedRow();

            BenthosBatchTableModel tableModel = getTableModel();
            BenthosBatchRowModel parentBatch = tableModel.getEntry(insertRow);

            // create batch rows

            SampleCategoryEnum sampleCategoryEnum = splitModel.getSelectedCategory();

            // Create rows in batch table model

            List<BenthosBatchRowModel> newBatches = Lists.newArrayList();
            for (SplitBenthosBatchRowModel row : splitModel.getRows()) {
                if (row.isValid()) {

                    // can keep this row
                    BenthosBatchRowModel newBatch = tableModel.createNewRow();

                    loadBatchRow(parentBatch,
                                 newBatch,
                                 sampleCategoryEnum,
                                 row.getCategoryValue(),
                                 row.getWeight(),
                                 null);

                    recomputeRowValidState(newBatch);
                    newBatches.add(newBatch);

                    tableModel.addNewRow(++insertRow, newBatch);
                }
            }

            // add new batches to his parent
            parentBatch.setChildBatch(newBatches);

            //TODO Should only save parentBatch (will persist all his childs)
            //saveRow(parentBatch);

            // save new batches
            saveRows(newBatches);

            BenthosBatchUIModel model = getModel();
            model.setLeafNumber(model.getLeafNumber() + newBatches.size() - 1);
        }

        recomputeBatchActionEnable();

//        // reselect this cell
//        AbstractSelectTableAction.doSelectCell(table, rowIndex, 0);
//        table.requestFocus();
    }

    public void updateTotalFromFrequencies(BenthosBatchRowModel row) {
        List<BenthosFrequencyRowModel> frequency = row.getFrequency();

        Integer totalNumber = 0;
        boolean onlyOneFrequency = false;
        if (CollectionUtils.isNotEmpty(frequency)) {
            for (BenthosFrequencyRowModel frequencyModel : frequency) {
                if (frequencyModel.getNumber() != null) {
                    totalNumber += frequencyModel.getNumber();
                }
            }
            onlyOneFrequency = frequency.size() == 1;
        }
        row.setComputedNumber(totalNumber);
        row.getFinestCategory().setOnlyOneFrequency(onlyOneFrequency);

    }

    public void saveRows(Iterable<BenthosBatchRowModel> rows) {
        for (BenthosBatchRowModel row : rows) {
            saveRow(row);
        }
    }

    protected void saveRow(BenthosBatchRowModel row) {

        FishingOperation fishingOperation = getModel().getFishingOperation();
        Preconditions.checkNotNull(fishingOperation);

        Preconditions.checkNotNull(row.getSpecies());
        SampleCategory<?> sampleCategory = row.getSampleCategory();
        Preconditions.checkNotNull(sampleCategory);
        Preconditions.checkNotNull(sampleCategory.getCategoryType());
        Preconditions.checkNotNull(sampleCategory.getCategoryValue());

        BenthosBatch catchBean = row.toBean();
        catchBean.setFishingOperation(fishingOperation);

        BenthosBatchRowModel parent = row.getParentBatch();
        if (parent != null) {
            catchBean.setParentBatch(parent.toBean());
        }

        // apply sample category
        catchBean.setSampleCategoryType(sampleCategory.getCategoryType());
        catchBean.setSampleCategoryValue(sampleCategory.getCategoryValue());
        catchBean.setSampleCategoryWeight(sampleCategory.getCategoryWeight());

        if (TuttiEntities.isNew(catchBean)) {

            BenthosBatchRowModel batchParent = row.getParentBatch();
            String parentBatchId = null;

            if (batchParent != null) {
                parentBatchId = batchParent.getId();
            }

            if (log.isInfoEnabled()) {
                log.info("Persist new species batch with parentId: " +
                         parentBatchId);
            }
            catchBean = persistenceService.createBenthosBatch(catchBean,
                                                              parentBatchId);
            row.setId(catchBean.getId());
        } else {
            if (log.isInfoEnabled()) {
                log.info("Persist existing species batch: " + catchBean.getId() + " (parent : " + catchBean.getParentBatch() + ")");
            }
            persistenceService.saveBenthosBatch(catchBean);
        }

        List<BenthosFrequencyRowModel> frequencyRows = row.getFrequency();

        List<BenthosBatchFrequency> frequency =
                BenthosFrequencyRowModel.toBeans(frequencyRows, catchBean);

        if (log.isInfoEnabled()) {
            log.info("Will save " + frequency.size() + " frequencies.");
        }
        frequency = persistenceService.saveBenthosBatchFrequency(
                catchBean.getId(), frequency);

        // push it back to row model
        frequencyRows = BenthosFrequencyRowModel.fromBeans(frequency);
        row.setFrequency(frequencyRows);
    }

    public String getFilterBenthosBatchRootButtonText(int rootNumber) {
        return _("tutti.editBenthosBatch.filterBatch.mode.root", rootNumber);
    }

    protected void recomputeBatchActionEnable() {

        int rowIndex = getTable().getSelectedRow();

        //TODO Improve this test
        boolean enableAdd = true;
//                CollectionUtils.isNotEmpty(getModel().getAvailableSpecies());

        boolean enableRename = false;
        boolean enableSplit = false;
        boolean enableRemove = false;
        boolean enableRemoveSub = false;
        boolean enableCreateMelag = false;

        if (rowIndex != -1) {

            // there is a selected row

            //TODO If there is some sub-batch, can remove them
            //TODO If there is no sub-batch, can split current batch

            BenthosBatchTableModel tableModel = getTableModel();
            BenthosBatchRowModel row = tableModel.getEntry(rowIndex);
            int selectedRowCount = getTable().getSelectedRowCount();

            enableSplit = true;
            enableRemove = true;
            enableRemoveSub = true;
            enableRename = true;
            enableCreateMelag = true;

            if (enableSplit) {

                // can split if selected batch is a leaf
                Map<Integer, SampleCategoryEnum> categoryEnumMap =
                        SampleCategoryEnum.toIdMapping();

                List<Integer> samplingOrderIds = getConfig().getServiceConfig().getSamplingOrderIds();
                int lastSamplingId = samplingOrderIds.get(samplingOrderIds.size() - 1);
                SampleCategoryEnum lastCategory = categoryEnumMap.get(lastSamplingId);

                // can split if selected batch is a leaf
                enableSplit = row.isBatchLeaf()
                              && selectedRowCount == 1
                              && !lastCategory.equals(row.getFinestCategory().getCategoryType())
                              && row.getNumber() == null
                              && (row.getComputedNumber() == null
                                  || row.getComputedNumber() == 0);
            }

            if (enableRename) {

                // can rename if selected batch is a parent
                enableRename = row.isBatchRoot()
                               && selectedRowCount == 1;
            }

            if (enableRemove) {

                // can always remove the batch
                enableRemove = selectedRowCount == 1;
            }

            if (enableRemoveSub) {

                // can remove sub batch if selected batch is not a leaf
                enableRemoveSub = !row.isBatchLeaf()
                                  && selectedRowCount == 1;
            }

            if (enableCreateMelag) {

                JXTable table = getTable();

                // can add species to a melag if several root are selected

                if (selectedRowCount < 2) {
                    enableCreateMelag = false;

                } else {
                    int[] selectedRows = table.getSelectedRows();
                    for (int selectedRowIndex : selectedRows) {
                        BenthosBatchRowModel selectedRow =
                                tableModel.getEntry(selectedRowIndex);

                        if (!selectedRow.isBatchRoot()) {
                            enableCreateMelag = false;
                            break;
                        }
                    }
                }
            }
        }

        BenthosBatchUIModel model = getModel();
        model.setCreateBatchEnabled(enableAdd);
        model.setSplitBatchEnabled(enableSplit);
        model.setRemoveBatchEnabled(enableRemove);
        model.setRemoveSubBatchEnabled(enableRemoveSub);
        model.setRenameBatchEnabled(enableRename);
        model.setCreateMelagEnabled(enableCreateMelag);
    }

    public void collectChildren(BenthosBatchRowModel row,
                                Set<BenthosBatchRowModel> collectedRows) {

        if (!row.isBatchLeaf()) {

            for (BenthosBatchRowModel batchChild : row.getChildBatch()) {
                collectedRows.add(batchChild);
                collectChildren(batchChild, collectedRows);
            }
        }
    }

    public BenthosBatchRowModel loadBatch(BenthosBatch aBatch,
                                          BenthosBatchRowModel parentRow,
                                          List<BenthosBatchRowModel> rows) {

        String id = aBatch.getId();

        List<BenthosBatchFrequency> frequencies =
                persistenceService.getAllBenthosBatchFrequency(id);

        BenthosBatchRowModel newRow =
                new BenthosBatchRowModel(aBatch, frequencies);

        List<Attachment> attachments =
                persistenceService.getAllAttachments(newRow.getObjectType(),
                                                     newRow.getObjectId());

        newRow.addAllAttachment(attachments);

        // set the surveycode, do it only on the parent,
        // the species of the parent is set to the children in loadBatchRow
        if (parentRow == null && context.isProtocolFilled()) {
            // get the surveycode from the species list of the model
            List<Species> speciesList = getDataContext().getReferentBenthosWithSurveyCode();
            int i = speciesList.indexOf(newRow.getSpecies());
            if (i > -1) {
                newRow.setSpecies(speciesList.get(i));
            }
        }

        SampleCategoryEnum sampleCategoryEnum = aBatch.getSampleCategoryType();

        Preconditions.checkNotNull(
                sampleCategoryEnum,
                "Can't have a batch with no sample category, but was: " + aBatch);

        loadBatchRow(parentRow,
                     newRow,
                     sampleCategoryEnum,
                     aBatch.getSampleCategoryValue(),
                     aBatch.getSampleCategoryWeight(),
                     aBatch.getSampleCategoryComputedWeight());

        rows.add(newRow);

        if (!aBatch.isChildBatchsEmpty()) {

            // create batch childs rows

            List<BenthosBatchRowModel> batchChilds = Lists.
                    newArrayListWithCapacity(aBatch.sizeChildBatchs());

            Float childrenWeights = 0f;
            for (BenthosBatch childBatch : aBatch.getChildBatchs()) {
                BenthosBatchRowModel childRow = loadBatch(childBatch, newRow, rows);
                if (childrenWeights != null) {
                    Float weight = childRow.getFinestCategory().getNotNullWeight();
                    if (weight == null) {
                        childrenWeights = null;
                    } else {
                        childrenWeights += weight;
                    }
                }
                batchChilds.add(childRow);
            }

            Float rowWeight = newRow.getFinestCategory().getNotNullWeight();
            boolean subSample = rowWeight != null && childrenWeights != null
                                && childrenWeights < rowWeight;
            for (BenthosBatchRowModel childRow : batchChilds) {
                childRow.getFinestCategory().setSubSample(subSample);
            }

            newRow.setChildBatch(batchChilds);
        }

        return newRow;
    }

    protected void loadBatchRow(BenthosBatchRowModel parentRow,
                                BenthosBatchRowModel newRow,
                                SampleCategoryEnum sampleCategoryEnum,
                                Serializable categoryValue,
                                Float categoryWeight,
                                Float categoryComputedWeight) {

        // get sample category from his type
        SampleCategory sampleCategory =
                newRow.getSampleCategory(sampleCategoryEnum);

        // fill it
        sampleCategory.setCategoryValue(categoryValue);
        sampleCategory.setCategoryWeight(categoryWeight);
        sampleCategory.setComputedWeight(categoryComputedWeight);

        // push it back to row as his *main* sample category
        newRow.setSampleCategory(sampleCategory);

        if (parentRow != null) {

            // copy back parent data (mainly other sample categories)

            newRow.setSpecies(parentRow.getSpecies());
            newRow.setSpeciesToConfirm(parentRow.isSpeciesToConfirm());
            newRow.setParentBatch(parentRow);

            newRow.setSpecies(parentRow.getSpecies());
            if (sampleCategoryEnum != SampleCategoryEnum.sortedUnsorted) {
                newRow.setSortedUnsortedCategory(parentRow.getSortedUnsortedCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.size) {
                newRow.setSizeCategory(parentRow.getSizeCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.sex) {
                newRow.setSexCategory(parentRow.getSexCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.maturity) {
                newRow.setMaturityCategory(parentRow.getMaturityCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.age) {
                newRow.setAgeCategory(parentRow.getAgeCategory());
            }
        }
    }

    protected <C extends Serializable> void addSampleCategoryColumnToModel(TableColumnModel columnModel,
                                                                           ColumnIdentifier<BenthosBatchRowModel> columnIdentifier,
                                                                           Decorator<C> decorator,
                                                                           TableCellRenderer defaultRenderer) {
        addColumnToModel(
                columnModel,
                SampleCategoryComponent.newEditor(decorator),
                SampleCategoryComponent.newRender(defaultRenderer,
                                                  decorator,
                                                  getConfig().getColorComputedWeights()),
                columnIdentifier);
    }

    public void removeFromSpeciesUsed(BenthosBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getSpecies());
        Preconditions.checkNotNull(row.getSortedUnsortedCategoryValue());
        if (log.isInfoEnabled()) {
            log.info("Remove from speciesUsed: " + decorate(row.getSortedUnsortedCategoryValue()) + " - " + decorate(row.getSpecies()));
        }
        BenthosBatchUIModel model = getModel();
        model.getSpeciesUsed().remove(row.getSortedUnsortedCategoryValue(),
                                      row.getSpecies());

        if (row.isBatchRoot()) {
            model.setRootNumber(model.getRootNumber() - 1);
        }
    }

    protected void addToSpeciesUsed(BenthosBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getSpecies());
        Preconditions.checkNotNull(row.getSortedUnsortedCategoryValue());
        if (log.isDebugEnabled()) {
            log.debug("Add to speciesUsed: " +
                      decorate(row.getSortedUnsortedCategoryValue()) +
                      " - " + decorate(row.getSpecies()));
        }
        BenthosBatchUIModel model = getModel();
        model.getSpeciesUsed().put(row.getSortedUnsortedCategoryValue(),
                                   row.getSpecies());

        model.setRootNumber(model.getRootNumber() + 1);
    }

    public Species openAddSpeciesDialog(String title, List<Species> species) {
        SelectSpeciesUI dialogContent = new SelectSpeciesUI(ui);
        SelectSpeciesUIModel model = dialogContent.getModel();
        model.setSelectedSpecies(null);
        model.setSpecies(species);

        openDialog(dialogContent, title, new Dimension(400, 130));

        return model.getSelectedSpecies();
    }

    protected BenthosBatch convertRowToBean(BenthosBatchRowModel row, boolean convertParent) {
        SampleCategory<?> sampleCategory = row.getFinestCategory();
        Preconditions.checkNotNull(sampleCategory);
        Preconditions.checkNotNull(sampleCategory.getCategoryType());
        Preconditions.checkNotNull(sampleCategory.getCategoryValue());

        BenthosBatch catchBean = row.toBean();

        // apply sample category
        catchBean.setSampleCategoryType(sampleCategory.getCategoryType());
        catchBean.setSampleCategoryValue(sampleCategory.getCategoryValue());
        catchBean.setSampleCategoryWeight(sampleCategory.getCategoryWeight());

        if (convertParent && row.getParentBatch() != null) {
            BenthosBatch parent = convertRowToBean(row.getParentBatch(), true);
            catchBean.setParentBatch(parent);
        }

        return catchBean;
    }
}