package fr.ifremer.tutti.ui.swing.content.cruise;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditCruiseUIHandler.java 581 2013-03-12 09:10:25Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/cruise/EditCruiseUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.referential.Gear;
import fr.ifremer.tutti.persistence.entities.referential.Person;
import fr.ifremer.tutti.persistence.entities.referential.Vessel;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.CloseableUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * Handler of UI {@link EditCruiseUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EditCruiseUIHandler extends AbstractTuttiUIHandler<EditCruiseUIModel, EditCruiseUI> implements CloseableUI {

    /** Logger. */
    private static final Log log = LogFactory.getLog(EditCruiseUIHandler.class);

    public static String getTitle(boolean exist) {

        String result;
        if (exist) {
            result = _("tutti.title.edit.cruise");
        } else {
            result = _("tutti.title.create.cruise");
        }
        return result;
    }

    /**
     * Persistence service.
     *
     * @since 0.1
     */
    private final PersistenceService persistenceService;

    public EditCruiseUIHandler(TuttiUI parentUi, EditCruiseUI ui) {
        super(parentUi.getHandler().getContext(), ui);
        persistenceService = context.getPersistenceService();
    }

    @Override
    public void beforeInitUI() {

        EditCruiseUIModel model = new EditCruiseUIModel();

        if (context.isCruiseFilled()) {

            // load existing cruise
            Cruise cruise = getDataContext().getCruise();

            model.fromBean(cruise);

            // load attachments
            List<Attachment> attachments = persistenceService.getAllAttachments(Integer.valueOf(model.getObjectId()));
            model.addAllAttachment(attachments);

            if (log.isDebugEnabled()) {
                log.debug("Will edit cruise: " + cruise.getId());
            }
        } else {

            // create new cruise
            Program program = getDataContext().getProgram();
            model.setProgram(program);
            model.setMultirigNumber(1);


            if (log.isDebugEnabled()) {
                log.debug("Will create a new cruise from program: " + program);
            }
        }
        listModelIsModify(model);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        List<Person> users = persistenceService.getAllPerson();

        EditCruiseUIModel model = getModel();

        final List<Vessel> scientificVesselList = getDataContext().getScientificVessels();
        final List<Vessel> fishingVesselList = getDataContext().getFishingVessels();
        final List<Vessel> allVesselList = Lists.newArrayList();
        allVesselList.addAll(scientificVesselList);
        allVesselList.addAll(fishingVesselList);

        final List<Gear> scientificGearList = getDataContext().getScientificGears();
        final List<Gear> fishingGearList = getDataContext().getFishingGears();
        final List<Gear> allGearList = Lists.newArrayList();
        allGearList.addAll(scientificGearList);
        allGearList.addAll(fishingGearList);

        // Change vessel list and gear list when vessel type changes
        model.addPropertyChangeListener(EditCruiseUIModel.PROPERTY_VESSEL_TYPE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                VesselTypeEnum vesselType = (VesselTypeEnum) evt.getNewValue();

                Predicate<Vessel> vesselPredicate = null;
                Predicate<Gear> gearPredicate = null;

                switch (vesselType) {
                    case FISHING:
                        vesselPredicate = new Predicate<Vessel>() {

                            public boolean apply(Vessel input) {
                                return !input.isScientificVessel();
                            }
                        };
                        gearPredicate = new Predicate<Gear>() {

                            public boolean apply(Gear input) {
                                return !input.isScientificGear();
                            }
                        };
                        break;

                    case SCIENTIFIC:
                        vesselPredicate = new Predicate<Vessel>() {

                            public boolean apply(Vessel input) {
                                return input.isScientificVessel();
                            }
                        };
                        gearPredicate = new Predicate<Gear>() {

                            public boolean apply(Gear input) {
                                return input.isScientificGear();
                            }
                        };
                        break;

                }
                ui.getVesselList().getHandler().clearFilters();
                ui.getGearList().getHandler().clearFilters();

                if (vesselPredicate != null) {
                    ui.getVesselList().getHandler().addFilter(vesselPredicate);
                }
                if (gearPredicate != null) {
                    ui.getGearList().getHandler().addFilter(gearPredicate);
                }

            }
        });

        initBeanFilterableComboBox(ui.getProgramComboBox(),
                                   persistenceService.getAllProgram(),
                                   model.getProgram());

        initBeanFilterableComboBox(ui.getCountryComboBox(),
                                   persistenceService.getAllCountry(),
                                   model.getCountry());


        initBeanList(ui.getVesselList(),
                     allVesselList,
                     model.getVessel());

        initBeanList(ui.getGearList(),
                     allGearList,
                     model.getGear());

        initBeanList(ui.getHeadOfMissionList(),
                     users,
                     model.getHeadOfMission());

        initBeanList(ui.getHeadOfSortRoomList(),
                     users,
                     model.getHeadOfSortRoom());

        SwingValidator validator = ui.getValidator();
        listenValidatorValid(validator, model);

        // if new fishingOperation can already cancel his creation
        model.setModify(model.isCreate());

        model.setVesselType(VesselTypeEnum.SCIENTIFIC);

        registerValidators(validator);

    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        clearValidators();

        ui.getAttachmentsButton().onCloseUI();
    }

    @Override
    public boolean quitUI() {
        boolean result = quitScreen(
                getModel().isValid(),
                getModel().isModify(),
                _("tutti.dialog.askCancelEditBeforeLeaving.cancelSaveCruise"),
                _("tutti.dialog.askSaveBeforeLeaving.saveCruise"),
                ui.getSaveButton().getAction()
        );
        return result;
    }

    @Override
    public SwingValidator<EditCruiseUIModel> getValidator() {
        return ui.getValidator();
    }

    public void generateCampaignName() {

        EditCruiseUIModel model = getModel();
        String name = model.getGeneratedCampaignName();
        model.setName(name);
    }

}
