package fr.ifremer.tutti.ui.swing.content.operation.benthos;

/*
 * #%L
 * Tutti :: UI
 * $Id: BenthosBatchUIHandler.java 82 2012-12-17 09:18:52Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/benthos/BenthosBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.BenthosBatch;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.editor.AttachmentCellComponent;
import fr.ifremer.tutti.ui.swing.util.editor.LongTextCellComponent;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.table.TableColumnModel;
import java.util.List;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class BenthosBatchUIHandler extends AbstractTuttiBatchTableUIHandler<BenthosBatchRowModel, BenthosBatchUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(BenthosBatchUIHandler.class);

    /**
     * UI.
     *
     * @since 0.2
     */
    private final BenthosBatchUI ui;

    public BenthosBatchUIHandler(FishingOperationsUI parentUi,
                                 BenthosBatchUI ui) {
        super(parentUi,
              new String[]{BenthosBatchUIModel.PROPERTY_TOTAL_WEIGHT,
                           BenthosBatchUIModel.PROPERTY_SAMPLE_TOTAL_WEIGHT},
              BenthosBatchRowModel.PROPERTY_SPECIES_TO_CONFIRM,
              BenthosBatchRowModel.PROPERTY_SPECIES,
              BenthosBatchRowModel.PROPERTY_WEIGHT,
              BenthosBatchRowModel.PROPERTY_COMMENT);
        this.ui = ui;

    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected BenthosBatchUIModel getModel() {
        return ui.getModel();
    }

    @Override
    protected BenthosBatchTableModel getTableModel() {
        return (BenthosBatchTableModel) getTable().getModel();
    }

    @Override
    protected JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected FishingOperation getFishingOperation() {
        return getModel().getFishingOperation();
    }

    @Override
    protected String[] getRowPropertiesToIgnore() {
        return new String[]{
                BenthosBatchRowModel.PROPERTY_SAMPLE_WEIGHT,
                BenthosBatchRowModel.PROPERTY_SAMPLING_RATIO
        };
    }

    @Override
    protected TableColumnModel createTableColumnModel() {

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        {
            // Species to confirm column

            addBooleanColumnToModel(columnModel,
                                    BenthosBatchTableModel.SPECIES_TO_CONFIRM,
                                    getTable());
        }

        List<Species> allSpecies = persistenceService.getAllSpecies();

        {
            // Species (by code) column

            Decorator<Species> decorator = getDecorator(
                    Species.class, DecoratorService.SPECIES_BY_CODE);

            addComboDataColumnToModel(columnModel,
                                      BenthosBatchTableModel.SPECIES_BY_CODE,
                                      decorator, allSpecies);
        }

        {
            // Species (by genusCode) column

            Decorator<Species> decorator = getDecorator(
                    Species.class, DecoratorService.SPECIES_BY_GENUS);

            addComboDataColumnToModel(columnModel,
                                      BenthosBatchTableModel.SPECIES_BY_GENUS_CODE,
                                      decorator, allSpecies);

        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  BenthosBatchTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // Sample weight column

            addColumnToModel(columnModel,
                             BenthosBatchTableModel.SAMPLE_WEIGHT);
        }

        { // SamplingRatio column

            addColumnToModel(columnModel,
                             BenthosBatchTableModel.SAMPLING_RATIO);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             LongTextCellComponent.newEditor(ui.getLongTextEditor()),
                             LongTextCellComponent.newRender(n_("tutti.tooltip.comment.none")),
                             BenthosBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellComponent.newEditor(ui.getAttachmentEditor()),
                             AttachmentCellComponent.newRender(
                                     getDecorator(Attachment.class, null),
                                     n_("tutti.tooltip.attachment.none")),
                             BenthosBatchTableModel.ATTACHMENTS);
        }
        return columnModel;
    }

    @Override
    protected void onRowModified(BenthosBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        boolean wasValid = row.isValid();

        boolean valid = isRowValid(row);

        if (log.isInfoEnabled()) {
            log.info("Was valid / Is valid: " + wasValid + " / " + valid);
        }
        row.setValid(valid);
    }

    @Override
    protected void onRowValidStateChanged(BenthosBatchRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        int rowIndex = getTableModel().getRowIndex(row);

        if (rowIndex > -1) {
            getTableModel().fireTableRowsUpdated(rowIndex, rowIndex);
        }
    }

    @Override
    protected void onRowModifyStateChanged(BenthosBatchRowModel row,
                                           Boolean oldValue,
                                           Boolean newValue) {
    }

    @Override
    protected void onModelRowsChanged(List<BenthosBatchRowModel> rows) {
        super.onModelRowsChanged(rows);

        // set valid flag
        for (BenthosBatchRowModel row : rows) {
            boolean valid = isRowValid(row);
            row.setValid(valid);
        }
    }

    @Override
    protected void saveSelectedRowIfRequired() {

        TuttiBeanMonitor<BenthosBatchRowModel> rowMonitor = getRowMonitor();

        BenthosBatchRowModel bean = rowMonitor.getBean();
        if (bean != null) {

            if (bean.isValid()) {
                // there is a valid bean attached to the monitor

                if (rowMonitor.wasModified()) {

                    // monitored bean was modified, save it
                    if (log.isInfoEnabled()) {
                        log.info("Row " + bean + " was modified, will save it");
                    }

                    saveRow(bean);

                    // clear modified flag on the monitor
                    rowMonitor.clearModified();
                }
            } else {

                // row is not valid can not save it

                BenthosBatch catchBean = bean.toBean();

                if (!TuttiEntities.isNew(catchBean)) {

                    // remove this
                    persistenceService.deleteBenthosBatch(catchBean.getId());
                }
            }
        }
    }

    @Override
    protected boolean isRowValid(BenthosBatchRowModel row) {
        boolean result = row.getSpecies() != null && row.getWeight() != null;
        return result;
    }

    @Override
    protected void saveRow(BenthosBatchRowModel row) {

        BenthosBatch catchBean = row.toBean();

        FishingOperation fishingOperation = getModel().getFishingOperation();
        catchBean.setFishingOperation(fishingOperation);
        if (log.isInfoEnabled()) {
            log.info("Selected fishingOperation: " + fishingOperation.getId());
        }

        if (TuttiEntities.isNew(catchBean)) {

            catchBean = persistenceService.createBenthosBatch(catchBean);
            row.setId(catchBean.getId());
        } else {
            persistenceService.saveBenthosBatch(catchBean);
        }
    }

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        JXTable table = getTable();

        if (table.isEditing()) {

            // make sure to stop editor
            table.editingCanceled(null);
        }

        if (fishingOperationMonitor.wasModified()) {

            // previous fishingOperation was modified, let's save it
            BenthosBatchUIModel beanToSave = fishingOperationMonitor.getBean();

            FishingOperation fishingOperation = beanToSave.toBean();

            if (log.isInfoEnabled()) {
                log.info("FishingOperation " + fishingOperation.getId() +
                         " was modified, will save it.");
            }

            persistenceService.saveFishingOperation(fishingOperation);
        }

        // make sure selection is empty (will remove bean from monitor)
        table.clearSelection();

        boolean empty = bean == null;

        BenthosBatchUIModel model = getModel();

        List<BenthosBatchRowModel> rows;

        String fishingoperationText;

        if (empty) {
            rows = null;
            bean = new FishingOperation();
            model.setFishingOperation(null);
            fishingoperationText = "";
        } else {

            Decorator<FishingOperation> decorator =
                    getDecorator(FishingOperation.class, null);

            fishingoperationText =
                    _("tutti.label.traitReminder", decorator.toString(bean));

            if (log.isInfoEnabled()) {
                log.info("Get species batch for fishingOperation: " +
                         bean.getId() + " - " + fishingoperationText);
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {
                List<BenthosBatch> catches =
                        persistenceService.getAllBenthosBatch(bean.getId());
                for (BenthosBatch aBatch : catches) {
                    BenthosBatchRowModel entry =
                            new BenthosBatchRowModel(aBatch);
                    rows.add(entry);
                }
            }
            model.setFishingOperation(bean);
        }

        model.fromBean(bean);
        model.setRows(rows);

        fishingOperationMonitor.clearModified();

        ui.getFishingOperationReminderLabel().setText(fishingoperationText);
    }


    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInitUI() {

        if (log.isInfoEnabled()) {
            log.info("beforeInit: " + ui);
        }

        BenthosBatchUIModel model = new BenthosBatchUIModel();
        ui.setContextValue(model);

        fishingOperationMonitor.setBean(model);
    }

    @Override
    public void afterInitUI() {

        if (log.isInfoEnabled()) {
            log.info("afterInit: " + ui);
        }

        initUI(ui);

        JXTable table = getTable();

        // create table column model
        TableColumnModel columnModel = createTableColumnModel();

        // create table model
        BenthosBatchTableModel tableModel =
                new BenthosBatchTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

}