package fr.ifremer.tutti.ui.swing.content.campaign;

/*
 * #%L
 * Tutti :: UI
 * $Id: SelectCampaignUIHandler.java 15 2012-11-24 13:10:24Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/campaign/SelectCampaignUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.CampaignBean;
import fr.ifremer.tutti.persistence.entities.SurveyBean;
import fr.ifremer.tutti.persistence.entities.TuttiBeans;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.TuttiScreen;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import jaxx.runtime.swing.editor.bean.BeanComboBox;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

/**
 * Main ui content to select campaign.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class SelectCampaignUIHandler extends AbstractTuttiUIHandler<SelectCampaignUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SelectCampaignUIHandler.class);

    protected final PersistenceService persistenceService;

    protected final SelectCampaignUI ui;

    public SelectCampaignUIHandler(TuttiUIContext context, SelectCampaignUI ui) {
        super(context);
        this.ui = ui;
        persistenceService = context.getService(PersistenceService.class);
    }

    @Override
    protected SelectCampaignUIModel getModel() {
        return ui.getModel();
    }

    @Override
    public void beforeInitUI() {
        SelectCampaignUIModel model = new SelectCampaignUIModel();

        List<SurveyBean> allSurveys = persistenceService.getAllSurveys();

        model.setSurveys(allSurveys);

        if (allSurveys.isEmpty()) {
            // do nothing

        } else {

            // get selected survey (if any)

            String surveyId = context.getSurveyId();
            SurveyBean selectedSurvey = null;
            List<CampaignBean> allCampaigns = null;
            if (surveyId != null) {
                selectedSurvey = TuttiBeans.findById(allSurveys, surveyId);
                allCampaigns = persistenceService.getAllCampaigns(surveyId);
            }

            model.setSurvey(selectedSurvey);
            model.setCampaigns(allCampaigns);

            if (CollectionUtils.isEmpty(allCampaigns)) {

                // nothing to select
            } else {
                String campaignId = context.getCampaignId();

                CampaignBean selectedCampaign = null;
                if (campaignId != null) {
                    selectedCampaign =
                            TuttiBeans.findById(allCampaigns, campaignId);
                }
                model.setCampaign(selectedCampaign);
            }
        }
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        SelectCampaignUIModel model = ui.getModel();

        initBeanComboBox(ui.getSurveyCombobox(),
                         model.getSurveys(),
                         model.getSurvey());

        initBeanComboBox(ui.getCampaignCombobox(),
                         model.getCampaigns(),
                         model.getCampaign());

        model.addPropertyChangeListener("survey", new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                SurveyBean newValue = (SurveyBean) evt.getNewValue();
                boolean noSurvey = newValue == null;
                context.setSurveyId(noSurvey ? null : newValue.getId());
                List<CampaignBean> campaigns;
                if (noSurvey) {
                    campaigns = Lists.newArrayList();
                } else {
                    campaigns = persistenceService.getAllCampaigns(newValue.getId());
                }
                SelectCampaignUIModel model = (SelectCampaignUIModel) evt.getSource();
                model.setCampaigns(campaigns);
                model.setCampaign(null);
            }
        });

        model.addPropertyChangeListener("campaigns", new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                // reload combo box
                BeanComboBox<CampaignBean> combobox = ui.getCampaignCombobox();
                List<CampaignBean> campaigns = (List<CampaignBean>) evt.getNewValue();
                combobox.setData(null);
                combobox.setData(campaigns);
            }
        });

        model.addPropertyChangeListener("campaign", new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                CampaignBean newValue = (CampaignBean) evt.getNewValue();
                context.setCampaignId(newValue == null ? null : newValue.getId());
            }
        });
    }

    @Override
    public void onCloseUI() {
        PropertyChangeListener[] listeners = ui.getModel().getPropertyChangeListeners();
        for (PropertyChangeListener listener : listeners) {
            getModel().removePropertyChangeListener(listener);
        }
    }

    public void showCatches() {

        context.setScreen(TuttiScreen.FILL_CATCHES);
    }

    public void showSelectedSurvey() {

        SurveyBean survey = getModel().getSurvey();
        context.setSurveyId(survey.getId());
        context.setScreen(TuttiScreen.EDIT_SURVEY);
    }

    public void showNewSurvey() {
        context.setSurveyId(null);
        context.setCampaignId(null);
        context.setScreen(TuttiScreen.EDIT_SURVEY);
    }

    public void showSelectedCampaign() {
        CampaignBean campaign = getModel().getCampaign();
        context.setCampaignId(campaign.getId());
        context.setScreen(TuttiScreen.EDIT_CAMPAIGN);
    }

    public void showNewCampaign() {
        context.setCampaignId(null);
        context.setScreen(TuttiScreen.EDIT_CAMPAIGN);
    }
}
