/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: ServiceFactory.java 1835 2010-12-21 09:28:19Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.1/vradi-services/src/main/java/com/jurismarches/vradi/services/ServiceFactory.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services;

import com.jurismarches.vradi.VradiServiceConfiguration;
import com.jurismarches.vradi.VradiServiceConfigurationHelper;
import com.jurismarches.vradi.entities.ClientImpl;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.GroupFormsImpl;
import com.jurismarches.vradi.entities.GroupImpl;
import com.jurismarches.vradi.entities.InfogeneImpl;
import com.jurismarches.vradi.entities.ModificationTagImpl;
import com.jurismarches.vradi.entities.QueryMakerImpl;
import com.jurismarches.vradi.entities.RootThesaurusImpl;
import com.jurismarches.vradi.entities.SendingImpl;
import com.jurismarches.vradi.entities.SessionImpl;
import com.jurismarches.vradi.entities.StatusImpl;
import com.jurismarches.vradi.entities.ThesaurusImpl;
import com.jurismarches.vradi.entities.UserImpl;
import com.jurismarches.vradi.entities.VradiUserImpl;
import com.jurismarches.vradi.entities.WebHarvestStreamImpl;
import com.jurismarches.vradi.entities.XmlFieldBindingImpl;
import com.jurismarches.vradi.entities.XmlStreamImpl;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.util.converter.ConverterUtil;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyServiceFactory;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.WikittyTreeNodeImpl;
import org.nuiton.wikitty.entities.WikittyUserImpl;

import java.util.Arrays;
import java.util.List;

/**
 * ServiceFactory.
 *
 * @author $Author: sletellier $
 * @version $Revision: 1835 $ $Date: 2010-12-21 10:28:19 +0100 (mar., 21 déc. 2010) $
 * @since 24 févr. 2010 22:18:17
 */
public class ServiceFactory {

    /** Log. */
    private static final Log log = LogFactory.getLog(ServiceFactory.class);

    /** Wikitty service (cached version) for remote access. */
    protected static WikittyService vradiWikittyService;

    /** Single instance of storage service. */
    protected static VradiStorageService vradiStorageService;

    /** Single instance of wikitty proxy. */
    protected static WikittyProxy wikittyProxy;

    protected static VradiServiceConfiguration config;

    
    static {
        // Dans un block static pour definition des variables 
        // d'environnement solr et jbossts
        // A voir pourquoi, les converters s'initialise mal durant les tests
        try {

            ConverterUtil.initConverters();
            config = VradiServiceConfiguration.getInstance();
            I18n.init(VradiServiceConfigurationHelper.getLocale(config));
        } catch (Exception eee) {
            log.error("Cant initialize Service factory : ", eee);
        }
    }

    public static synchronized VradiStorageService getVradiStorageService() {
        if (vradiStorageService == null) {
            vradiStorageService = new VradiStorageServiceImpl();
        }
        return vradiStorageService;
    }

    /**
     * Creates wikitty service (cached).
     * 
     * Getter for service, with default service configuration.
     *
     * @return a unique instance of WikittyService
     */
    public static synchronized WikittyService getWikittyService() {
        if (vradiWikittyService == null) {
            getWikittyService(config);
        }

        return vradiWikittyService;
    }

    /**
     * Creates wikitty service (cached).
     *
     * Constructor added for UI embedded mode, with custom configuration.
     * 
     * @param config configuration of vradi service
     * @return a unique instance of WikittyService
     */
    public static synchronized WikittyService getWikittyService(ApplicationConfig config) {
        if (vradiWikittyService == null) {

            vradiWikittyService = WikittyServiceFactory.buildWikittyService(config);
        }

        return vradiWikittyService;
    }

    /**
     * Creates wikitty proxy.
     *
     * @return a unique instance of WikittyProxy.
     */
    public static synchronized WikittyProxy getWikittyProxy() {
        if (wikittyProxy == null) {
            WikittyService wikittyService = getWikittyService();

            // init proxy on cached service
            wikittyProxy = new WikittyProxy(wikittyService);

            // post operation : register extensions
            updateExtensions(wikittyProxy);

            // post operation : reindex data if necessary
            reindexData(wikittyProxy);
        }

        return wikittyProxy;
    }

    /**
     * Register all extensions.
     * 
     * Already existing extensions with same version are skipped,
     * new extension are created and stored.
     * 
     * @param localWikittyProxy wikitty proxy
     */
    static void updateExtensions(WikittyProxy localWikittyProxy) {

        // easier if ordered
        // TODO EC20100616 don't work with required extensions :(
        List<WikittyExtension> extensions = Arrays.asList(
            ClientImpl.extensionClient,
            FormImpl.extensionForm,
            GroupImpl.extensionGroup,
            InfogeneImpl.extensionInfogene,
            ModificationTagImpl.extensionModificationTag,
            QueryMakerImpl.extensionQueryMaker,
            RootThesaurusImpl.extensionRootThesaurus,
            SendingImpl.extensionSending,
            SessionImpl.extensionSession,
            StatusImpl.extensionStatus,
            ThesaurusImpl.extensionThesaurus,
            WikittyTreeNodeImpl.extensionWikittyTreeNode,
            UserImpl.extensionUser,
            VradiUserImpl.extensionVradiUser,
            WebHarvestStreamImpl.extensionWebHarvestStream,
            WikittyUserImpl.extensionWikittyUser,
            XmlFieldBindingImpl.extensionXmlFieldBinding,
            XmlStreamImpl.extensionXmlStream,
            GroupFormsImpl.extensionGroupForms);

        localWikittyProxy.storeExtension(extensions);
    }

    /**
     * Check if version has changed and launch wikitty service reindex.
     * 
     * @param localWikittyProxy local proxy
     */
    protected static void reindexData(WikittyProxy localWikittyProxy) {
        try {

            String currentVersion = VradiServiceConfigurationHelper.getApplicationVersion(config);
            String lastVersion = VradiServiceConfigurationHelper.getServiceVersion(config);

            // different is enougth to reindex
            // currentVersion is never null, lastVersion could be
            if (!currentVersion.equals(lastVersion)) {

                // do version change migration
                ServiceMigration.versionChangeMigration(lastVersion, currentVersion, localWikittyProxy);

                // Dont reindex data on version change
//                if (log.isInfoEnabled()) {
//                    log.info(_("Version change detected : %s. Reindexing data...", lastVersion));
//                }
//                long timeBefore = System.currentTimeMillis();
//
//                // call syncEngin
//                localWikittyProxy.syncEngin();
//
//                long timeAfter = System.currentTimeMillis();
//
//                if (log.isInfoEnabled()) {
//                    log.info("Reindexing completed in " + (timeAfter - timeBefore) + " ms");
//                }

                // save new version
                VradiServiceConfigurationHelper.setServiceVersion(config, currentVersion);
                config.saveForUser();
            }
            else {
                if (log.isInfoEnabled()) {
                    log.info("No version change, skipping reindexing.");
                }
            }

            // post operation : register migration classes
            ServiceMigration.configureMigration();

        } catch (Exception eee) {
            log.error("Failled to reindex data : ", eee);
        }
    }
}
