/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: FileServiceProxy.java 1817 2010-12-01 17:20:15Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.1/vradi-services/src/main/java/com/jurismarches/vradi/services/FileServiceProxy.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package com.jurismarches.vradi.services;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;

import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.RequestEntity;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.params.HttpClientParams;
import org.apache.commons.httpclient.util.EncodingUtil;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jurismarches.vradi.VradiConstants;

/**
 * This proxy can handle i/o API and manage file upload on servlet.
 * 
 * FileService -> FileServiceProxy -> FileServlet -> FileServiceImpl.
 * 
 * Based on commons http client.
 * 
 * Principe:
 * <ul>
 *  <li>Post : upload file
 *  <li>Get : download file
 * </ul>
 * 
 * @author chatellier
 * @version $Revision: 1817 $
 * 
 * Last update : $Date: 2010-12-01 18:20:15 +0100 (mer., 01 déc. 2010) $
 * By : $Author: sletellier $
 */
public class FileServiceProxy implements FileService {

    /** log. */
    private static final Log log = LogFactory.getLog(FileServiceProxy.class);
    
    /** Remote service endpoint. Endpoint is http://xxx/vradi/file */
    protected String serviceEndpoint;

    /**
     * Constructor with remote service endpoint.
     * 
     * @param serviceEndpoint remote service endpoint
     */
    public FileServiceProxy(String serviceEndpoint) {
        if (log.isDebugEnabled()) {
            log.debug("Using " + FileServiceProxy.class.getSimpleName() + " to manage file upload");
        }
        this.serviceEndpoint = serviceEndpoint;
    }

    @Override
    public String uploadTemplate(File file, String extensionName)
            throws VradiException {
        String postUrl = buildUrl(serviceEndpoint, true,
                VradiConstants.FileType.TEMPLATE.toString(), extensionName);
        return uploadFile(file, postUrl);
    }

    @Override
    public String uploadWebHarvestScript(File file)
            throws VradiException {
        String postUrl = buildUrl(serviceEndpoint, true,
                VradiConstants.FileType.WEB_HARVEST.toString(), file.getName());
        return uploadFile(file, postUrl);
    }

    @Override
    public String uploadFormAttachment(File file, String formId)
            throws VradiException {
        String postUrl = buildUrl(serviceEndpoint, true,
                VradiConstants.FileType.ATTACHMENTS_FORM_FILES.toString(),
                formId, file.getName());
        return uploadFile(file, postUrl);
    }

    @Override
    public String uploadTempFile(File file) throws VradiException {
        String postUrl = buildUrl(serviceEndpoint, true,
                VradiConstants.FileType.TEMP_FILE.toString(), file.getName());
        return uploadFile(file, postUrl);
    }

    @Override
    public String uploadFormEmbeddedFile(File file, String formId)
            throws VradiException {
        String postUrl = buildUrl(serviceEndpoint, true,
                VradiConstants.FileType.EMBEDDED_FORM_FILES.toString(), formId,
                file.getName());
        return uploadFile(file, postUrl);
    }

    /**
     * Build post url.
     * 
     * @param endpoint endpoint
     * @param convertUrl if url must be encode
     * @param additionalParams parameters to be added to the url
     * @return the built url
     * @throws VradiException if url error
     */
    protected String buildUrl(String endpoint, boolean convertUrl, String... additionalParams)
            throws VradiException {
        
        StringBuilder url = new StringBuilder(endpoint);
        if (url.lastIndexOf("/") == url.length()-1) {
            url = url.delete(url.length() -1, url.length());
        }
        
        for (String additionalParam : additionalParams) {
            url.append("/");

            if (convertUrl) {
                try {
                    additionalParam = URLEncoder.encode(additionalParam, "UTF-8");
                } catch (UnsupportedEncodingException eee) {
                    throw new VradiException("Cant parse param '" + additionalParam + "'", eee);
                }
            }
            url.append(additionalParam);
        }
        String urlAsString = url.toString();
        return urlAsString;
    }

    protected String uploadFile(final File file, String postUrl)
            throws VradiException {

        String uri;

        try {
            // create client and timeout
            HttpClientParams params = new HttpClientParams();
            params.setConnectionManagerTimeout(8000);
            HttpClient client = new HttpClient(params);

            // send file via post
            PostMethod postMethod = new PostMethod(postUrl);

            final String filename = file.getName();

            // multipart version
            // sletellier : hack to keep file name encoding
            Part[] parts = { new FilePart(filename, file) {

                /**
                 * Write the disposition header to the output stream
                 * @param out The output stream
                 * @throws IOException If an IO problem occurs
                 * @see Part#sendDispositionHeader(java.io.OutputStream)
                 */
                @Override
                protected void sendDispositionHeader(OutputStream out)
                    throws IOException {
                    
                    super.sendDispositionHeader(out);
                    if (filename != null) {
                        out.write(EncodingUtil.getBytes(FILE_NAME, "UTF-8"));
                        out.write(EncodingUtil.getBytes(QUOTE, "UTF-8"));
                        out.write(EncodingUtil.getBytes(filename, "UTF-8"));
                        out.write(EncodingUtil.getBytes(QUOTE, "UTF-8"));
                    }
                }
            } };
            RequestEntity requestEntity = new MultipartRequestEntity(parts,
                    postMethod.getParams());

            postMethod.setRequestEntity(requestEntity);
            int response = client.executeMethod(postMethod);

            // 200 = http OK
            if (response != 200) {
                if (log.isDebugEnabled()) {
                    log.debug("File upload servlet response : " + response);
                }
                throw new VradiException(_("File upload failed : server response %d", response));
            }

            Header header = postMethod.getResponseHeader("fileuri");
            if (header == null || StringUtils.isEmpty(header.getValue())) {
                if (header != null && log.isErrorEnabled()) {
                    log.error("Can't read upload response as valid file uri : " + header.getValue());
                }
                throw new VradiException(_("File upload failed : invalid response"));
            }
            else {
                uri = header.getValue();
            }

            // get uploaded file uri
            postMethod.releaseConnection();
        } catch (FileNotFoundException ex) {
            throw new VradiException("Can't upload file", ex);
        } catch (HttpException ex) {
            throw new VradiException("Can't upload file", ex);
        } catch (IOException ex) {
            throw new VradiException("Can't upload file", ex);
        }
        return uri;

    }

    protected File downloadFile(String fileUrl) throws VradiException {

        String[] splited = fileUrl.split("/");

        // Get last
        String fullFileName = splited[splited.length - 1];

        // Convert name
        try {
            fullFileName = URLDecoder.decode(fullFileName, "UTF-8");
        } catch (UnsupportedEncodingException eee) {
            throw new VradiException("Failled to decode " + fullFileName, eee);
        }

        // Extract file and ext
        int posPoint = fullFileName.lastIndexOf('.');
        String fileName = fullFileName.substring(0, posPoint -1);
        String fileExt = fullFileName.substring(posPoint);

        File resultFile;
        try {
            // temp downloaded file
            resultFile = File.createTempFile(fileName, fileExt);
            resultFile.deleteOnExit();
            
            // download
            URL url = new URL(fileUrl);
            FileUtils.copyURLToFile(url, resultFile);
            
            if (log.isDebugEnabled()) {
                log.debug("File downloaded to " + resultFile);
            }
            
        } catch (IOException ex) {
            throw new VradiException("Can't download file", ex);
        }

        return resultFile;
    }

    @Override
    public File downloadWebHarvestScript(String uri) throws VradiException {
        return downloadWebHarvestScript(true, uri);
    }

    @Override
    public File downloadWebHarvestScript(boolean convertUrl, String uri) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, convertUrl, VradiConstants.FileType.WEB_HARVEST.toString(), uri);
        File file = downloadFile(fileUrl);
        return file;
    }

    @Override
    public File downloadTemplate(String extensionName, String templateName) throws VradiException {
        return downloadTemplate(true, extensionName, templateName);
    }

    @Override
    public File downloadTemplate(boolean convertUrl, String extensionName, String templateName) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, convertUrl, VradiConstants.FileType.TEMPLATE.toString(), extensionName, templateName);
        File file = downloadFile(fileUrl);
        return file;
    }

    @Override
    public File downloadPDF(String uri) throws VradiException {
        return downloadPDF(true, uri);
    }

    @Override
    public File downloadPDF(boolean convertUrl, String uri) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, convertUrl, VradiConstants.FileType.PDF.toString(), uri);
        File file = downloadFile(fileUrl);
        return file;
    }

    @Override
    public String getPDFUrl(String uri) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, true, VradiConstants.FileType.PDF.toString(), uri);
        return fileUrl;
    }

    @Override
    public File downloadFormAttachment(String uri) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, true, VradiConstants.FileType.ATTACHMENTS_FORM_FILES.toString(), uri);
        File file = downloadFile(fileUrl);
        return file;
    }

    @Override
    public File downloadFormAttachment(String fileName, String formId) throws VradiException {
        return downloadFormAttachment(true, fileName, formId);
    }

    @Override
    public File downloadFormAttachment(boolean convertUrl, String fileName, String formId) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, convertUrl, VradiConstants.FileType.ATTACHMENTS_FORM_FILES.toString(), formId, fileName);
        File file = downloadFile(fileUrl);
        return file;
    }

    @Override
    public File downloadFormEmbeddedFile(String uri) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, false, VradiConstants.FileType.EMBEDDED_FORM_FILES.toString(), uri);
        File file = downloadFile(fileUrl);
        return file;
    }

    @Override
    public File downloadFormEmbeddedFile(String fileName, String formId) throws VradiException {
        return downloadFormEmbeddedFile(true, fileName, formId);
    }

    @Override
    public File downloadFormEmbeddedFile(boolean convertUrl, String fileName, String formId) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, convertUrl, VradiConstants.FileType.EMBEDDED_FORM_FILES.toString(), formId, fileName);
        File file = downloadFile(fileUrl);
        return file;
    }

    @Override
    public String getFormEmbeddedUrl(String fileName, String formId) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, true, VradiConstants.FileType.EMBEDDED_FORM_FILES.toString(), formId, fileName);
        return fileUrl;
    }

    @Override
    public File downloadRequestHistory(String uri) throws VradiException {
        return downloadRequestHistory(true, uri);
    }

    @Override
    public File downloadRequestHistory(boolean convretUrl, String uri) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, convretUrl, VradiConstants.FileType.QUERY_HISTORY.toString(), uri);
        File file = downloadFile(fileUrl);
        return file;
    }

    @Override
    public String getRequestHistoryURL(String uri) throws VradiException {
        String fileUrl = buildUrl(serviceEndpoint, true, VradiConstants.FileType.QUERY_HISTORY.toString(), uri);
        return fileUrl;
    }
}
