/*
 * #%L
 * Vradi :: Services
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.UpdateResponse;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyImportExportService;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyServiceListener;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.VradiConstants;
import com.jurismarches.vradi.VradiServiceConfiguration;
import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.beans.XmlStreamImportResult;
import com.jurismarches.vradi.entities.Client;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Group;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.QueryMaker;
import com.jurismarches.vradi.entities.Sending;
import com.jurismarches.vradi.entities.Session;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.entities.User;
import com.jurismarches.vradi.entities.VradiUser;
import com.jurismarches.vradi.entities.WebHarvestStream;
import com.jurismarches.vradi.entities.XmlStream;
import com.jurismarches.vradi.services.managers.BindingManager;
import com.jurismarches.vradi.services.managers.ClientManager;
import com.jurismarches.vradi.services.managers.FormManager;
import com.jurismarches.vradi.services.managers.FormTypeManager;
import com.jurismarches.vradi.services.managers.MailingManager;
import com.jurismarches.vradi.services.managers.SearchManager;
import com.jurismarches.vradi.services.managers.TemplateManager;
import com.jurismarches.vradi.services.managers.ThesaurusManager;
import com.jurismarches.vradi.services.tasks.TasksManager;

/**
 * Vradi storage implementation.
 * 
 * @author morin
 * @author chatellier
 */
public class VradiStorageServiceImpl implements VradiStorageService {

    static private final Log log = LogFactory.getLog(VradiStorageServiceImpl.class);

    /**
     * Proxy to store and retrieve data.
     */
    protected WikittyProxy wikittyProxy;

    protected WikittyImportExportService importExportService;
    
    protected ThesaurusManager thesaurusManager;
    protected BindingManager bindingManager;
    protected FormManager formManager;
    protected FormTypeManager formTypeManager;
    protected ClientManager clientManager;
    protected SearchManager searchManager;
    protected MailingManager mailingManager;
    protected TasksManager tasksManager;

    /**
     * Remote instantiation and empty constructor for hessian.
     */
    public VradiStorageServiceImpl() {
        this(ServiceFactory.getWikittyProxy());
    }

    /**
     * This constructor exists for local wikittyProxy instanciation.
     * 
     * @param wikittyProxy custom wikitty proxy
     */
    protected VradiStorageServiceImpl(WikittyProxy wikittyProxy) {
        this.wikittyProxy = wikittyProxy;

        thesaurusManager = new ThesaurusManager(wikittyProxy);
        clientManager = new ClientManager(wikittyProxy);
        formTypeManager = new FormTypeManager(wikittyProxy);
        mailingManager = new MailingManager(wikittyProxy);

        searchManager = new SearchManager(wikittyProxy, thesaurusManager);
        formManager = new FormManager(wikittyProxy, thesaurusManager, searchManager);
        bindingManager = new BindingManager(wikittyProxy, formTypeManager, formManager);
        tasksManager = new TasksManager(wikittyProxy, mailingManager, bindingManager);

        // init cron tasks
        tasksManager.initTasks();
        wikittyProxy.getWikittyService().addWikittyServiceListener(tasksManager, WikittyService.ServiceListenerType.ALL);
    }

    /**
     * Get {@link WikittyImportExportService} singl instance.
     * 
     * @return WikittyImportExportService singleton
     */
    protected WikittyImportExportService getImportExportService() {
        if (importExportService == null) {
            importExportService = new WikittyImportExportService(wikittyProxy.getSecurityToken(), wikittyProxy.getWikittyService());
        }
        return importExportService;
    }

    @Override
    public void deleteClient(String clientId) throws VradiException  {
        clientManager.deleteClient(clientId);
    }

    @Override
    public void deleteUser(String userId) throws VradiException  {
        clientManager.deleteUser(userId);
    }

    @Override
    public void deleteGroup(String groupId) throws VradiException {
        clientManager.deleteGroup(groupId);
    }

    @Override
    public List<Status> createDefaultStatuses() throws VradiException {
        return formManager.createDefaultStatuses();
    }
   
    @Override
    public Map<Group, List<QueryBean>> findQueriesReturningForm(Form form)
            throws VradiException {
        return searchManager.findQueriesReturningForm(form);
    }

    @Override
    public WikittyExtension getFormType(String name) throws VradiException {
        return formTypeManager.getFormType(name);
    }

    @Override
    public Map<String, FieldType> getFormTypeFields(String name) throws VradiException {
        return formTypeManager.getFormTypeFields(name);
    }

    @Override
    public Session createNewSession() throws VradiException {
        return formManager.createNewSession();
    }

    @Override
    public Session bindForms(Session session) throws VradiException {
        return formManager.bindForms(session);
    }

    @Override
    public List<Sending> createAllSending(Session session, Group group, List<Form> formsToBind) throws VradiException {
        return formManager.createAllSending(session, group, formsToBind, false);
    }

    @Override
    public Sending createSending(Session session, User user, List<Form> formsToBind) throws VradiException {
        return formManager.createUserSending(session, user, formsToBind, false);
    }

    @Override
    public List<Sending> removeAllSending(Session session, Form form, Group group)
            throws VradiException{
        return formManager.removeAllSending(session, form, group);
    }

    @Override
    public Sending removeSending(Session session, Form form, User user)
            throws VradiException{
        return formManager.removeSending(session, form, user);
    }

    @Override
    public List<Session> getSessions(Date sessionDate) throws VradiException {
        return formManager.getSessions(sessionDate);
    }

    @Override
    public Session getLastCloseSession() throws VradiException {
        return formManager.getLastCloseSession();
    }

    @Override
    public Session getLastOpenSession() throws VradiException {
        return formManager.getLastOpenSession();
    }

    @Override
    public String getWebHarvestPreviewURL(WebHarvestStream stream) throws VradiException {
        return bindingManager.getWebHarvestPreviewUrlAsString(stream);
    }

    /**
     * Creates and store forms from an {@code WebHarvestStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * Input xmlStream is modified by this method (wikitty obselete).
     * 
     * @param webHarvestStream stream to import
     * @return a structure containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    @Override
    public XmlStreamImportResult importFormsFromWebHarvestStream(WebHarvestStream webHarvestStream)
            throws VradiException {
        return bindingManager.importFormsFromWebHarvestStream(webHarvestStream);
    }

    /**
     * Creates and store forms from an {@code XmlStream} by using the XmlStreamBinding
     * to link xml stream field values with form fields.
     *
     * Input xmlStream is modified by this method (wikitty obselete).
     * 
     * @param xmlStream stream to import
     * @return a structure containing :
     * - the number of created forms
     * - the number of already existing forms
     * - the number of forms created with date parsing error
     * - the number of forms created with number parsing error
     *
     * @throws VradiException for various possible errors
     */
    @Override
    public XmlStreamImportResult importFormsFromXmlStream(XmlStream xmlStream)
            throws VradiException {
        return bindingManager.importFormsFromXmlStream(xmlStream);
    }

    @Override
    public VradiUser createUser(VradiUser user) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateVradiUser(" + user.getLogin() + ")");
        }
        
        Search search = Search.query().eq(VradiUser.FIELD_WIKITTYUSER_LOGIN, user.getLogin());
        Criteria criteria = search.criteria();
        criteria.setEndIndex(1); // one is enought
        
        PagedResult<VradiUser> users = wikittyProxy.findAllByCriteria(VradiUser.class, criteria);
        
        if (users.getNumFound() > 0) {
            if (log.isDebugEnabled()) {
                log.debug("User " + user.getLogin() + " already exists");
            }
            return null;
            
        } else {
            return wikittyProxy.store(user);
        }
    }

    @Override
    public VradiUser loginUser(String login, String mp5password) {
        if (log.isDebugEnabled()) {
            log.debug("logVradiUser(" + login + ", " + mp5password + ")");
        }
        
        Criteria criteria = Search.query()
                .eq(VradiUser.FQ_FIELD_WIKITTYUSER_LOGIN, login)
                .eq(VradiUser.FQ_FIELD_WIKITTYUSER_PASSWORD, mp5password)
                .criteria();
        criteria.setEndIndex(1); // max 1
        
        VradiUser vradiUser = wikittyProxy.findByCriteria(VradiUser.class, criteria);
        return vradiUser;
    }

    /*@Override
    public void importData(File file) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("importData(file)");
        }

        // TODO: create an archive of vradi.home directory before

        try {
            String toURI = file.toURI().toString();
            log.info("importing file: " + toURI);
            
            WikittyImportExportService importExportService = getImportExportService();
            importExportService.syncImportFromUri(WikittyImportExportService.FORMAT.CSV, toURI);
    
            log.info("import done");
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't import data", e);
            }
            throw new VradiException("Can't import data", e);
        }
    }

    @Override
    public String exportData() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("exportData()");
        }

        try {
            Search search = Search.query().keyword("*");
            Criteria criteria = search.criteria();

            WikittyImportExportService importExportService = getImportExportService();
            String export = importExportService.syncExportAllByCriteria(WikittyImportExportService.FORMAT.CSV, criteria);
            log.info("export done");
            return export;

        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't export data", e);
            }
            throw new VradiException("Can't export data", e);
        }
    }*/

    @Override
    public void importAsCSV(String uri) throws VradiException {

        if (log.isDebugEnabled()) {
            log.debug("Importing data as CSV with url = " + uri);
        }

        try {
            WikittyImportExportService ieService = getImportExportService();
            ieService.syncImportFromUri(WikittyImportExportService.FORMAT.CSV, uri);
        }
        catch (WikittyException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't import data", eee);
            }
            throw new VradiException("Can't import data", eee);
        }
    }

    @Override
    public String exportAsCSV(Criteria criteria) throws VradiException {

        //FIXME JC 01/09/2010 Should create a file and make him downloadable through FileService.

        if (log.isDebugEnabled()) {
            log.debug("Exporting data as CSV with criteria " + criteria);
        }

        String result = null;
        try {
            WikittyImportExportService ieService = getImportExportService();
            result = ieService.syncExportAllByCriteria(WikittyImportExportService.FORMAT.CSV, criteria);
        }
        catch (WikittyException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't export data", eee);
            }
        }
        return result;
    }

    @Override
    public void reindexData() {
        if (log.isDebugEnabled()) {
            log.debug("reindexData()");
        }
        
        UpdateResponse response = wikittyProxy.getWikittyService().syncEngin(null);
        if (log.isDebugEnabled()) {
            log.debug(response.toString());
        }
    }

    public List<Thesaurus> proposeThesaurus(Form form) throws VradiException {
        return thesaurusManager.proposeThesaurus(form);
    }

    @Override
    public void archiveQueries(QueryMaker queryMaker)
            throws VradiException {
        clientManager.archiveQueries(queryMaker);
    }
    
    @Override
    public Collection<String> getTemplateFilenames(WikittyExtension extension) {
        return formTypeManager.getTemplateFilenames(extension);
    }

    @Override
    public void setAssociatedFields(String extensionName,
                                           String templateName,
                                    Map<String, String> fieldMap)
            throws VradiException {

        formTypeManager.setAssociatedFields(extensionName, templateName, fieldMap);

    }

    @Override
    public String generatePDF(Form form, boolean force) throws VradiException {

        String uri = null;

        String extensionName = null;
        for(String ext : form.getExtensionNames()) {
            if(!ext.equals(Infogene.EXT_INFOGENE)
                    && !ext.equals(Form.EXT_FORM)) {
                extensionName = ext;
                break;
            }

        }
        WikittyExtension extension = formTypeManager.getFormType(extensionName);
        // VradiHelper.getFormTypeTemplate(extension);
        String template = extension.getTagValue(FormTypeManager.TYPE_TEMPLATE);
        File templateFile = formTypeManager.getTemplate(extensionName, template);
        Map<String, String> fieldMap = getAssociatedFields(extension.getName(), template);

        if (log.isDebugEnabled()) {
            log.debug("Generating PDF file for form " + form.getWikittyId() +
                    " (template = " + template + ")");
        }

        try {
            // but only those associated with current templates !
            List<Form> currentForms = Collections.singletonList(form);
            List<File> filesToAdd = generateFilledDocumentInPDF(
                            templateFile, currentForms, fieldMap, force);
            
            uri = filesToAdd.get(0).getName();
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't generate pdf", ex);
            }
            throw new VradiException("Can't generate pdf", ex);
        }

        return uri;
    }

    @Override
    public Map<String, String> getAssociatedFields(String extensionName,
                                                   String templateName)
            throws VradiException {
        return formTypeManager.getAssociatedFields(extensionName, templateName);
    }


    @Override
    public Map<Group, List<QueryBean>> getQueriesToModifyAfterThesaurusModification(
            String rootThesaurusName, String thesaurusName) {
        return thesaurusManager.getQueriesToModifyAfterThesaurusModification(rootThesaurusName, thesaurusName);
    }

    @Override
    public List<String> getDocumentFields(WikittyExtension extension, String templateName) throws VradiException {

        List<String> documentsFields = null;

        try {
            File templateDir = VradiServiceConfiguration.getInstance().getTemplatesDir();
            File extensionTemplate = new File(templateDir, extension.getName());
            File templateFile = new File(extensionTemplate, templateName);
    
            TemplateManager templateManager = new TemplateManager(templateFile);
            documentsFields = templateManager.getDocumentFields();
        }
        catch (VradiException ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't get documents fields", ex);
            }
            throw ex;
        }
        return documentsFields;
    }

    /**
     * Generate offer PDF form given forms with specified template.
     * 
     * @param template template to use to generate PDF
     * @param forms to to generate
     * @param fieldBindings map between template field name and form fields names
     * @param replace replace already existing generated forms
     * @return generated pdf files
     * @throws VradiException
     */
    protected List<File> generateFilledDocumentInPDF(File template, List<Form> forms,
                                            Map<String, String> fieldBindings,
                                            boolean replace)
            throws VradiException {

        List<File> generatedPDFList = null;
        
        if(template == null) {
            if (log.isWarnEnabled()) {
                log.warn("Try to generate form pdf without template");
            }
        }
        else {
            TemplateManager templateManager = new TemplateManager(template);
            
            // split fqn form field name in an array
            // composed of extension name, and extension field
            // TODO EC20100510 set in commun with same function in
            // {@link generateFilledDocumentInPDF(TemplateManager, Form form, Map<String, String[]>, boolean)
            Map<String, String[]> map = new HashMap<String, String[]>();
            for(Map.Entry<String, String> binding : fieldBindings.entrySet()) {
                String fieldFQName = binding.getValue();
                if(fieldFQName != null && fieldFQName.indexOf(".") > 0) {
                    int dot = fieldFQName.indexOf(".");
                    String[] fqField = new String[2];
                    fqField[0] = fieldFQName.substring(0, dot);
                    fqField[1] = fieldFQName.substring(dot + 1);
                    map.put((String)binding.getKey(), fqField);
                }
            }

            generatedPDFList = new ArrayList<File>();
            for (Form form : forms) {
                File f = generateFilledDocumentInPDF(templateManager, form, map, replace);
                generatedPDFList.add(f);
            }
        }

        return generatedPDFList;
    }
    
    /**
     * Generate a single pdf file for specified form.
     * 
     * @param templateManager pdf template manager
     * @param form form to get pdf
     * @param fieldBindings association between field name and form fields (couple extension name, extension field)
     * @param replace replace already existing pdf
     * @return the pdf file associated with pdf
     * @throws VradiException
     */
    protected File generateFilledDocumentInPDF(TemplateManager templateManager,
            Form form, Map<String, String[]> fieldBindings,
            boolean replace) throws VradiException {

        File result = mailingManager.getGeneratedPdfFile(form);
        if (!replace && result.exists()) {
            return result;
        }

        Map<String, Object> fieldValues = new HashMap<String, Object>();
        for(Map.Entry<String, String[]> binding : fieldBindings.entrySet()) {
            String[] fqField = binding.getValue();
            Object value = form.getField(fqField[0], fqField[1]);
            fieldValues.put((String)binding.getKey(),
            value != null ? value.toString() : null);
        }
        List<File> attachments = new ArrayList<File>();
        if(form.getFiles() != null) {
            File embeddedFilesDir = VradiServiceConfiguration.getInstance().getEmbeddedFilesDir();
            File formFilesDir = new File(embeddedFilesDir, form.getWikittyId());
            
            for(String file : form.getFiles()) {
                File f = new File(formFilesDir, file);
                attachments.add(f);
            }
        }
        templateManager.generateDoc(result.getPath(), fieldValues,
                attachments.toArray(new File[attachments.size()]));
        
        return result;
    }

    @Override
    public Session sendMessages(String sessionId) throws VradiException {

        Session session = wikittyProxy.restore(Session.class, sessionId);

        if (session.getStatus() == VradiConstants.SessionStatus.SENT.getValue()) {
            throw new VradiException("Session is already sent, abording");
        }

        boolean resumeSession = false;
        if (session.getStatus() == VradiConstants.SessionStatus.STOPPED.getValue()) {
            log.info("Session was stopped, resume sending");
            session.setStatus(VradiConstants.SessionStatus.ACTIVE.getValue());
            session = wikittyProxy.store(session);
            resumeSession = true;
        }

        for (String sendingId : session.getSending()) {

            // Flag to detect if it's canceled by user
            session = wikittyProxy.restore(Session.class, session.getWikittyId());
            if (session.getStatus() == VradiConstants.SessionStatus.STOPPED.getValue()) {
                log.warn("Sending stopped by user");
                return session;
            }

            Sending sending = wikittyProxy.restore(Sending.class, sendingId);
            if (resumeSession && sending.getStatus() != VradiConstants.SendingStatus.TO_SEND.getValue()) {
                if (log.isDebugEnabled()) {
                    log.debug("Sending " + sendingId + " already sent, skip");
                }
            }
            else {
                // Send
                sendMessage(sendingId, session.getParagraph());
            }
        }

        // Set session to sent
        session.setStatus(VradiConstants.SessionStatus.SENT.getValue());
        session.setSessionDate(new Date());
        session = wikittyProxy.store(session);

        return session;
    }

    @Override
    public Sending sendMessage(String sendingId, String sessionParagraph) throws VradiException {
        return sendMessage(sendingId, sessionParagraph, null);
    }

    @Override
    public Sending sendMessage(String sendingId, String sessionParagraph, String email) throws VradiException {

        Sending sending = wikittyProxy.restore(Sending.class, sendingId);

        // Throw sending in error (email or pdf not found) or deleted
        if (sending.getStatus() == VradiConstants.SendingStatus.DELETED.getValue()){
            return sending;
        }

        // If its client or group sending : do nothing
        if (sending.getUser() == null) {
            sending.setStatus(VradiConstants.SendingStatus.SENT.getValue());
            return wikittyProxy.store(sending);
        }

        // Get email address
        // si aucune n'est specifiée, on utilise celle du client
        if (StringUtils.isBlank(email)) {
            User user = clientManager.getUser(sending.getUser());
            email = user.getEmail();
            
            // il peut arriver qu'un user n'est pas d'email et que l'admin
            // souhaite quand meme faire l'envoie d'une session.
            // dans ce cas on ne traite pas l'envoi de mail courant.
            if (StringUtils.isBlank(email)) {
                if (log.isWarnEnabled()) {
                    log.warn("User " + user.getName() + " has not valid email (skip current mail sending)");
                }
                return sending;
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Sending message to " + email);
        }

        String subject = VradiServiceConfiguration.getInstance().getOfferMailSubject();

        // Set message
        String message = sessionParagraph + "\n\n" + sending.getParagraph();

        Set<String> formsIds = sending.getForm();

        // Do nothing
        if (formsIds == null) {
            return sending;
        }
        List<Form> forms = wikittyProxy.restore(Form.class, new ArrayList<String>(formsIds));

        boolean receptionProof = sending.getReceptionProof();

        // Post mail and save msg id
        String messageId = null;
        try {
            messageId = mailingManager.postMail(
                email, subject, message, forms, receptionProof);
        } catch (VradiException eee) {
            // Set sending in error
            sending.setStatus(VradiConstants.SendingStatus.ERROR.getValue());
            wikittyProxy.store(sending);
            throw eee;
        }
        sending.setMessageId(messageId);

        // Set sending status
        sending.setStatus(receptionProof ?
                VradiConstants.SendingStatus.WAITING_RECEPTION_PROOF.getValue() :
                VradiConstants.SendingStatus.SENT.getValue());

        // Update sending
        sending = wikittyProxy.store(sending);

        return sending;
    }

    @Override
    public void receiveMessages() throws VradiException {
        mailingManager.receiveMails();
    }

    @Override
    public Session stopSentMail(Session session) throws VradiException {

        // Flag status to stop traitement
        session.setStatus(VradiConstants.SessionStatus.STOPPED.getValue());

        // Update
        session = wikittyProxy.store(session);

        return session;
    }

    @Override
    public List<String> getAllWebHarvestScripts() {
        File scriptDir = VradiServiceConfiguration.getInstance().getWebHarvestScriptDir();

        List<String> files = new ArrayList<String>();
        for (File file : scriptDir.listFiles()) {
            if (file.isFile()) {
                files.add(file.getName());
            }
        }

        return files;
    }

    /**
     * Generate all the PDFs needed to send a session.
     *
     * @param sessionId the session wikitty id
     */
    @Override
    public void generatePDFForSession(String sessionId){
        List<Form> formsToGenerate = new ArrayList<Form>();

        Session session = wikittyProxy.restore(Session.class, sessionId);

        Set<String> sendingIds = session.getSending();

        for (String id : sendingIds) {
            Sending sending = wikittyProxy.restore(Sending.class, id);

            Set<String> formsIds = sending.getForm();

            if (formsIds != null) {
                for (String formId : formsIds) {
                    Form form = wikittyProxy.restore(Form.class, formId);

                    if (!formsToGenerate.contains(form)) {
                        formsToGenerate.add(form);
                    }
                }
            }
        }

        for (Form form : formsToGenerate) {
            try {
                generatePDF(form, true);
            } catch (VradiException eee) {
                log.error("An error occurred while generating PDF file for " +
                        "form : " + form.getWikittyId(), eee);
            }
        }
    }

    /**
     * Get all the groups a user is in.
     * @param userId the user we want the groups of
     * @return the group for our user
     * @throws VradiException
     */
    public List<Group> getGroupsForUser(String userId) throws VradiException {
        return clientManager.getGroupsForUser(userId);
    }

    @Override
    public String exportClientDB() throws VradiException {

        // creation du criteria wikitty
        // (export de tous les wikitty avec extension "client"
        // "groupement" ou "user"
        Search restriction = Search.query().or(); //definit un "groupement" qui se fera en "or"
        restriction.eq(org.nuiton.wikitty.search.Element.ELT_EXTENSION, Group.EXT_GROUP);
        restriction.eq(org.nuiton.wikitty.search.Element.ELT_EXTENSION, Client.EXT_CLIENT);
        restriction.eq(org.nuiton.wikitty.search.Element.ELT_EXTENSION, User.EXT_USER);
        Criteria criteria = restriction.criteria();

        // export
        String vscContent = exportAsCSV(criteria);

        return vscContent;
    }

    /**
     * Get the status passed in parameter. Init status if it have not been done
     * already
     *
     * @param formStatus the status from enum
     * @return the Status object
     * @throws VradiException if an error occur during statuses initialisation
     */
    @Override
    public Status getStatus(VradiConstants.FormStatus formStatus) throws VradiException {

        List<Status> statuses = createDefaultStatuses();

        for (Status status : statuses) {
            if (status.getName().equals(formStatus.getName())) {
                return status;
            }
        }
        return null;
    }

    @Override
    public Session getSessionOfSending(String sendingId) throws VradiException {
        return formManager.getSessionOfSending(sendingId);
    }

    @Override
    public void deleteSession(String sessionId) throws VradiException {
        formManager.deleteSession(sessionId);
    }

    @Override
    public boolean deleteStatus(List<String> statusIds) throws VradiException {
        return formManager.deleteStatuses(statusIds);
    }

    @Override
    public boolean checkTemplateExist(String extensionName) throws VradiException {
        boolean fileExists;

        try {
            WikittyExtension extension = getFormType(extensionName);

            String templateName = formTypeManager.getFormTypeTemplateName(extension);

            if (templateName == null) {
                return false;
            }

            // just verify file existence
            // File file = storageService.getTemplate(extensionName, template);
            File template = formTypeManager.getTemplate(extensionName, templateName);

            fileExists = (template != null && template.exists()); 
        } catch (Exception eee) {
            throw new VradiException("Can't check if template existe : ", eee);
        }
        return fileExists;
    }

    @Override
    public boolean removeFormsFromSession(List<Form> forms) throws VradiException {
        return formManager.removeFormsFromSession(forms);
    }
}
