/*
 * *##%
 * Vradi :: Services
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */
package com.jurismarches.vradi.services.managers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.jurismarches.vradi.entities.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.Criteria;
import org.nuiton.wikitty.FacetTopic;
import org.nuiton.wikitty.PagedResult;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.search.Element;
import org.nuiton.wikitty.search.Like;
import org.nuiton.wikitty.search.Search;

import com.jurismarches.vradi.beans.QueryBean;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.search.CompareFilter;
import com.jurismarches.vradi.services.search.Filter;
import com.jurismarches.vradi.services.search.FilterList;
import com.jurismarches.vradi.services.search.VradiQueryParser;

/**
 * Class containing the methods to manage the thesaurus :
 * - node creation, update, retrieving, deletion
 * - get child node number
 * - get children
 *
 * TODO SLE 27/07/10 : Do for descripteurs
 *
 * @author schorlet
 * @version $Revision: 1184 $ $Date: 2010-08-05 15:20:36 +0200 (jeu., 05 août 2010) $
 * @date 2010-01-22 20:18:29
 */
public class ThesaurusManager {
    
    /** log. */
    private static final Log log = LogFactory.getLog(ThesaurusManager.class);

    protected final WikittyProxy proxy;
    
    public ThesaurusManager(WikittyProxy proxy) {
        this.proxy = proxy;
    }

    /**
     * Return root thesaurus (ie thesaurus nodes without parent).
     * 
     * @return a {@link RootThesaurus} collection
     * @throws VradiException
     */
    public List<RootThesaurus> getRootThesaurus() throws VradiException {

        Criteria criteria = Search.query()
                .eq(Element.ELT_EXTENSION, RootThesaurusImpl.EXT_ROOTTHESAURUS)
                .criteria();

        PagedResult<RootThesaurus> rootThesaurusResult = proxy.findAllByCriteria(RootThesaurus.class, criteria);
        List<RootThesaurus> rootThesaurus = rootThesaurusResult.getAll();

        if (log.isInfoEnabled()) {
            log.info("Root thesaurus found : " + rootThesaurus.size());
            if (log.isDebugEnabled()) {
                log.debug("Root thesaurus list : " + rootThesaurus);
            }
        }
        return rootThesaurus;
    }

    public List<Thesaurus> getAllThesaurus() throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getAllThesaurus()");
        }

        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION,
                Thesaurus.EXT_THESAURUS).criteria();

        PagedResult<Thesaurus> findAllByCriteria = proxy.findAllByCriteria(
                Thesaurus.class, criteria);
        List<Thesaurus> nodes = findAllByCriteria.getAll();
        
        log.info(String.format("found %s thesaurus entries", nodes.size()));
        return nodes;
    }

    public Thesaurus getThesaurus(String thesaurusId) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus(" + thesaurusId + ")");
        }
        Thesaurus node = proxy.restore(Thesaurus.class, thesaurusId);
        return node;
    }

    public List<Thesaurus> getThesaurus(List<String> thesaurusIds)
            throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus(" + thesaurusIds + ")");
        }
        List<Thesaurus> nodes = proxy.restore(Thesaurus.class, thesaurusIds);

        return nodes;
    }

    public List<Thesaurus> getChildrenThesaurus(String thesaurusId)
            throws VradiException {
        if (log.isTraceEnabled()) {
            log.trace("getChildrenThesaurus(" + thesaurusId + ")");
        }

        Search query = Search.query();
        query.eq(Element.ELT_EXTENSION, Thesaurus.EXT_THESAURUS);
        query.eq(Thesaurus.FQ_FIELD_TREENODE_PARENT, thesaurusId);

        Criteria criteria = query.criteria();
        PagedResult<Thesaurus> nodes = proxy
                .findAllByCriteria(Thesaurus.class, criteria);
        List<Thesaurus> all = nodes.getAll();

        return all;
    }

    public int getNbFormsForThesaurus(String thesaurusId)
            throws VradiException {
        if (log.isTraceEnabled()) {
            log.trace("getNbFormsForThesaurus(" + thesaurusId + ")");
        }

        Map.Entry<Thesaurus, Integer> entry =
                proxy.restoreNode(Thesaurus.class, thesaurusId, null);

        if (entry == null) {
            return 0;
        }

        return entry.getValue();
    }
    
    /**
     * Propose thesaurus nodes that might be in relation with a specified form
     *
     * @param form      the <code>Form</code> containing the information needed
     *                  to search the thesaurus nodes
     * @return a list of <code>Thesaurus</code>
     * @throws VradiException
     */
    public List<Thesaurus> proposeThesaurus(Form form) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("proposeThesaurus(form)");
        }
        
        if (form == null) {
            return null;
        }
        
        // search for the specified form
        Criteria criteria = Search.query()
            .eq(Element.ELT_EXTENSION, Form.EXT_FORM)
            .eq(Form.FQ_FIELD_INFOGENE_ID, form.getId())
            .criteria();

        // search within the specified thesaurus nodes or all
        List<Thesaurus> nodes = getAllThesaurus();

        Set<String> formNodes = form.getThesaurus();
        
        // add a facet criteria for each thesaurus node id,
        // searching for tags of each thesaurus nodes
        for (Thesaurus node : nodes) {
            if (formNodes != null && formNodes.contains(node.getWikittyId())) {
                continue;
            }

            Set<String> tags = node.getTags();

            if (tags == null || tags.isEmpty()) {
                continue;
            }
            
            Search tagSearch = Search.query(Search.KIND.OR);
            for (String tag : tags) {
                if (!tag.isEmpty()) {
                    tagSearch.keyword(tag);
                }
            }

            Criteria facetCriteria = tagSearch.criteria(node.getWikittyId());
            criteria.addFacetCriteria(facetCriteria);
        }

        // execute the search
        PagedResult<Form> forms = proxy.findAllByCriteria(Form.class, criteria);
        List<Thesaurus> result = new ArrayList<Thesaurus>();
        
        // collects topic names (which are thesaurus node ids)
        if (forms != null && forms.getNumFound() > 0) {
            List<String> thesaurusIds = new ArrayList<String>();
            
            Map<String, List<FacetTopic>> facetsMap = forms.getFacets();
            for (Map.Entry<String, List<FacetTopic>> entry : facetsMap.entrySet()) {
                List<FacetTopic> facetTopics = entry.getValue();
                for (FacetTopic facetTopic : facetTopics) {
                    if (facetTopic.getCount() > 0) {
                        thesaurusIds.add(facetTopic.getTopicName());
                        break;
                    }
                }
            }
            
            if (!thesaurusIds.isEmpty()) {
                result = getThesaurus(thesaurusIds);
            }
        }
        
        return result;
    }

    /**
     * Gets the query makers whose queries are potentially to modify
     * after a thesaurus node modification
     *
     * @param thesaurusName the modified thesaurus node
     * @return a map containing the query makers and their queries which contains
     *         the thesaurus node name
     */
    public Map<Group, List<QueryBean>> getQueriesToModifyAfterThesaurusModification(
            String rootThesaurusName, String thesaurusName) {
        if (log.isDebugEnabled()) {
            log.debug("getQueriesToModifyAfterThesaurusModification(" + rootThesaurusName + ", " + thesaurusName + ")");
        }
        Map<Group, List<QueryBean>> results = new HashMap<Group, List<QueryBean>>();
        
        if (rootThesaurusName == null) {
            return results;
        }

        String requestPart = rootThesaurusName + ":" + (thesaurusName == null ? "" : thesaurusName);
        Criteria criteria = Search.query()
            .eq(Element.ELT_EXTENSION, Group.EXT_GROUP)
            .like(Group.FQ_FIELD_QUERYMAKER_QUERIES, requestPart, Like.SearchAs.AsText)
            .criteria();
        
        PagedResult<Group> pagedResult = proxy.findAllByCriteria(Group.class, criteria);
        List<Group> groups = pagedResult.getAll();

        if (log.isDebugEnabled()) {
            log.debug("[getQueriesToModifyAfterThesaurusModification]  " + groups.size() +
                    "Groups found for request part requestPart : " + requestPart);
        }

        for (Group group : groups) {
            
            Set<String> queries = group.getQueries();
            List<QueryBean> queriesToModify = new ArrayList<QueryBean>();
            
            for (String query : queries) {
                try {
                    QueryBean queryBean = new QueryBean(query, group.getWikittyId());
                    FilterList filter = VradiQueryParser.parse(queryBean.getQuery());
                    
                    if (isThesaurusInQuery(filter, rootThesaurusName, thesaurusName)) {
                        queriesToModify.add(queryBean);
                    }
                    
                } catch (Exception e) {
                    // ignored exception
                    log.warn(e.getMessage(), e);
                }
            }
            
            if (!queriesToModify.isEmpty()) {
                results.put(group, queriesToModify);
            }
        }

        return results;
    }

    boolean isThesaurusInQuery(FilterList list, String rootThesaurusName, String thesaurusName) {
        boolean insideQuery = false;
        List<Filter> filters = list.getFilters();
        
        for (Filter filter : filters) {
            if (filter instanceof FilterList) {
                insideQuery = isThesaurusInQuery((FilterList) filter, rootThesaurusName, thesaurusName);

            } else if (filter instanceof CompareFilter) {
                insideQuery = isThesaurusInQuery((CompareFilter) filter, rootThesaurusName, thesaurusName);
            }
            
            if (insideQuery) {
                break;
            }
        }
        return insideQuery;
    }

    boolean isThesaurusInQuery(CompareFilter compareFilter,
                               String rootThesaurusName, String thesaurusName) {
        String name = compareFilter.getName();
        String value = compareFilter.getValue();
        boolean result = rootThesaurusName.equals(name)
                && (thesaurusName == null || thesaurusName.equals(value));

        if (log.isDebugEnabled()) {
            log.debug("[isThesaurusInQuery] Root Thesaurus name : " + rootThesaurusName +
                    " name : " + name + " ThesaurusName : " +
                    thesaurusName + " value : " + value + " result : " + result);
        }
        return result;
    }

    void replaceThesaurusInQuery(FilterList list,
            String oldRootThesaurusName, String newRootThesaurusName,
            String oldThesaurusName, String newThesaurusName) {

        List<Filter> filters = list.getFilters();
        
        for (Filter filter : filters) {
            if (filter instanceof FilterList) {
                replaceThesaurusInQuery((FilterList) filter,
                        oldRootThesaurusName, newRootThesaurusName,
                        oldThesaurusName, newThesaurusName);

            } else if (filter instanceof CompareFilter) {
                replaceThesaurusInQuery((CompareFilter) filter,
                        oldRootThesaurusName, newRootThesaurusName,
                        oldThesaurusName, newThesaurusName);
            }
        }
    }

    void replaceThesaurusInQuery(CompareFilter compareFilter,
            String oldRootThesaurusName, String newRootThesaurusName,
            String oldThesaurusName, String newThesaurusName) {
        String value = compareFilter.getValue();
        
        if (value.equals(oldThesaurusName)) {
            String name = compareFilter.getName();

            if (oldRootThesaurusName.equals(name)) {

                compareFilter.setName(newRootThesaurusName);
                compareFilter.setValue(newThesaurusName);
            }
        }
    }

}
