package com.franciaflex.faxtomail.ui.swing.content.attachment;

/*
 * #%L
 * FaxToMail :: UI
 * $Id: AttachmentEditorUIHandler.java 128 2014-06-03 15:42:28Z echatellier $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.1/faxtomail-ui-swing/src/main/java/com/franciaflex/faxtomail/ui/swing/content/attachment/AttachmentEditorUIHandler.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentImpl;
import com.franciaflex.faxtomail.ui.swing.content.MainUI;
import com.franciaflex.faxtomail.ui.swing.content.pdfeditor.PDFEditorUI;
import com.franciaflex.faxtomail.ui.swing.util.AbstractToolbarPopupHandler;
import com.franciaflex.faxtomail.ui.swing.util.FaxToMailUIUtil;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;

import org.apache.commons.collections4.CollectionUtils;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.nuiton.util.DesktopUtil;
import org.nuiton.util.FileUtil;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.Desktop;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import javax.swing.AbstractCellEditor;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JTable;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;

import static org.nuiton.i18n.I18n.t;

/**
 * @author kmorin <morin@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class AttachmentEditorUIHandler extends AbstractToolbarPopupHandler<AttachmentEditorUIModel, AttachmentEditorUI> {

    public static final List<String> EDITABLE_EXTENSIONS = Lists.newArrayList("PDF", "PNG", "JPG", "JPEG", "GIF",
                                                                              "TIF", "BMP", "TXT");

    @Override
    public void afterInit(AttachmentEditorUI ui) {

        ui.getFile().setDialogOwner(ui);
        super.afterInit(ui);

        initTable(ui.getAttachments());

        getModel().addPropertyChangeListener(AttachmentModelAware.PROPERTY_ATTACHMENT, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                AbstractTableModel tableModel = (AbstractTableModel) getUI().getAttachments().getModel();
                tableModel.fireTableDataChanged();
            }
        });
    }

    protected void initTable(JXTable table) {

        TableModel attachmentTableModel = new AbstractTableModel() {
            @Override
            public int getRowCount() {
                return CollectionUtils.size(getModel().getAttachment());
            }

            @Override
            public int getColumnCount() {
                return 1;
            }

            @Override
            public Object getValueAt(int rowIndex, int columnIndex) {
                return Iterables.get(getModel().getAttachment(), rowIndex);
            }

            @Override
            public boolean isCellEditable(int rowIndex, int columnIndex) {
                return true;
            }
        };

        attachmentTableModel.addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                getUI().pack();
            }
        });

        TableColumnModel columnModel = new DefaultTableColumnModel();
        TableColumn column = new TableColumn();
        column.setCellRenderer(new AttachmentItemRenderer());
        column.setCellEditor(new AttachmentItemEditor());
        columnModel.addColumn(column);

        table.setModel(attachmentTableModel);
        table.setColumnModel(columnModel);
        table.addHighlighter(HighlighterFactory.createAlternateStriping());
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getFile();
    }

    @Override
    public void onCloseUI() {
    }

    @Override
    public SwingValidator<AttachmentEditorUIModel> getValidator() {
        return null;
    }

    public void addAttachment() {
        Attachment attachment = new AttachmentImpl();
        attachment.setAddedByUser(true);
        File file = ui.getFile().getSelectedFile();
        if (file != null) {
            try {
                // warning, this not NOT be closed (will be by blob on commit)
                InputStream is = new FileInputStream(file);
                AttachmentFile attachmentFile = getContext().getEmailService().getAttachmentFileFromStream(is);
                attachmentFile.setFilename(file.getName());
                attachment.setOriginalFile(attachmentFile);
            } catch (Exception e) {
                getContext().getErrorHelper().showErrorDialog(t("swing.error.cannot.copy.file"), e);
            }
            getModel().addAttachment(attachment);
            resetFields();
        }
    }

    protected void resetFields() {
        ui.getFile().setSelectedFilePath(null);
    }

    public void openAttachment(Attachment attachment) {

        getModel().fireAttachmentOpened(attachment, true);
        File file = attachment.getOriginalFile().getFile();

        Desktop desktop = FaxToMailUIUtil.getDesktopForOpen();
        try {
            desktop.open(file);
        } catch (IOException e) {
            getContext().getErrorHelper().showErrorDialog(t("swing.error.cannot.open.file"));
        }
    }

    public void editAttachment(Attachment attachment) {
        AttachmentEditorUIModel model = getModel();

        AttachmentFile attachmentFile = attachment.getEditedFile();
        if (attachmentFile == null) {
            model.fireAttachmentOpened(attachment, true);
            attachmentFile = attachment.getOriginalFile();

        } else {
            model.fireAttachmentOpened(attachment, false);
        }
        File file = attachmentFile.getFile();

        if (model.isEditable()) {
            String extension = FileUtil.extension(attachmentFile.getFilename());
            if (model.isEditable() && EDITABLE_EXTENSIONS.contains(extension.toUpperCase())) {

                MainUI mainUI = getContext().getMainUI();
                mainUI.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                PDFEditorUI dialogContent = new PDFEditorUI(ui);
                dialogContent.getModel().fromEntity(attachment);
                mainUI.setCursor(Cursor.getDefaultCursor());
                openDialog(dialogContent, file.getName(), mainUI.getSize());
                dialogContent.getModel().toEntity(attachment);

            } else {
                Desktop desktop = FaxToMailUIUtil.getDesktopForOpen();
                try {
                    desktop.open(file);
                } catch (IOException e) {
                    getContext().getErrorHelper().showErrorDialog(t("swing.error.cannot.open.file"));
                }
            }

        } else {
            Desktop desktop = FaxToMailUIUtil.getDesktopForBrowse();
            try {
                desktop.open(file);
            } catch (IOException e) {
                getContext().getErrorHelper().showErrorDialog(t("swing.error.cannot.open.file"));
            }
        }
    }

    public void removeAttachment(AttachmentItem attachmentItem) {
        Attachment attachment = attachmentItem.getAttachment();
        boolean hackDialog = ui.isAlwaysOnTop();
        if (hackDialog) {
            ui.setAlwaysOnTop(false);
        }
        int answer = JOptionPane.showConfirmDialog(ui,
                                                   t("faxtomail.attachmentEditor.deleteAttachment.message",
                                                     attachment.getOriginalFile().getFilename()),
                                                   t("faxtomail.attachmentEditor.deleteAttachment.title"),
                                                   JOptionPane.YES_NO_OPTION
        );
        if (hackDialog) {
            ui.setAlwaysOnTop(true);
        }

        if (answer == JOptionPane.YES_OPTION) {

            ui.getAttachments().getCellEditor().stopCellEditing();
//            getPersistenceService().deleteAttachment(attachment.getId());
            getModel().removeAttachment(attachment);

            ui.pack();
        }
    }

    public boolean isAttachmentEditable(Attachment attachment) {
        String extension = FileUtil.extension(attachment.getOriginalFile().getFilename());
        return EDITABLE_EXTENSIONS.contains(extension.toUpperCase());
    }

    public boolean isAttachmentRemovable(Attachment attachment) {
        return attachment != null && attachment.getTopiaId() == null;
    }

    protected class AttachmentItemRenderer extends AttachmentItem implements TableCellRenderer {

        public AttachmentItemRenderer() {
            setHandler(AttachmentEditorUIHandler.this);
        }

        public Component getTableCellRendererComponent(JTable table, Object value,
                                                       boolean isSelected, boolean hasFocus, int row, int column) {
            Attachment attachment = (Attachment) value;
            setAttachment(attachment);

            if (attachment != null) {
                JLabel label = getAttachmentNameLabel();
                String actionIconName = Boolean.TRUE.equals(attachment.getAddedByUser()) ? "user" : "email";
                Icon icon = SwingUtil.createActionIcon(actionIconName);
                label.setIcon(icon);
            }

            setEditable(getModel().isEditable());

            return this;
        }
    }

    protected class AttachmentItemEditor extends AbstractCellEditor implements TableCellEditor {

        protected AttachmentItem attachmentItem;

        public AttachmentItemEditor() {
            attachmentItem = new AttachmentItem();
            attachmentItem.setHandler(AttachmentEditorUIHandler.this);
        }

        public Component getTableCellEditorComponent(JTable table, Object value,
                                                     boolean isSelected, int row, int column) {
            Attachment attachment = (Attachment) value;
            attachmentItem.setAttachment(attachment);

            if (attachment != null) {
                JLabel label = attachmentItem.getAttachmentNameLabel();
                String actionIconName = Boolean.TRUE.equals(attachment.getAddedByUser()) ? "user" : "email";
                Icon icon = SwingUtil.createActionIcon(actionIconName);
                label.setIcon(icon);
            }

            attachmentItem.setEditable(getModel().isEditable());

            return attachmentItem;
        }

        @Override
        public Object getCellEditorValue() {
            return attachmentItem.getAttachment();
        }

    }

}
