/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.LatencyTracker;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.exceptions.BootstrappingException;
import com.datastax.driver.core.exceptions.DriverInternalError;
import com.datastax.driver.core.exceptions.OverloadedException;
import com.datastax.driver.core.exceptions.QueryValidationException;
import com.datastax.driver.core.exceptions.UnavailableException;
import com.datastax.driver.core.exceptions.UnpreparedException;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.HdrHistogram.Histogram;
import org.HdrHistogram.Recorder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class PercentileTracker
implements LatencyTracker {
    private static final Logger logger = LoggerFactory.getLogger(PercentileTracker.class);
    private final long highestTrackableLatencyMillis;
    private final int numberOfSignificantValueDigits;
    private final int minRecordedValues;
    private final long intervalMs;
    private final ConcurrentMap<Object, Recorder> recorders;
    private final ConcurrentMap<Object, CachedHistogram> cachedHistograms;
    private static final Set<Class<? extends Exception>> EXCLUDED_EXCEPTIONS = ImmutableSet.of(UnavailableException.class, OverloadedException.class, BootstrappingException.class, UnpreparedException.class, QueryValidationException.class);

    protected PercentileTracker(long highestTrackableLatencyMillis, int numberOfSignificantValueDigits, int minRecordedValues, long intervalMs) {
        this.highestTrackableLatencyMillis = highestTrackableLatencyMillis;
        this.numberOfSignificantValueDigits = numberOfSignificantValueDigits;
        this.minRecordedValues = minRecordedValues;
        this.intervalMs = intervalMs;
        this.recorders = new ConcurrentHashMap<Object, Recorder>();
        this.cachedHistograms = new ConcurrentHashMap<Object, CachedHistogram>();
    }

    protected abstract Object computeKey(Host var1, Statement var2, Exception var3);

    @Override
    public void update(Host host, Statement statement, Exception exception, long newLatencyNanos) {
        if (!this.include(host, statement, exception)) {
            return;
        }
        long latencyMs = TimeUnit.NANOSECONDS.toMillis(newLatencyNanos);
        try {
            Recorder recorder = this.getRecorder(host, statement, exception);
            if (recorder != null) {
                recorder.recordValue(latencyMs);
            }
        }
        catch (ArrayIndexOutOfBoundsException e) {
            logger.warn("Got request with latency of {} ms, which exceeds the configured maximum trackable value {}", (Object)latencyMs, (Object)this.highestTrackableLatencyMillis);
        }
    }

    public long getLatencyAtPercentile(Host host, Statement statement, Exception exception, double percentile) {
        Preconditions.checkArgument((percentile >= 0.0 && percentile < 100.0 ? 1 : 0) != 0, (String)"percentile must be between 0.0 and 100 (was %s)", (Object[])new Object[]{percentile});
        Histogram histogram = this.getLastIntervalHistogram(host, statement, exception);
        if (histogram == null || histogram.getTotalCount() < (long)this.minRecordedValues) {
            return -1L;
        }
        return histogram.getValueAtPercentile(percentile);
    }

    private Recorder getRecorder(Host host, Statement statement, Exception exception) {
        Object key = this.computeKey(host, statement, exception);
        if (key == null) {
            return null;
        }
        Recorder recorder = (Recorder)this.recorders.get(key);
        if (recorder == null) {
            recorder = new Recorder(this.highestTrackableLatencyMillis, this.numberOfSignificantValueDigits);
            Recorder old = this.recorders.putIfAbsent(key, recorder);
            if (old != null) {
                recorder = old;
            } else {
                this.cachedHistograms.putIfAbsent(key, CachedHistogram.empty());
            }
        }
        return recorder;
    }

    private Histogram getLastIntervalHistogram(Host host, Statement statement, Exception exception) {
        Object key = this.computeKey(host, statement, exception);
        if (key == null) {
            return null;
        }
        try {
            Histogram staleHistogram;
            Recorder recorder;
            SettableFuture future;
            CachedHistogram newEntry;
            CachedHistogram entry;
            do {
                if ((entry = (CachedHistogram)this.cachedHistograms.get(key)) == null) {
                    return null;
                }
                long age = System.currentTimeMillis() - entry.timestamp;
                if (age < this.intervalMs) {
                    return (Histogram)entry.histogram.get();
                }
                recorder = (Recorder)this.recorders.get(key);
                staleHistogram = (Histogram)entry.histogram.get(0L, TimeUnit.MILLISECONDS);
            } while (!this.cachedHistograms.replace(key, entry, newEntry = new CachedHistogram((ListenableFuture<Histogram>)(future = SettableFuture.create()))));
            Histogram newHistogram = recorder.getIntervalHistogram(staleHistogram);
            future.set((Object)newHistogram);
            return newHistogram;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return null;
        }
        catch (ExecutionException e) {
            throw new DriverInternalError("Unexpected error", e.getCause());
        }
        catch (TimeoutException e) {
            throw new DriverInternalError("Unexpected timeout while getting histogram", e);
        }
    }

    @Override
    public void onRegister(Cluster cluster) {
    }

    @Override
    public void onUnregister(Cluster cluster) {
    }

    protected boolean include(Host host, Statement statement, Exception exception) {
        if (exception == null) {
            return true;
        }
        return !EXCLUDED_EXCEPTIONS.contains(exception.getClass());
    }

    public static abstract class Builder<B, T> {
        protected final long highestTrackableLatencyMillis;
        protected int numberOfSignificantValueDigits = 3;
        protected int minRecordedValues = 1000;
        protected long intervalMs = TimeUnit.MINUTES.toMillis(5L);

        Builder(long highestTrackableLatencyMillis) {
            this.highestTrackableLatencyMillis = highestTrackableLatencyMillis;
        }

        protected abstract B self();

        public B withNumberOfSignificantValueDigits(int numberOfSignificantValueDigits) {
            this.numberOfSignificantValueDigits = numberOfSignificantValueDigits;
            return this.self();
        }

        public B withMinRecordedValues(int minRecordedValues) {
            this.minRecordedValues = minRecordedValues;
            return this.self();
        }

        public B withInterval(long interval, TimeUnit unit) {
            this.intervalMs = TimeUnit.MILLISECONDS.convert(interval, unit);
            return this.self();
        }

        public abstract T build();
    }

    static class CachedHistogram {
        final ListenableFuture<Histogram> histogram;
        final long timestamp;

        CachedHistogram(ListenableFuture<Histogram> histogram) {
            this.histogram = histogram;
            this.timestamp = System.currentTimeMillis();
        }

        static CachedHistogram empty() {
            return new CachedHistogram((ListenableFuture<Histogram>)Futures.immediateFuture(null));
        }
    }
}

