/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.redis.connection;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.ClusterRedirectException;
import org.springframework.data.redis.ClusterStateFailureException;
import org.springframework.data.redis.ExceptionTranslationStrategy;
import org.springframework.data.redis.TooManyClusterRedirectionsException;
import org.springframework.data.redis.connection.ClusterCommandExecutionFailureException;
import org.springframework.data.redis.connection.ClusterNodeResourceProvider;
import org.springframework.data.redis.connection.ClusterTopology;
import org.springframework.data.redis.connection.ClusterTopologyProvider;
import org.springframework.data.redis.connection.RedisClusterNode;
import org.springframework.data.redis.connection.util.ByteArraySet;
import org.springframework.data.redis.connection.util.ByteArrayWrapper;
import org.springframework.lang.Nullable;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

public class ClusterCommandExecutor
implements DisposableBean {
    private AsyncTaskExecutor executor;
    private final ClusterTopologyProvider topologyProvider;
    private final ClusterNodeResourceProvider resourceProvider;
    private final ExceptionTranslationStrategy exceptionTranslationStrategy;
    private int maxRedirects = 5;

    public ClusterCommandExecutor(ClusterTopologyProvider topologyProvider, ClusterNodeResourceProvider resourceProvider, ExceptionTranslationStrategy exceptionTranslation) {
        if (this.executor == null) {
            ThreadPoolTaskExecutor threadPoolTaskExecutor = new ThreadPoolTaskExecutor();
            threadPoolTaskExecutor.initialize();
            this.executor = threadPoolTaskExecutor;
        }
        Assert.notNull((Object)topologyProvider, (String)"ClusterTopologyProvider must not be null!");
        Assert.notNull((Object)resourceProvider, (String)"ClusterNodeResourceProvider must not be null!");
        Assert.notNull((Object)exceptionTranslation, (String)"ExceptionTranslationStrategy must not be null!");
        this.topologyProvider = topologyProvider;
        this.resourceProvider = resourceProvider;
        this.exceptionTranslationStrategy = exceptionTranslation;
    }

    public ClusterCommandExecutor(ClusterTopologyProvider topologyProvider, ClusterNodeResourceProvider resourceProvider, ExceptionTranslationStrategy exceptionTranslation, @Nullable AsyncTaskExecutor executor) {
        this(topologyProvider, resourceProvider, exceptionTranslation);
        this.executor = executor;
    }

    public <T> NodeResult<T> executeCommandOnArbitraryNode(ClusterCommandCallback<?, T> cmd) {
        Assert.notNull(cmd, (String)"ClusterCommandCallback must not be null!");
        ArrayList<RedisClusterNode> nodes = new ArrayList<RedisClusterNode>(this.getClusterTopology().getActiveNodes());
        return this.executeCommandOnSingleNode(cmd, (RedisClusterNode)nodes.get(new Random().nextInt(nodes.size())));
    }

    public <S, T> NodeResult<T> executeCommandOnSingleNode(ClusterCommandCallback<S, T> cmd, RedisClusterNode node) {
        return this.executeCommandOnSingleNode(cmd, node, 0);
    }

    private <S, T> NodeResult<T> executeCommandOnSingleNode(ClusterCommandCallback<S, T> cmd, RedisClusterNode node, int redirectCount) {
        Assert.notNull(cmd, (String)"ClusterCommandCallback must not be null!");
        Assert.notNull((Object)node, (String)"RedisClusterNode must not be null!");
        if (redirectCount > this.maxRedirects) {
            throw new TooManyClusterRedirectionsException(String.format("Cannot follow Cluster Redirects over more than %s legs. Please consider increasing the number of redirects to follow. Current value is: %s.", redirectCount, this.maxRedirects));
        }
        RedisClusterNode nodeToUse = this.lookupNode(node);
        Object client = this.resourceProvider.getResourceForSpecificNode(nodeToUse);
        Assert.notNull(client, (String)"Could not acquire resource for node. Is your cluster info up to date?");
        try {
            NodeResult<T> nodeResult = new NodeResult<T>(node, cmd.doInCluster(client));
            return nodeResult;
        }
        catch (RuntimeException ex) {
            DataAccessException translatedException = this.convertToDataAccessException(ex);
            if (translatedException instanceof ClusterRedirectException) {
                ClusterRedirectException cre = (ClusterRedirectException)((Object)translatedException);
                NodeResult<T> nodeResult = this.executeCommandOnSingleNode(cmd, this.topologyProvider.getTopology().lookup(cre.getTargetHost(), cre.getTargetPort()), redirectCount + 1);
                return nodeResult;
            }
            throw translatedException != null ? translatedException : ex;
        }
        finally {
            this.resourceProvider.returnResourceForSpecificNode(nodeToUse, client);
        }
    }

    private RedisClusterNode lookupNode(RedisClusterNode node) {
        try {
            return this.topologyProvider.getTopology().lookup(node);
        }
        catch (ClusterStateFailureException e) {
            throw new IllegalArgumentException(String.format("Node %s is unknown to cluster", node), (Throwable)((Object)e));
        }
    }

    public <S, T> MultiNodeResult<T> executeCommandOnAllNodes(ClusterCommandCallback<S, T> cmd) {
        return this.executeCommandAsyncOnNodes(cmd, this.getClusterTopology().getActiveMasterNodes());
    }

    public <S, T> MultiNodeResult<T> executeCommandAsyncOnNodes(ClusterCommandCallback<S, T> callback, Iterable<RedisClusterNode> nodes) {
        Assert.notNull(callback, (String)"Callback must not be null!");
        Assert.notNull(nodes, (String)"Nodes must not be null!");
        ArrayList<RedisClusterNode> resolvedRedisClusterNodes = new ArrayList<RedisClusterNode>();
        ClusterTopology topology = this.topologyProvider.getTopology();
        for (RedisClusterNode node : nodes) {
            try {
                resolvedRedisClusterNodes.add(topology.lookup(node));
            }
            catch (ClusterStateFailureException e) {
                throw new IllegalArgumentException(String.format("Node %s is unknown to cluster", node), (Throwable)((Object)e));
            }
        }
        LinkedHashMap<NodeExecution, Future<NodeResult<T>>> futures = new LinkedHashMap<NodeExecution, Future<NodeResult<T>>>();
        for (RedisClusterNode node : resolvedRedisClusterNodes) {
            futures.put(new NodeExecution(node), this.executor.submit(() -> this.executeCommandOnSingleNode(callback, node)));
        }
        return this.collectResults(futures);
    }

    private <T> MultiNodeResult<T> collectResults(Map<NodeExecution, Future<NodeResult<T>>> futures) {
        boolean done = false;
        MultiNodeResult result = new MultiNodeResult();
        HashMap<RedisClusterNode, Throwable> exceptions = new HashMap<RedisClusterNode, Throwable>();
        HashSet<String> saveGuard = new HashSet<String>();
        while (!done) {
            done = true;
            for (Map.Entry<NodeExecution, Future<NodeResult<T>>> entry : futures.entrySet()) {
                DataAccessException ex;
                if (!entry.getValue().isDone() && !entry.getValue().isCancelled()) {
                    done = false;
                    continue;
                }
                NodeExecution execution = entry.getKey();
                try {
                    String futureId = ObjectUtils.getIdentityHexString(entry.getValue());
                    if (saveGuard.contains(futureId)) continue;
                    if (execution.isPositional()) {
                        result.add(execution.getPositionalKey(), entry.getValue().get());
                    } else {
                        result.add(entry.getValue().get());
                    }
                    saveGuard.add(futureId);
                }
                catch (ExecutionException e) {
                    ex = this.convertToDataAccessException((Exception)e.getCause());
                    exceptions.put(execution.getNode(), ex != null ? ex : e.getCause());
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    ex = this.convertToDataAccessException((Exception)e.getCause());
                    exceptions.put(execution.getNode(), ex != null ? ex : e.getCause());
                    break;
                }
            }
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                done = true;
                Thread.currentThread().interrupt();
            }
        }
        if (!exceptions.isEmpty()) {
            throw new ClusterCommandExecutionFailureException(new ArrayList(exceptions.values()));
        }
        return result;
    }

    public <S, T> MultiNodeResult<T> executeMultiKeyCommand(MultiKeyClusterCommandCallback<S, T> cmd, Iterable<byte[]> keys) {
        HashMap<RedisClusterNode, PositionalKeys> nodeKeyMap = new HashMap<RedisClusterNode, PositionalKeys>();
        int index = 0;
        for (byte[] key : keys) {
            for (RedisClusterNode node : this.getClusterTopology().getKeyServingNodes(key)) {
                nodeKeyMap.computeIfAbsent(node, val -> PositionalKeys.empty()).append(PositionalKey.of(key, index++));
            }
        }
        LinkedHashMap<NodeExecution, Future<NodeResult<T>>> futures = new LinkedHashMap<NodeExecution, Future<NodeResult<T>>>();
        for (Map.Entry entry : nodeKeyMap.entrySet()) {
            if (!((RedisClusterNode)entry.getKey()).isMaster()) continue;
            for (PositionalKey key : (PositionalKeys)entry.getValue()) {
                futures.put(new NodeExecution((RedisClusterNode)entry.getKey(), key), this.executor.submit(() -> this.executeMultiKeyCommandOnSingleNode(cmd, (RedisClusterNode)entry.getKey(), key.getBytes())));
            }
        }
        return this.collectResults(futures);
    }

    private <S, T> NodeResult<T> executeMultiKeyCommandOnSingleNode(MultiKeyClusterCommandCallback<S, T> cmd, RedisClusterNode node, byte[] key) {
        Assert.notNull(cmd, (String)"MultiKeyCommandCallback must not be null!");
        Assert.notNull((Object)node, (String)"RedisClusterNode must not be null!");
        Assert.notNull((Object)key, (String)"Keys for execution must not be null!");
        Object client = this.resourceProvider.getResourceForSpecificNode(node);
        Assert.notNull(client, (String)"Could not acquire resource for node. Is your cluster info up to date?");
        try {
            NodeResult<T> nodeResult = new NodeResult<T>(node, cmd.doInCluster(client, key), key);
            return nodeResult;
        }
        catch (RuntimeException ex) {
            DataAccessException translatedException = this.convertToDataAccessException(ex);
            throw translatedException != null ? translatedException : ex;
        }
        finally {
            this.resourceProvider.returnResourceForSpecificNode(node, client);
        }
    }

    private ClusterTopology getClusterTopology() {
        return this.topologyProvider.getTopology();
    }

    @Nullable
    private DataAccessException convertToDataAccessException(Exception e) {
        return this.exceptionTranslationStrategy.translate(e);
    }

    public void setMaxRedirects(int maxRedirects) {
        this.maxRedirects = maxRedirects;
    }

    public void destroy() throws Exception {
        if (this.executor instanceof DisposableBean) {
            ((DisposableBean)this.executor).destroy();
        }
        if (this.resourceProvider instanceof DisposableBean) {
            ((DisposableBean)this.resourceProvider).destroy();
        }
    }

    private static class PositionalKeys
    implements Iterable<PositionalKey> {
        private final List<PositionalKey> keys;

        private PositionalKeys(List<PositionalKey> keys) {
            this.keys = keys;
        }

        static PositionalKeys empty() {
            return new PositionalKeys(new ArrayList<PositionalKey>());
        }

        static PositionalKeys of(byte[] ... keys) {
            ArrayList<PositionalKey> result = new ArrayList<PositionalKey>(keys.length);
            for (int i = 0; i < keys.length; ++i) {
                result.add(PositionalKey.of(keys[i], i));
            }
            return new PositionalKeys(result);
        }

        static PositionalKeys of(PositionalKey ... keys) {
            PositionalKeys result = PositionalKeys.empty();
            result.append(keys);
            return result;
        }

        void append(PositionalKey ... keys) {
            this.keys.addAll(Arrays.asList(keys));
        }

        int indexOf(PositionalKey key) {
            return this.keys.indexOf(key);
        }

        @Override
        public Iterator<PositionalKey> iterator() {
            return this.keys.iterator();
        }
    }

    private static class PositionalKey {
        private final ByteArrayWrapper key;
        private final int position;

        private PositionalKey(ByteArrayWrapper key, int position) {
            this.key = key;
            this.position = position;
        }

        static PositionalKey of(byte[] key, int index) {
            return new PositionalKey(new ByteArrayWrapper(key), index);
        }

        byte[] getBytes() {
            return this.key.getArray();
        }

        public ByteArrayWrapper getKey() {
            return this.key;
        }

        public int getPosition() {
            return this.position;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            PositionalKey that = (PositionalKey)o;
            if (this.position != that.position) {
                return false;
            }
            return ObjectUtils.nullSafeEquals((Object)this.key, (Object)that.key);
        }

        public int hashCode() {
            int result = ObjectUtils.nullSafeHashCode((Object)this.key);
            result = 31 * result + this.position;
            return result;
        }
    }

    public static class MultiNodeResult<T> {
        List<NodeResult<T>> nodeResults = new ArrayList<NodeResult<T>>();
        Map<PositionalKey, NodeResult<T>> positionalResults = new LinkedHashMap<PositionalKey, NodeResult<T>>();

        private void add(NodeResult<T> result) {
            this.nodeResults.add(result);
        }

        private void add(PositionalKey key, NodeResult<T> result) {
            this.positionalResults.put(key, result);
            this.add(result);
        }

        public List<NodeResult<T>> getResults() {
            return Collections.unmodifiableList(this.nodeResults);
        }

        public List<T> resultsAsList() {
            return this.toList(this.nodeResults);
        }

        public List<T> resultsAsListSortBy(byte[] ... keys) {
            if (this.positionalResults.isEmpty()) {
                ArrayList<NodeResult<T>> clone = new ArrayList<NodeResult<T>>(this.nodeResults);
                clone.sort(new ResultByReferenceKeyPositionComparator(keys));
                return this.toList(clone);
            }
            TreeMap<PositionalKey, NodeResult<T>> result = new TreeMap<PositionalKey, NodeResult<T>>(new ResultByKeyPositionComparator(keys));
            result.putAll(this.positionalResults);
            return result.values().stream().map(tNodeResult -> ((NodeResult)tNodeResult).value).collect(Collectors.toList());
        }

        @Nullable
        public T getFirstNonNullNotEmptyOrDefault(@Nullable T returnValue) {
            for (NodeResult<T> nodeResult : this.nodeResults) {
                if (nodeResult.getValue() == null) continue;
                if (nodeResult.getValue() instanceof Map) {
                    if (!CollectionUtils.isEmpty((Map)((Map)nodeResult.getValue()))) continue;
                    return nodeResult.getValue();
                }
                if (nodeResult.getValue() instanceof Collection && CollectionUtils.isEmpty((Collection)((Collection)nodeResult.getValue()))) {
                    return nodeResult.getValue();
                }
                return nodeResult.getValue();
            }
            return returnValue;
        }

        private List<T> toList(Collection<NodeResult<T>> source) {
            ArrayList<T> result = new ArrayList<T>();
            for (NodeResult<T> nodeResult : source) {
                result.add(nodeResult.getValue());
            }
            return result;
        }

        private static class ResultByKeyPositionComparator
        implements Comparator<PositionalKey> {
            private final PositionalKeys reference;

            ResultByKeyPositionComparator(byte[] ... keys) {
                this.reference = PositionalKeys.of(keys);
            }

            @Override
            public int compare(PositionalKey o1, PositionalKey o2) {
                return Integer.compare(this.reference.indexOf(o1), this.reference.indexOf(o2));
            }
        }

        private static class ResultByReferenceKeyPositionComparator
        implements Comparator<NodeResult<?>> {
            private final List<ByteArrayWrapper> reference;

            ResultByReferenceKeyPositionComparator(byte[] ... keys) {
                this.reference = new ArrayList<ByteArrayWrapper>(new ByteArraySet(Arrays.asList(keys)));
            }

            @Override
            public int compare(NodeResult<?> o1, NodeResult<?> o2) {
                return Integer.compare(this.reference.indexOf(((NodeResult)o1).key), this.reference.indexOf(((NodeResult)o2).key));
            }
        }
    }

    public static class NodeResult<T> {
        private RedisClusterNode node;
        @Nullable
        private T value;
        private ByteArrayWrapper key;

        public NodeResult(RedisClusterNode node, @Nullable T value) {
            this(node, value, new byte[0]);
        }

        public NodeResult(RedisClusterNode node, @Nullable T value, byte[] key) {
            this.node = node;
            this.value = value;
            this.key = new ByteArrayWrapper(key);
        }

        @Nullable
        public T getValue() {
            return this.value;
        }

        public RedisClusterNode getNode() {
            return this.node;
        }

        public byte[] getKey() {
            return this.key.getArray();
        }

        @Nullable
        public <U> U mapValue(Function<? super T, ? extends U> mapper) {
            Assert.notNull(mapper, (String)"Mapper function must not be null!");
            return mapper.apply(this.getValue());
        }
    }

    private static class NodeExecution {
        private final RedisClusterNode node;
        @Nullable
        private final PositionalKey positionalKey;

        NodeExecution(RedisClusterNode node) {
            this(node, null);
        }

        NodeExecution(RedisClusterNode node, @Nullable PositionalKey positionalKey) {
            this.node = node;
            this.positionalKey = positionalKey;
        }

        RedisClusterNode getNode() {
            return this.node;
        }

        PositionalKey getPositionalKey() {
            return this.positionalKey;
        }

        boolean isPositional() {
            return this.positionalKey != null;
        }
    }

    public static interface MultiKeyClusterCommandCallback<T, S> {
        public S doInCluster(T var1, byte[] var2);
    }

    public static interface ClusterCommandCallback<T, S> {
        public S doInCluster(T var1);
    }
}

