package org.planx.xmlstore.stores;

import java.io.IOException;
import java.io.Serializable;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.util.Collections;
import java.util.List;
import org.planx.xmlstore.*;
import org.planx.xmlstore.nameserver.GlobalNameServer;
import org.planx.xmlstore.references.*;
import org.planx.xmlstore.routing.DistributedMap;
import org.planx.xmlstore.routing.Kademlia;
import org.planx.xmlstore.routing.Identifier;

/**
 * A peer-to-peer <code>XMLStore</code> that uses {@link Kademlia} as routing module.
 * It only operates on location independent <code>ValueReference</code>s. If an
 * <code>XMLStore</code> that does not operate on <code>ValueReference</code>s is
 * used as the underlying <code>XMLStore</code> (e.g. {@link LocalXMLStore}), it must
 * be wrapped in a {@link TranslatorXMLStore}.
 *
 * @author Thomas Ambus
 */
public class DistributedXMLStore extends AbstractXMLStore {
    private NetworkSkeleton networkSkeleton;
    private DistributedMap distMap;
    private InetSocketAddress tcpAddr;
    private NameServer nameServer = null;

    /**
     * Creates a DistributedXMLStore which listens for routing requests on the
     * specified UDP port and listens for data transfer requests on the specified
     * TCP port. The XML Store is bootstraped to an existing XML Store network
     * by specifying a bootstrap IP and UDP port. If the bootstrap IP is
     * <code>null</code> the XML Store will not attempt to connect to an existing
     * network. The DistributedXMLStore only saves locally.
     *
     * @param xmlstore     The underlying <code>XMLStore</code>.
     * @param udpPort      UDP port on which to listen for routing requests
     * @param tcpPort      TCP port on which to listen for data transfer requests
     * @param bootstrap    IP and UDP port of an existing peer in the network
     */
    public DistributedXMLStore(XMLStore xmlstore, int udpPort, int tcpPort,
                           InetSocketAddress bootstrap) throws IOException {
        super(xmlstore);

        String name = xmlstore.toString();
        distMap = new Kademlia(name, udpPort, tcpPort, bootstrap, null);

        // Create and start network TCP listener
        tcpAddr = new InetSocketAddress(InetAddress.getLocalHost(), tcpPort);
        networkSkeleton = new NetworkSkeleton(xmlstore, tcpPort);
    }

    /**
     * Returns a {@link GlobalNameServer} that uses the same <code>DistributedMap</code>
     * as this <code>DistributedXMLStore</code>.
     */
    public NameServer getNameServer() {
        if (nameServer == null) {
            nameServer = new GlobalNameServer(distMap);
        }
        return nameServer;
    }

    public void close() throws IOException {
        checkClosed();
        distMap.close();
        networkSkeleton.close();
        super.close();
    }

    /**
     * Always saves locally even if the node might already be stored elsewhere
     * in the network. This is because it is considered very likely that the
     * node will be loaded locally later.
     */
    public Reference save(Node node) throws IOException {
        checkClosed();

        ValueReference vref = (ValueReference) xmlstore.save(node);

        // Lookup vref globally and get list of InetSocketAddresses where
        // vref is already stored
        Identifier id = vref.asIdentifier();
        List locations = (List) distMap.get(id);

        // If vref not published, publish that data is stored at this peer
        if (locations == null) {
            distMap.put(id, (Serializable) Collections.singletonList(tcpAddr));
        }
        return vref;
    }

    /**
     * If the <code>Reference</code> is not known in the underlying
     * <code>XMLStore</code> it is looked up in the network and loaded from
     * another peer if it exists. If the <code>Reference</code> could
     * not be found an <code>UnknownReferenceException</code> is thrown.
     */
    protected Node resolvedLoad(Reference vref) throws IOException, UnknownReferenceException {
        checkClosed();
        ValueReference gref = (ValueReference) vref;

        try {
            // Attempt to load node locally
            return xmlstore.load(gref);

        } catch (UnknownReferenceException ue) {

            // Lookup root ref globally
            List locations = (List) distMap.get(gref.asIdentifier());

            if (locations == null)
                throw new UnknownReferenceException
                    ("Reference not found locally or globally");

            // Attempt to load from each peer in succession,
            // stopping when the node has been successfully loaded from a peer
            Node node = null;
            for (int i=0,max=locations.size(); i<max; i++) {
                XMLStore extStore = null;
                try {
                    InetSocketAddress addr = (InetSocketAddress) locations.get(i);
                    extStore = new NetworkProxy(addr);
                    node = extStore.load(gref); // possibly relative reference

                    // Load whole tree so NetworkProxy can be closed
                    // TODO: Make lazy loading system that doesn't require a NetworkProxy
                    // to be open forever
                    loadTree(node);

                    extStore.close();
                    extStore = null;
                    break;
                } catch (IOException e) {
                    e.printStackTrace();
                } catch (UnknownReferenceException e) {
                    e.printStackTrace();
                } finally {
                    if (extStore != null) {
                        extStore.close();
                        extStore = null;
                    }
                }
            }

            if (node == null)
                throw new UnknownReferenceException
                    ("Could not load node from external peer");
            return node;
        }
    }

    private void loadTree(Node node) {
        List<? extends Node> children = node.getChildren();
        for (Node n : children) {
            loadTree(n);
        }
    }
}
