package org.planx.xmlstore.references;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import org.planx.msd.Discriminator;
import org.planx.msd.DiscriminatorFactory;
import org.planx.msd.Memory;
import org.planx.msd.array.ByteArrayDiscriminator;
import org.planx.msd.util.DiscriminatorAdapter;
import org.planx.util.Array;
import org.planx.util.Converter;
import org.planx.xmlstore.Attribute;
import org.planx.xmlstore.Node;
import org.planx.xmlstore.io.Streamer;
import org.planx.xmlstore.nodes.SystemNode;
import org.planx.xmlstore.routing.Identifier;

/**
 * A <code>ValueReference</code> created from a content hash of a <code>Node</code>
 * (currently MD5). This gives a unique location independent <code>ValueReference</code>
 * that can be computed anywhere based on <code>Node</code> data.
 *
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public class ContentValueReference implements ValueReference, Serializable,
                                          Comparable<ContentValueReference> {
    public static final byte CLASS_ID = 0x02;
    private static final int LENGTH = 16;
    private byte[] contentHash;
    private transient int hash = 0;

    /**
     * Creates a new <code>ContentValueReference</code> by computing a content
     * hash of the tree rooted at the specified <code>Node</code>.
     */
    public ContentValueReference(Node node) {
        contentHash = reference(node, null).contentHash;
    }

    ContentValueReference(byte[] contentHash) {
        if (contentHash.length != LENGTH)
            throw new IllegalArgumentException("Invalid content hash length");
        this.contentHash = contentHash;
    }

    public static ContentValueReference reference(Node node,
                                ReferenceListener listener) {
        if (node instanceof SystemNode) {
            SystemNode n = (SystemNode) node;
            if (n.getReference() instanceof ContentValueReference)
                return (ContentValueReference) n.getReference();
        }
        return computeMD5(node, listener);
    }

    private static ContentValueReference computeMD5(Node node,
                                ReferenceListener listener) {
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            return computeMD5(node, md, listener);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    private static ContentValueReference computeMD5(Node node,
                         MessageDigest mdFactory,
                         ReferenceListener listener) throws CloneNotSupportedException {
        if (listener != null) {
            ContentValueReference vref = listener.lookup(node);
            if (vref != null) return vref;
        }

        MessageDigest md = (MessageDigest) mdFactory.clone();
        md.reset();

        md.update(node.getNodeValue().getBytes());
        if (node.getType() == Node.ELEMENT) {
            // compute MD5 of attributes
            // attributes list might be immutable, so it cannot be sorted in-place
            List<Attribute> attributes = node.getAttributes();
            Attribute[] attrs = new Attribute[attributes.size()];
            attrs = attributes.toArray(attrs);
            Arrays.sort(attrs);

            for (int i=0; i<attrs.length; i++) {
                md.update(attrs[i].getName().getBytes());
                md.update(attrs[i].getValue().getBytes());
            }

            List<? extends Node> children = node.getChildren();
            for (Node n : children) {
                ContentValueReference vref = computeMD5(n, mdFactory, listener);
                md.update(vref.contentHash);
            }
        }
        ContentValueReference ref = new ContentValueReference(md.digest());
        if (listener != null) listener.referenceComputed(node, ref);
        return ref;
    }

    /**
     * Returns this reference as an <code>Identifier</code> acceptable
     * by the routing layer.
     */
    public Identifier asIdentifier() {
        return new Identifier(contentHash);
    }

    /**
     * Returns <code>true</code> if and only if the specified object
     * is also a <code>ContentValueReference</code> and it represents
     * the same content hash
     * as this <code>ContentValueReference</code>.
     */
    public boolean equals(Object o) {
        if (o == this) return true;
        if (!(o instanceof ContentValueReference)) return false;
        ContentValueReference vref = (ContentValueReference) o;
        return Arrays.equals(contentHash, vref.contentHash);
    }

    public int compareTo(ContentValueReference vref) {
        if (vref == null) throw new NullPointerException();
        return Array.compareTo(contentHash, vref.contentHash);
    }

    public int hashCode() {
        if (hash == 0) hash = Arrays.hashCode(contentHash);
        return hash;
    }

    public String toString() {
        return Converter.toHex(contentHash);
    }

    public static Streamer<ContentValueReference> getStreamer() {
        return new Streamer<ContentValueReference>() {
            public void toStream(DataOutput out, ContentValueReference vref)
                    throws IOException {
                out.write(vref.contentHash);
            }
            public ContentValueReference fromStream(DataInput in)
                    throws IOException {
                byte[] id = new byte[LENGTH];
                in.readFully(id);
                return new ContentValueReference(id);
            }
        };
    }

    public static Discriminator<ContentValueReference> getDiscriminator() {
        return getDiscriminator(DiscriminatorFactory.instance().getMemory());
    }

    public static Discriminator<ContentValueReference> getDiscriminator(
                                                        Memory memory) {
        return new DiscriminatorAdapter<ContentValueReference,byte[]>
                                (new ByteArrayDiscriminator(memory)) {
            protected byte[] transform(ContentValueReference vref) {
                return vref.contentHash;
            }
        };
    }
}
