/*
 * #%L
 * Wikitty :: wikitty-jdbc-impl
 * 
 * $Id: WikittyJDBCUtil.java 444 2010-10-21 14:32:50Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-2.2.2/wikitty-jdbc-impl/src/main/java/org/nuiton/wikitty/jdbc/WikittyJDBCUtil.java $
 * %%
 * Copyright (C) 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.wikitty.jdbc;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import javax.sql.XADataSource;

import javax.transaction.TransactionManager;
import org.apache.commons.beanutils.BeanMap;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.dbcp.managed.BasicManagedDataSource;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyException;

/**
 *
 * @author morin
 * @version $Revision: 444 $
 *
 * Last update: $Date: 2010-10-21 16:32:50 +0200 (jeu., 21 oct. 2010) $
 * by : $Author: bpoussin $
 */
public class WikittyJDBCUtil {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyJDBCUtil.class);

    /** extension list column in the wikitty_admin table */
    static final public String COL_EXTENSION  = "extension_list";
    /** version column in the admin tables */
    static final public String COL_VERSION = "version";
    /** id column in all the tables */
    static final public String COL_ID = "id";
    /** name column in the extension_admin table */
    static final public String COL_NAME = "name";
    /** requires column in the extension_admin table */
    static final public String COL_REQUIRES = "requires";
    /** tagvalues column in the extension_admin table */
    static final public String COL_TAGVALUES = "tagvalues";
    /** field name column in the data tables */
    static final public String COL_FIELDNAME = "fieldName";
    /** field type column in the extension_data table */
    static final public String COL_FIELDTYPE = "fieldType";
    /** boolean value column in the wikitty_data table */
    static final public String COL_BOOLEAN_VALUE = "booleanValue";
    /** number value column in the wikitty_data table */
    static final public String COL_NUMBER_VALUE = "numberValue";
    /** text value column in the wikitty_data table */
    static final public String COL_TEXT_VALUE = "textValue";
    /** date value column in the wikitty_data table */
    static final public String COL_DATE_VALUE = "dateValue";
    /** deletion date column in wikitty_admin table */
    static final public String COL_DELETION_DATE = "deletionDate";

    /** basic selection without where clause query property name */
    static final public String QUERY_SELECT = "jdbc.queries.select";
    /** basic selection without where clause query property name */
    static final public String QUERY_SELECT_NOTDELETED = "jdbc.queries.select.notdeleted";
    /** basic selection with where clause query property name */
    static final public String QUERY_SELECT_WHERE = "jdbc.queries.select.where";
    /** basic selection with where clause query property name */
    static final public String QUERY_SELECT_TWO_WHERE = "jdbc.queries.select.two.where";
    /** not deleted data selection with where clause query property name */
    static final public String QUERY_SELECT_WHERE_NOTDELETED = "jdbc.queries.select.where.notdeleted";

    /** wikitty_admin table creation query property name */
    static final public String QUERY_CREATION_WIKITTY_ADMIN_TEST =
            "jdbc.queries.creation.wikitty.admin.test";
    static final public String QUERY_CREATION_WIKITTY_ADMIN =
            "jdbc.queries.creation.wikitty.admin";
    /** wikitty_admin table creation query property name */
    static final public String QUERY_CREATION_WIKITTY_DATA_TEST =
            "jdbc.queries.creation.wikitty.data.test";
    static final public String QUERY_CREATION_WIKITTY_DATA =
            "jdbc.queries.creation.wikitty.data";
    /** insertion in the admin table query property name */
    static final public String QUERY_INSERT_WIKITTY_ADMIN = "jdbc.queries.insert.wikitty.admin";
    /** update in the admin table query property name */
    static final public String QUERY_UPDATE_WIKITTY_ADMIN = "jdbc.queries.update.wikitty.admin";
    /** insertion in the data table query property name */
    static final public String QUERY_INSERT_WIKITTY_DATA = "jdbc.queries.insert.wikitty.data";
    /** deletion in the admin table query property name */
    static final public String QUERY_DELETE_WIKITTY_ADMIN = "jdbc.queries.delete.wikitty.admin";
    /** deletion in the data table query property name */
    static final public String QUERY_DELETE_WIKITTY_DATA = "jdbc.queries.delete.wikitty.data";

    /** clear extension query property name */
    static final public String QUERY_CLEAR_EXTENSION = "jdbc.queries.clear.extension";
    /** clear wikitty query property name */
    static final public String QUERY_CLEAR_WIKITTY = "jdbc.queries.clear.wikitty";

    /** extension_admin table creation query property name */
    static final public String QUERY_CREATION_EXTENSION_ADMIN_TEST =
            "jdbc.queries.creation.extension.admin.test";
    static final public String QUERY_CREATION_EXTENSION_ADMIN =
            "jdbc.queries.creation.extension.admin";
    /** extension_data table creation query property name */
    static final public String QUERY_CREATION_EXTENSION_DATA_TEST =
            "jdbc.queries.creation.extension.data.test";
    static final public String QUERY_CREATION_EXTENSION_DATA =
            "jdbc.queries.creation.extension.data";
    /** insertion in the admin table query property name */
    static final public String QUERY_INSERT_EXTENSION_ADMIN =
            "jdbc.queries.insert.extension.admin";
    /** insertion in the data table query property name */
    static final public String QUERY_INSERT_EXTENSION_DATA =
            "jdbc.queries.insert.extension.data";

    /** JDBC JDBC_DRIVER property name */
    static protected String JDBC_DRIVER = "jdbc.con.driver";
    /** JDBC_HOST property name */
    static protected String JDBC_HOST = "jdbc.con.host";
    /** JDBC_USER_NAME property name */
    static protected String JDBC_USER_NAME = "jdbc.con.userName";
    /** JDBC_PASSWORD property name */
    static protected String JDBC_PASSWORD = "jdbc.con.password";
    /** JDBC_XADATASOURCE property name */
    static protected String JDBC_XADATASOURCE = "jdbc.xadatasource";

    /** admin table name */
    static protected String TABLE_WIKITTY_ADMIN = "wikitty_admin";
    /** data table name */
    static protected String TABLE_WIKITTY_DATA = "wikitty_data";
    /** admin table name */
    static protected String TABLE_EXTENSION_ADMIN = "extension_admin";
    /** data table name */
    static protected String TABLE_EXTENSION_DATA = "extension_data";


    /**
     * Loads the properties in the {@code wikitty-jdbc-config.properties} file.
     *
     * @param properties custom properties to override default configuration
     * @return the properties for the connection and the queries
     */
    public static synchronized Properties loadProperties(Properties properties) {
        Properties queryConfig = new Properties();
        Properties databaseConfig = new Properties(queryConfig);
        
        InputStream streamQuery = null;
        InputStream streamConfig = null;
        try {
            // FIXME poussin 20100112 perhaps used nuitonutil.ApplicationConfig
            
            // queries
            URL url = ClassLoader.getSystemResource("wikitty-jdbc-query.properties");
            if (url == null) {
                ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
                url = contextClassLoader.getResource("wikitty-jdbc-query.properties");
            }
            
            if (log.isInfoEnabled()) {
                log.info("Reading resource from: " + url);
            }
            // url can't be null
            streamQuery = url.openStream();
            queryConfig.load(streamQuery);
        
            // config
            url = ClassLoader.getSystemResource("wikitty-jdbc-config.properties");
            if (url == null) {
                ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
                url = contextClassLoader.getResource("wikitty-jdbc-config.properties");
            }

            if (url == null) {
                if (log.isInfoEnabled()) {
                    log.info("No wikitty-jdbc-config.properties file found in classpath (skip default configuration loading)");
                }
            }
            else {
                if (log.isInfoEnabled()) {
                    log.info("Reading resource from: " + url);
                }
                streamConfig = url.openStream();
                databaseConfig.load(streamConfig);
            }
        
            // extra config
            if (properties != null) {
                databaseConfig.putAll(properties);
            }
            
        } catch (IOException eee) {
            throw new WikittyException("Unable to load property file", eee);
        }
        finally {
            IOUtils.closeQuietly(streamQuery);
            IOUtils.closeQuietly(streamConfig);
        }

        return databaseConfig;
    }

    private static Map<String, BasicManagedDataSource> dataSources =
        new HashMap<String, BasicManagedDataSource>();

    /**
     * Get a new connection instance (i.e. it opens a new transaction) plug on
     * JTA.
     *
     * @param conf configuration
     * @return a new Connection (db transaction)
     */
    public static synchronized Connection getConnection(Properties conf) {
        try {
            TransactionManager transactionManager = com.arjuna.ats.jta.TransactionManager.transactionManager();

            String jdbcUrl = String.format("%s:%s@%s", conf.getProperty(JDBC_USER_NAME),
                    conf.getProperty(JDBC_PASSWORD), conf.getProperty(JDBC_HOST));
            
            if (!dataSources.containsKey(jdbcUrl)) {
                log.info("Creating BasicManagedDataSource for: " + jdbcUrl);

                // Create datasource
                BasicManagedDataSource dataSource = new BasicManagedDataSource();

                // if xadatasource
                String xaDataSourceClassName = conf.getProperty(JDBC_XADATASOURCE);
                if(xaDataSourceClassName != null) {
                    XADataSource xaDataSource = (XADataSource) Class.forName(xaDataSourceClassName).newInstance();

                    BeanMap beanMap = new BeanMap(xaDataSource);
                    Set<String> fields = beanMap.keySet();

                    // Inject properties in xadatasource
                    for(Entry<Object, Object> properties : conf.entrySet()) {
                        String propertyName = (String) properties.getKey();
                        if (propertyName.startsWith(JDBC_XADATASOURCE + "." + xaDataSourceClassName + ".")) {
                            propertyName = propertyName.replaceFirst(JDBC_XADATASOURCE + "." + xaDataSourceClassName + ".", "");
                            if(fields.contains(propertyName)) {
                                String propertyValue = (String) properties.getValue();
                                BeanUtils.setProperty(xaDataSource, propertyName, propertyValue);
                            } else {
                                log.warn("Invalid property " + propertyName + " for XADatasource " + Arrays.toString(fields.toArray()));
                            }
                        }                        
                    }
                    dataSource.setXaDataSourceInstance(xaDataSource);
                } else {
                    log.warn("No xadatasource is used, data integrity is not guarantee");
                }

                // else standard datasource
                dataSource.setDriverClassName(conf.getProperty(JDBC_DRIVER));
                dataSource.setUrl(conf.getProperty(JDBC_HOST));
                dataSource.setUsername(conf.getProperty(JDBC_USER_NAME));
                dataSource.setPassword(conf.getProperty(JDBC_PASSWORD));
                dataSource.setTransactionManager(transactionManager);

                dataSources.put(jdbcUrl, dataSource);
            }
            
            BasicManagedDataSource dataSource = dataSources.get(jdbcUrl);
            Connection connection = dataSource.getConnection();
            return connection;
            
        } catch(Exception eee) {
            throw new WikittyException(String.format(
                    "Can't connect to database %s %s with login %s",
                    JDBC_DRIVER, JDBC_HOST, JDBC_USER_NAME), eee);
        }
    }

    /**
     * Closes a connection.
     *
     * @param connection the connection to close
     */
    public static void closeQuietly(Connection connection) {
        try {
            connection.close();
        } catch (SQLException e) {
            log.error("SQLException while closing connection", e);
        }
    }

    /**
     * Get a new connection instance (i.e. it opens a new transaction).
     *
     * @return a new Connection (db transaction)
     * @throws SQLException if the connection fails
     */
    public static synchronized Connection getJDBCConnection(Properties conf) {
        try {
            Connection connection = getConnection(conf);
            connection.setAutoCommit(false);
            return connection;
        } catch(SQLException eee) {
            throw new WikittyException(
                    "Can't set connection auto commit to false", eee);
        }
    }

    /**
     * Closes a connection (i.e. transaction) and commit data.
     *
     * @param connection the connection to close and commit
     */
    public static void commitJDBCConnection(Connection connection) {
        try {
            connection.commit();
        } catch(SQLException eee) {
            throw new WikittyException("Can't commit transaction", eee);
        } finally {
            try {
                connection.close();
            }  catch(SQLException eee) {
                throw new WikittyException("Can't close connection", eee);
            }
        }
    }

    /**
     * Closes a connection (i.e. transaction) and rollback data.
     *
     * @param connection the connection to close and rollback
     */
    public static void rollbackJDBCConnection(Connection connection) {
        try {
            connection.rollback();
        } catch(SQLException eee) {
            throw new WikittyException("Can't rollback transaction", eee);
        } finally {
            try {
                connection.close();
            }  catch(SQLException eee) {
                throw new WikittyException("Can't close connection", eee);
            }
        }
    }

    /**
     * Execute query.
     * 
     * @param connection connection to use
     * @param query sql query to do
     * @param args arguments for the query
     */
    static public void doQuery(Connection connection, String query, Object ... args) throws SQLException {
        PreparedStatement sta = connection.prepareStatement(query);
        for (int i=0; i<args.length; i++) {
            if (args[i] instanceof Date) {
                // force for java.util.Date to TIMESTAMP, because some driver
                // (Postgresql) don't support it naturaly
                sta.setObject(i + 1, args[i], Types.TIMESTAMP);
            } else {
                sta.setObject(i + 1, args[i]);
            }
        }
        sta.execute();
    }

}
