/* *##% ToPIA - Migration service
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*/
package org.nuiton.topia.migration.common;

import org.nuiton.util.Version;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * MapAdapterImpl.java
 * 
 * Implementation des MapAdapter.
 * 
 * @author Chatellier Eric
 * @author Chevallereau Benjamin
 * @author Eon Sébastien
 * @author Trève Vincent
 * @version $Revision: 1459 $
 * 
 * Last update : $Date: 2009-05-16 09:56:47 +0200 (Sat, 16 May 2009) $
 */
public class MapAdapterImpl implements MapAdapter, MapAdapterAdmin {

    /**
     * Map contenant les informations de l'objet entrant.
     */
    private Map<String, Object> InnerMap;
    /**
     * Map contenant les informations de l'objet sortant.
     */
    private Map<String, Object> OuterMap;
    /**
     * Identifiant de l'objet entrant.
     */
    private Serializable idInner;
    /**
     * Nom de l'attribut de l'identifiant
     */
    private String nameAttId;
    /**
     * Nom de la classe de l'objet entrant.
     */
    private ProxyClass InnerClass;
    /**
     * Nom de la classe de l'objet sortant.
     */
    private ProxyClass OuterClass;
    /**
     * Version entrante
     */
    private Version InnerVersion;
    /**
     * Version sortante
     */
    private Version OuterVersion;

    /**
     * Logger (common-logging)
     */
    // private static Log logger =
    // LogFactory.getLog(ConfigurationAdapter.class);
    /**
     * Constructeur vide.
     */
    public MapAdapterImpl() {
        InnerMap = new HashMap<String, Object>();
        OuterMap = new HashMap<String, Object>();
    }

    /**
     * Constructeur qui permet d'initialiser complètement l'objet.
     * @param _inner
     * @param nameInnerClass
     * @param _nameAttId
     * @param _idInner 
     */
    public MapAdapterImpl(Map<String, Object> _inner,
            ProxyClass nameInnerClass, String _nameAttId, Serializable _idInner) {
        InnerMap = new HashMap<String, Object>(_inner);

        Set<String> removeKey = new TreeSet<String>();

        Set<String> s = InnerMap.keySet();
        for (String key : s) {
            Pattern pattern = Pattern.compile("\\$.*\\$");
            Matcher matcher = pattern.matcher(key);
            if (matcher.find()) {
                removeKey.add(key);
            }
        }

        for (String key : removeKey) {
            InnerMap.remove(key);
        }

        OuterMap = new HashMap<String, Object>();
        InnerClass = nameInnerClass;
        idInner = _idInner;
        nameAttId = _nameAttId;

        InnerMap.remove(nameAttId);
    }

    /**
     * Modificateur qui permet d'initialiser l'entrée de l'objet.
     * @param _nameAttId
     */
    @Override
    public void setInner(ProxyClass nameClass, String _nameAttId,
            Serializable _idInner, Map<String, Object> _inner) {
        InnerMap = _inner;
        InnerClass = nameClass;
        idInner = _idInner;
        nameAttId = _nameAttId;
        InnerMap.remove(nameAttId);
    }

    /**
     * Modificateur qui permet d'initialiser l'entrée de l'objet en supposant
     * que l'objet entrant est de la même classe que celui sortant.
     * @param _nameAttId 
     */
    @Override
    public void setInner(String _nameAttId, Serializable _idInner,
            Map<String, Object> _inner) {
        InnerMap = _inner;
        idInner = _idInner;
        nameAttId = _nameAttId;
        InnerMap.remove(nameAttId);
    }

    /**
     * Accesseur permettant de récupérer les valeurs de l'objet entrant.
     *
     * @param key
     *            Clé du champ
     * @return Valeur du champ
     * @throws ExceptionAttributeUndefined
     */
    @Override
    public Serializable getOldValue(String key)
            throws ExceptionAttributeUndefined {
        if (InnerMap.containsKey(key)) {
            return (Serializable) InnerMap.get(key);
        } else {
            throw new ExceptionAttributeUndefined();
        }
    }

    /**
     * Accesseur permettant de récupérer les valeurs de l'objet sortant.
     *
     * @param key
     *            Clé du champ
     * @return Valeur du champ
     * @throws ExceptionAttributeUndefined
     */
    @Override
    public Serializable getValue(String key) throws ExceptionAttributeUndefined {
        try {
            return (Serializable) OuterMap.get(key);
        } catch (Exception e) {
            throw new ExceptionAttributeUndefined();
        }
    }

    /**
     * Modificateur permettant de définir la valeur d'un attribut de l'objet
     * sortant.
     *
     * @param nameAtt
     *            Nom de l'attribut
     * @param valueAtt
     *            Valeur de l'attribut
     */
    @Override
    public void setValue(String nameAtt, Serializable valueAtt) {
        if (OuterMap == null) {
            OuterMap = new HashMap<String, Object>();
        }
        OuterMap.put(nameAtt, valueAtt);
    }

    /**
     * Fonction qui permet de récupérer une Map afin de la sauvegarder
     * directement en base.
     *
     * @return La map complète.
     */
    @Override
    public Map<String, Object> getOuterMap() {
        Map<String, Object> m = new HashMap<String, Object>(OuterMap);
        m.put(nameAttId, idInner);
        if (OuterClass != null) {
            m.put("$type$", OuterClass.getCanonicalName());
        }
        return m;
    }

    /**
     * Fonction qui permet de récupérer une Map contenant toutes les
     * informations de l'objet en entrée.
     *
     * @return La map complète.
     */
    @Override
    public Map<String, Object> getInnerMap() {
        Map<String, Object> m = new HashMap<String, Object>(InnerMap);
        m.put(nameAttId, idInner);
        if (InnerClass != null) {
            m.put("$type$", InnerClass.getCanonicalName());
        }
        return m;
    }

    /**
     * Retourne le nom de la classe persistante d'entrée.
     *
     * @return le nom de la classe.
     */
    @Override
    public ProxyClass getInnerClass() {
        return InnerClass;
    }

    /**
     * Retourne le nom de la classe persistante de sortie.
     *
     * @return le nom de la classe.
     */
    @Override
    public ProxyClass getOuterClass() {
        return OuterClass;
    }

    /**
     * Retourne la version en entrée.
     *
     * @return la version en entrée.
     */
    @Override
    public Version getInnerVersion() {
        return InnerVersion;
    }

    /**
     * Modificateur de la version entrante.
     *
     * @param innerVersion
     *            la version entrante.
     */
    @Override
    public void setInnerVersion(Version innerVersion) {
        InnerVersion = innerVersion;
    }

    /**
     * Retourne la version en sortie.
     *
     * @return la version en sortie.
     */
    @Override
    public Version getOuterVersion() {
        return OuterVersion;
    }

    /**
     * Modificateur de la version sortante.
     *
     * @param outerVersion
     *            la version sortante.
     */
    @Override
    public void setOuterVersion(Version outerVersion) {
        OuterVersion = outerVersion;
    }

    /**
     * Méthode pour basculer la version sortante sur la version entrante.
     */
    @Override
    public void switchVersion() {
        InnerVersion = OuterVersion;
        InnerMap.clear();
        InnerMap.putAll(OuterMap);
        InnerClass = OuterClass.clone();

        OuterVersion = null;
        OuterMap.clear();
        OuterClass = null;
    }

    /**
     * Calcul le code de hashage.
     *
     * @return le hash code.
     */
    @Override
    public int hashCode() {
        return this.idInner.hashCode();
    }

    /**
     * Comparaison de maps.
     *
     * @param obj
     *            un autre objet.
     * @return l'égalité des deux objets.
     */
    @Override
    public boolean equals(Object obj) {
        if (obj instanceof MapAdapter) {
            MapAdapter other = (MapAdapter) obj;
            return (this.idInner.equals(other));
        }
        return false;
    }

    /**
     * Retourne l'identifiant de l'objet.
     *
     * @return l'identifiant.
     */
    @Override
    public Serializable getIdInner() {
        return idInner;
    }

    /**
     * Recopie un attribut
     *
     * @param nameAtt
     *            le nom de l'attribut
     */
    @Override
    public void copy(String nameAtt) {
        if (InnerMap.containsKey(nameAtt)) {
            OuterMap.put(nameAtt, InnerMap.get(nameAtt));
        } else {
            OuterMap.put(nameAtt, null);
        }
    }

    /**
     * Recopie tous les attributs.
     */
    @Override
    public void copyAll() {
        Set<String> s = InnerMap.keySet();
        for (Iterator<String> iter = s.iterator(); iter.hasNext();) {
            String element = iter.next();
            OuterMap.put(element, InnerMap.get(element));
        }
    }

    /**
     * Suppression d'un attribut de la map de sortie.
     *
     * @param nameAtt
     *            Nom de l'attribut.
     * @throws ExceptionAttributeUndefined
     */
    @Override
    public void delete(String nameAtt) throws ExceptionAttributeUndefined {
        if (OuterMap.containsKey(nameAtt)) {
            OuterMap.remove(nameAtt);
        } else {
            throw new ExceptionAttributeUndefined();
        }
    }

    /**
     * Suppression de tous les attributs de la map de sortie.
     */
    @Override
    public void deleteAll() {
        OuterMap.clear();
    }

    /**
     * Modificateur de la classe sortante.
     *
     * @param outerClass
     *            La classe de sortie.
     */
    @Override
    public void setOuterClass(ProxyClass outerClass) {
        OuterClass = outerClass;
    }

    @Override
    public String toString() {
        return "Inner => " + this.InnerClass + ":" + this.InnerMap.toString() + "\n" + "Outer => " + this.OuterClass + ":" + this.OuterMap.toString();
    }

    @Override
    public void setOldValue(String nameAttribut, Serializable value) {
        InnerMap.put(nameAttribut, value);
    }
}
