package org.nuiton.topia.persistence.internal.support;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: TopiaServiceSupportImpl.java 3088 2014-04-30 11:17:38Z bleny $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-beta-4/topia-persistence/src/main/java/org/nuiton/topia/persistence/internal/support/TopiaServiceSupportImpl.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.topia.persistence.TopiaService;
import org.nuiton.topia.persistence.internal.AbstractTopiaApplicationContext;
import org.nuiton.topia.persistence.support.TopiaServiceSupport;

import java.lang.reflect.InvocationTargetException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * FIXME AThimel 11/10/13 All this life-cycle has to be reviewed
 *
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class TopiaServiceSupportImpl implements TopiaServiceSupport {

    private static final Log log = LogFactory.getLog(TopiaServiceSupportImpl.class);

    protected ImmutableMap<String, TopiaService> services;

    protected AbstractTopiaApplicationContext topiaApplicationContext;

    public TopiaServiceSupportImpl(AbstractTopiaApplicationContext topiaApplicationContext) {
        this.topiaApplicationContext = topiaApplicationContext;
        this.services = loadServices(topiaApplicationContext.getConfiguration());
    }

    public void init() {
        preInitServices(this.services);
        topiaApplicationContext.getHibernateProvider().getHibernateConfiguration(); // force mapping loading
        postInitServices(this.services);
    }

    protected void preInitServices(Map<String, TopiaService> services) {
        for (TopiaService service : services.values()) {
            service.preInit(topiaApplicationContext);
        }
    }

    protected void postInitServices(Map<String, TopiaService> services) {
        for (TopiaService service : services.values()) {
            service.postInit(topiaApplicationContext);
        }
    }

    protected ImmutableMap<String, TopiaService> loadServices(ImmutableMap<String, String> configuration) {
        // recherche des services present dans la config
        Map<String, TopiaService> services = new HashMap<String, TopiaService>();
        Map<String, Map<String, String>> servicesConfigurations = new HashMap<String, Map<String, String>>();
        for (Map.Entry<String, String> entry : configuration.entrySet()) {
            String prefixedConfigurationKey = entry.getKey();
            String configurationValue = entry.getValue();
            if (prefixedConfigurationKey.startsWith("topia.service.")) {
                String configurationKey = StringUtils.removeStart(prefixedConfigurationKey, "topia.service.");
                String[] split = StringUtils.split(configurationKey, '.');
                Preconditions.checkState(split.length > 0, "'" + prefixedConfigurationKey + "' is not a valid configuration key");
                String serviceName = split[0];
                if (split.length == 1) {
                    // service declaration instantiate it
                    try {
                        Class<?> forName = Class.forName(configurationValue);
                        Object newInstance = forName.getConstructor().newInstance();
                        TopiaService service = (TopiaService) newInstance;
                        services.put(serviceName, service);
                        if (log.isInfoEnabled()) {
                            log.info("instantiated service " + serviceName + ": " + service);
                        }
                    } catch (ClassNotFoundException e) {
                        throw new TopiaException("unable to find topia service class " + configurationValue, e);
                    } catch (InstantiationException e) {
                        throw new TopiaException("unable to instantiate class " + configurationValue, e);
                    } catch (IllegalAccessException e) {
                        throw new TopiaException("unable to instantiate class " + configurationValue, e);
                    } catch (InvocationTargetException e) {
                        throw new TopiaException("unable to instantiate class " + configurationValue, e);
                    } catch (NoSuchMethodException e) {
                        throw new TopiaException("unable to instantiate class " + configurationValue, e);
                    }
                } else {
                    // service configuration, save it to push it back later
                    Map<String, String> serviceConfiguration = servicesConfigurations.get(serviceName);
                    if (serviceConfiguration == null) {
                        serviceConfiguration = new LinkedHashMap<String, String>();
                        servicesConfigurations.put(serviceName, serviceConfiguration);
                    }
                    String serviceConfigurationKey = StringUtils.removeStart(configurationKey, serviceName + ".");
                    serviceConfiguration.put(serviceConfigurationKey, configurationValue);
                }
            }
        }
        // configure services
        for (Map.Entry<String, TopiaService> entry : services.entrySet()) {
            String serviceName = entry.getKey();
            TopiaService service = entry.getValue();
            Map<String, String> serviceConfiguration = servicesConfigurations.get(serviceName);
            if (serviceConfiguration == null) {
                serviceConfiguration = Collections.emptyMap();
            }
            if (log.isInfoEnabled()) {
                log.info("for service " + serviceName + ", will set configuration " + serviceConfiguration);
            }
            service.setConfiguration(serviceConfiguration);
        }
        return ImmutableMap.copyOf(services);
    }

    @Override
    public Map<String, TopiaService> getServices() {
        return services;
    }

    @Override
    public <T extends TopiaService> Map<String, T> getServices(Class<T> interfaceService) {
        Map<String, T> result = new LinkedHashMap<String, T>();
        for (Map.Entry<String, TopiaService> entry : services.entrySet()) {
            String serviceName = entry.getKey();
            TopiaService service = entry.getValue();
            if (interfaceService.isAssignableFrom(service.getClass())) {
                result.put(serviceName, (T) service);
            }
        }
        return result;
    }
}
