package org.nuiton.topia.persistence.internal.support;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: TopiaFiresSupport.java 3002 2014-02-04 16:35:06Z athimel $
 * $HeadURL: https://svn.nuiton.org/topia/tags/topia-3.0-alpha-12/topia-persistence/src/main/java/org/nuiton/topia/persistence/internal/support/TopiaFiresSupport.java $
 * %%
 * Copyright (C) 2004 - 2014 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.beans.VetoableChangeSupport;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.support.TopiaJpaSupport;
import org.nuiton.topia.persistence.support.TopiaListenableSupport;
import org.nuiton.topia.persistence.TopiaPersistenceContext;
import org.nuiton.topia.persistence.TopiaVetoException;
import org.nuiton.topia.persistence.event.TopiaContextEvent;
import org.nuiton.topia.persistence.event.TopiaEntitiesEvent;
import org.nuiton.topia.persistence.event.TopiaEntitiesVetoable;
import org.nuiton.topia.persistence.event.TopiaEntityEvent;
import org.nuiton.topia.persistence.event.TopiaEntityListener;
import org.nuiton.topia.persistence.event.TopiaEntityVetoable;
import org.nuiton.topia.persistence.event.TopiaSchemaListener;
import org.nuiton.topia.persistence.event.TopiaTransactionEvent;
import org.nuiton.topia.persistence.event.TopiaTransactionListener;
import org.nuiton.topia.persistence.event.TopiaTransactionVetoable;
import org.nuiton.topia.persistence.event.EntityState;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.CategorisedListenerSet;
import org.nuiton.util.ListenerSet;

/**
 * TODO-fdesbois-20100507 : Need translation of javadoc.
 * <p/>
 * Contient l'ensemble de la partie listener et vetoable c'est à dire la
 * gestion, les fires, ...
 *
 * @author jruchaud <jruchaud@codelutin.com>
 */
public class TopiaFiresSupport implements TopiaListenableSupport {

    private static final Log log = LogFactory.getLog(TopiaFiresSupport.class);

    /** used to fire read event */
    final static Object NO_CHANGE = new Object();

    /** used to collect entity modification during transaction */
    protected Map<TopiaEntity, EntityState> transactionEntities = new IdentityHashMap<TopiaEntity, EntityState>();

    protected Set<PropertyChangeListener> propertyChangeListeners = new HashSet<PropertyChangeListener>();

    /* Pour la transaction */
    protected ListenerSet<TopiaTransactionListener> transactionListeners = new ListenerSet<TopiaTransactionListener>();

    protected ListenerSet<TopiaTransactionVetoable> transactionVetoables = new ListenerSet<TopiaTransactionVetoable>();

    /* Pour les entités */
    protected CategorisedListenerSet<TopiaEntityListener> entityListeners = new CategorisedListenerSet<TopiaEntityListener>();

    protected CategorisedListenerSet<TopiaEntityVetoable> entityVetoables = new CategorisedListenerSet<TopiaEntityVetoable>();

    /* Pour les listes d'entités */
    protected ListenerSet<TopiaEntitiesVetoable> entitiesVetoables = new ListenerSet<TopiaEntitiesVetoable>();

    /* Pour les actions du topia context */
    protected ListenerSet<TopiaSchemaListener> topiaSchemaListeners = new ListenerSet<TopiaSchemaListener>();

    /**
     * used to register objects loaded during transaction.
     *
     * @param entity the loaded entity
     */
    public void warnOnLoadEntity(TopiaEntity entity) {
        if (log.isDebugEnabled()) {
            log.debug("warnOnReadEntity");
        }
        EntityState state = transactionEntities.get(entity);
        if (state == null) {
            state = new EntityState();
            transactionEntities.put(entity, state);
        }
        state.addLoad();
    }

    /**
     * used to register objects created during transaction.
     *
     * @param entity the created entity
     */
    public void warnOnCreateEntity(TopiaEntity entity) {
        if (log.isDebugEnabled()) {
            log.debug("warnOnCreateEntity");
        }
        EntityState state = transactionEntities.get(entity);
        if (state == null) {
            state = new EntityState();
            transactionEntities.put(entity, state);
        }
        state.addCreate();
    }

    /**
     * used to register objects loaded during transaction.
     *
     * @param entity the read entity
     */
    public void warnOnReadEntity(TopiaEntity entity) {
        if (log.isDebugEnabled()) {
            log.debug("warnOnReadEntity");
        }
        EntityState state = transactionEntities.get(entity);
        if (state == null) {
            state = new EntityState();
            transactionEntities.put(entity, state);
        }
        state.addRead();
    }

    /**
     * used to register objects modified during transaction.
     *
     * @param entity the updated entity
     */
    public void warnOnUpdateEntity(TopiaEntity entity) {
        if (log.isDebugEnabled()) {
            log.debug("warnOnUpdateEntity");
        }

        EntityState state = transactionEntities.get(entity);
        if (state == null) {
            state = new EntityState();
            transactionEntities.put(entity, state);
        }
        state.addUpdate();
    }

    /**
     * used to register objects deleted during transaction.
     *
     * @param entity the deleted entity
     */
    public void warnOnDeleteEntity(TopiaEntity entity) {
        if (log.isDebugEnabled()) {
            log.debug("warnOnDeleteEntity");
        }
        EntityState state = transactionEntities.get(entity);
        if (state == null) {
            state = new EntityState();
            transactionEntities.put(entity, state);
        }
        state.addDelete();
    }

    /* Fires sur les transactions */

    public void fireOnBeginTransaction(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnBeginTransaction");
        }
        TopiaTransactionEvent e = new TopiaTransactionEvent(context);
        for (TopiaTransactionVetoable listener : transactionVetoables) {
            try {
                listener.beginTransaction(e);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    public void fireOnPostCommit(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPostCommit");
        }
        TopiaTransactionEvent e = new TopiaTransactionEvent(context, transactionEntities);
        for (TopiaTransactionListener listener : transactionListeners) {
            try {
                listener.commit(e);
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't fireOnPostCommit", eee);
                }
            }
        }
        transactionEntities.clear();
    }

    public void fireOnPostRollback(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPostRollback");
        }
        TopiaTransactionEvent e = new TopiaTransactionEvent(context, transactionEntities);
        for (TopiaTransactionListener listener : transactionListeners) {
            try {
                listener.rollback(e);
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't fireOnPostRollback", eee);
                }
            }
        }
        transactionEntities.clear();
    }

    /* Fires sur les entités */

    public void fireOnPreCreate(TopiaPersistenceContext context,
                                TopiaEntity entity, Object[] state) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPreCreate");
        }
        TopiaEntityEvent event = new TopiaEntityEvent(context, entity, state);
        for (Iterator<TopiaEntityVetoable> l = entityVetoables.iterator(entity.getClass()); l.hasNext();) {
            try {
                l.next().create(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    public void fireOnPostCreate(TopiaPersistenceContext context,
                                 TopiaEntity entity, Object[] state) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPostCreate");
        }
        warnOnCreateEntity(entity);
        TopiaEntityEvent event = new TopiaEntityEvent(context, entity, state);
        for (Iterator<TopiaEntityListener> l = entityListeners.iterator(entity.getClass()); l.hasNext();) {
            try {
                l.next().create(event);
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't fireOnPostCreate for entity: " + entity, eee);
                }
            }
        }
    }

    public void fireOnPreLoad(TopiaPersistenceContext context,
                              TopiaEntity entity, Object[] state) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPreLoad");
        }
        TopiaEntityEvent event = new TopiaEntityEvent(context, entity, state);
        for (Iterator<TopiaEntityVetoable> l = entityVetoables.iterator(entity .getClass()); l.hasNext();) {
            try {
                l.next().load(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    public void fireOnPostLoad(TopiaPersistenceContext context,
                               TopiaEntity entity, Object[] state) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPostLoad");
        }
        warnOnLoadEntity(entity);
        TopiaEntityEvent event = new TopiaEntityEvent(context, entity, state);
        for (Iterator<TopiaEntityListener> l = entityListeners.iterator(entity.getClass()); l.hasNext();) {
            try {
                l.next().load(event);
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't fireOnPostLoad for entity: " + entity, eee);
                }
            }
        }
    }

    public void fireOnPreUpdate(TopiaPersistenceContext context,
                                TopiaEntity entity, Object[] state) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPreUpdate");
        }
        TopiaEntityEvent event = new TopiaEntityEvent(context, entity, state);
        for (Iterator<TopiaEntityVetoable> l = entityVetoables.iterator(entity.getClass()); l.hasNext();) {
            try {
                l.next().update(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    public void fireOnPostUpdate(TopiaPersistenceContext context,
                                 TopiaEntity entity, Object[] state) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPostUpdate");
        }
        warnOnUpdateEntity(entity);
        TopiaEntityEvent event = new TopiaEntityEvent(context, entity, state);
        for (Iterator<TopiaEntityListener> l = entityListeners.iterator(entity.getClass()); l.hasNext();) {
            try {
                l.next().update(event);
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't fireOnPostUpdate for entity: " + entity, eee);
                }
            }
        }
    }

    public void fireOnPreDelete(TopiaPersistenceContext context,
                                TopiaEntity entity, Object[] state) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPreDelete");
        }
        TopiaEntityEvent event = new TopiaEntityEvent(context, entity, state);
        for (Iterator<TopiaEntityVetoable> l = entityVetoables.iterator(entity.getClass()); l.hasNext();) {
            try {
                l.next().delete(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    public void fireOnPostDelete(TopiaPersistenceContext context,
                                 TopiaEntity entity, Object[] state) {
        if (log.isDebugEnabled()) {
            log.debug("fireOnPostDelete");
        }
        warnOnDeleteEntity(entity);
        TopiaEntityEvent event = new TopiaEntityEvent(context, entity, state);
        for (Iterator<TopiaEntityListener> l = entityListeners.iterator(entity.getClass()); l.hasNext();) {
            try {
                l.next().delete(event);
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't fireOnPostDelete for entity: " + entity, eee);
                }
            }
        }
    }

    /* Fires sur les propriétés */

    public void fireOnPreRead(VetoableChangeSupport vetoables,
                              TopiaEntity entity,
                              String propertyName,
                              Object value) {

        if (log.isDebugEnabled()) {
            log.debug("fireOnPreRead");
        }
        try {
            vetoables.fireVetoableChange(propertyName, value, NO_CHANGE);
        } catch (Exception eee) {
            throw new TopiaVetoException(eee);
        }
    }

    public void fireOnPostRead(PropertyChangeSupport listeners,
                               TopiaEntity entity, String propertyName,
                               Object value) {

        if (log.isDebugEnabled()) {
            log.debug("fireOnPostRead");
        }
        warnOnReadEntity(entity);
        try {
            listeners.firePropertyChange(propertyName, value, NO_CHANGE);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't fireOnPostRead", eee);
            }
        }

    }

    public void fireOnPostRead(PropertyChangeSupport listeners,
                               TopiaEntity entity,
                               String propertyName,
                               int index,
                               Object value) {

        if (log.isDebugEnabled()) {
            log.debug("fireOnPostRead");
        }
        warnOnReadEntity(entity);
        try {
            listeners.fireIndexedPropertyChange(propertyName, index, value, NO_CHANGE);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't fireOnPostRead", eee);
            }
        }

    }

    public void fireOnPreWrite(VetoableChangeSupport vetoables,
                               TopiaEntity entity,
                               String propertyName,
                               Object oldValue,
                               Object newValue) {

        if (log.isDebugEnabled()) {
            log.debug("fireOnPreWrite");
        }
        try {
            vetoables.fireVetoableChange(propertyName, oldValue, newValue);
        } catch (Exception eee) {
            throw new TopiaVetoException(eee);
        }
    }

    public void fireOnPostWrite(PropertyChangeSupport listeners,
                                TopiaEntity entity,
                                String propertyName,
                                Object oldValue,
                                Object newValue) {

        if (log.isDebugEnabled()) {
            log.debug("fireOnPostWrite");
        }
        warnOnUpdateEntity(entity);
        if (propertyChangeListeners.size() > 0) {
            PropertyChangeEvent e = new PropertyChangeEvent(entity, propertyName, oldValue, newValue);
            for (PropertyChangeListener l : propertyChangeListeners) {
                try {
                    l.propertyChange(e);
                } catch (Exception eee) {
                    if (log.isErrorEnabled()) {
                        log.error("Can't fire property change for: " + propertyName, eee);
                    }
                }
            }
        }
        try {
            listeners.firePropertyChange(propertyName, oldValue, newValue);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't fireOnPostWrite: " + propertyName, eee);
            }
        }
    }

    public void fireOnPostWrite(PropertyChangeSupport listeners,
                                TopiaEntity entity,
                                String propertyName,
                                int index,
                                Object oldValue,
                                Object newValue) {

        if (log.isDebugEnabled()) {
            log.debug("fireOnPostWrite");
        }
        warnOnUpdateEntity(entity);
        try {
            listeners.fireIndexedPropertyChange(propertyName, index, oldValue, newValue);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't fireOnPostWrite", eee);
            }
        }
    }

    /**
     * Notify topia context listeners for create schema pre operation
     *
     * @param context topia context
     */
    public void firePreCreateSchema(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("firePreCreateSchema");
        }
        TopiaContextEvent event = new TopiaContextEvent(context);
        for (TopiaSchemaListener topiaSchemaListener : topiaSchemaListeners) {
            try {
                topiaSchemaListener.preCreateSchema(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    /**
     * Notify topia context listeners for create schema post operation
     *
     * @param context topia context
     */
    public void firePostCreateSchema(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("firePostCreateSchema");
        }
        TopiaContextEvent event = new TopiaContextEvent(context);
        for (TopiaSchemaListener topiaSchemaListener : topiaSchemaListeners) {
            try {
                topiaSchemaListener.postCreateSchema(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    /**
     * Notify topia context listeners for create schema pre operation
     *
     * @param context topia context
     */
    public void firePreUpdateSchema(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("firePostCreateSchema");
        }
        TopiaContextEvent event = new TopiaContextEvent(context);
        for (TopiaSchemaListener topiaSchemaListener : topiaSchemaListeners) {
            try {
                topiaSchemaListener.preUpdateSchema(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    /**
     * Notify topia context listeners for create schema post operation
     *
     * @param context topia context
     */
    public void firePostUpdateSchema(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("firePostCreateSchema");
        }
        TopiaContextEvent event = new TopiaContextEvent(context);
        for (TopiaSchemaListener topiaSchemaListener : topiaSchemaListeners) {
            try {
                topiaSchemaListener.postUpdateSchema(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    /**
     * Notify topia context listeners for schema restore pre operation
     *
     * @param context topia context
     */
    public void firePreRestoreSchema(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("firePreRestoreSchema");
        }
        TopiaContextEvent event = new TopiaContextEvent(context);
        for (TopiaSchemaListener topiaSchemaListener : topiaSchemaListeners) {
            try {
                topiaSchemaListener.preRestoreSchema(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    /**
     * Notify topia context listeners for schema restore post operation
     *
     * @param context topia context
     */
    public void firePostRestoreSchema(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("firePostRestoreSchema");
        }
        TopiaContextEvent event = new TopiaContextEvent(context);
        for (TopiaSchemaListener topiaSchemaListener : topiaSchemaListeners) {
            try {
                topiaSchemaListener.postRestoreSchema(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    /**
     * Notify topia context listeners for drop schema pre operation
     *
     * @param context topia context
     * @since 3.0
     */
    public void firePreDropSchema(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("firePreDropSchema");
        }
        TopiaContextEvent event = new TopiaContextEvent(context);
        for (TopiaSchemaListener topiaSchemaListener : topiaSchemaListeners) {
            try {
                topiaSchemaListener.preDropSchema(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }

    /**
     * Notify topia context listeners for drop schema post operation
     *
     * @param context topia context
     * @since 3.0
     */
    public void firePostDropSchema(TopiaPersistenceContext context) {
        if (log.isDebugEnabled()) {
            log.debug("firePostDropSchema");
        }
        TopiaContextEvent event = new TopiaContextEvent(context);
        for (TopiaSchemaListener topiaSchemaListener : topiaSchemaListeners) {
            try {
                topiaSchemaListener.postDropSchema(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
    }


    /**
     * Notify entities listeners for load operation
     *
     * @param <E>      type of entities
     * @param context  context used
     * @param entities entities loaded
     * @return the list of entities loaded
     */
    public <E extends TopiaEntity> List<E> fireEntitiesLoad(
            TopiaJpaSupport context, List<E> entities) {
        if (log.isDebugEnabled()) {
            log.debug("fireEntitiesLoad");
        }

        List<E> result = entities;
        for (TopiaEntitiesVetoable entitiesVetoable : entitiesVetoables) {
            try {
                //FIXME tchemit 20100513 Why instanciate n events? if necessary MUST add a comment
                TopiaEntitiesEvent<E> event = new TopiaEntitiesEvent<E>(context, result);
                result = entitiesVetoable.load(event);
            } catch (Exception eee) {
                throw new TopiaVetoException(eee);
            }
        }
        return result;
    }

    /* Getters */

    public CategorisedListenerSet<TopiaEntityListener> getEntityListeners() {
        return entityListeners;
    }

    public CategorisedListenerSet<TopiaEntityVetoable> getEntityVetoables() {
        return entityVetoables;
    }

    public ListenerSet<TopiaTransactionListener> getTransactionListeners() {
        return transactionListeners;
    }

    public ListenerSet<TopiaTransactionVetoable> getTransactionVetoable() {
        return transactionVetoables;
    }

    public ListenerSet<TopiaSchemaListener> getTopiaSchemaListeners() {
        return topiaSchemaListeners;
    }

    public ListenerSet<TopiaEntitiesVetoable> getTopiaEntitiesVetoable() {
        return entitiesVetoables;
    }

    /* Adders */

    public void addTopiaEntityListener(TopiaEntityListener listener) {
        addTopiaEntityListener(TopiaEntity.class, listener);
    }

    public void addTopiaEntityListener(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener can not be null.");
        }
        entityListeners.add(entityClass, listener);
    }

    public void addTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        addTopiaEntityVetoable(TopiaEntity.class, vetoable);
    }

    public void addTopiaEntityVetoable(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityVetoable vetoable) {
        if (vetoable == null) {
            throw new NullPointerException("listener can not be null.");
        }
        entityVetoables.add(entityClass, vetoable);
    }

    public void addTopiaTransactionListener(TopiaTransactionListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener can not be null.");
        }
        transactionListeners.add(listener);
    }

    public void addTopiaTransactionVetoable(TopiaTransactionVetoable vetoable) {
        if (vetoable== null) {
            throw new NullPointerException("listener can not be null.");
        }
        transactionVetoables.add(vetoable);
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener can not be null.");
        }
        propertyChangeListeners.add(listener);
    }

    public void addTopiaSchemaListener(TopiaSchemaListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener can not be null.");
        }
        topiaSchemaListeners.add(listener);
    }

    public void addTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable) {
        if (vetoable == null) {
            throw new NullPointerException("listener can not be null.");
        }
        entitiesVetoables.add(vetoable);
    }

    /* Removers */

    public void removeTopiaEntityListener(TopiaEntityListener listener) {
        removeTopiaEntityListener(TopiaEntity.class, listener);
    }

    public void removeTopiaEntityListener(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener can not be null.");
        }
        entityListeners.remove(entityClass, listener);
    }

    public void removeTopiaEntityVetoable(TopiaEntityVetoable vetoable) {
        removeTopiaEntityVetoable(TopiaEntity.class, vetoable);
    }

    public void removeTopiaEntityVetoable(
            Class<? extends TopiaEntity> entityClass,
            TopiaEntityVetoable vetoable) {
        if (vetoable == null) {
            throw new NullPointerException("listener can not be null.");
        }
        entityVetoables.remove(entityClass, vetoable);
    }

    public void removeTopiaTransactionListener(TopiaTransactionListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener can not be null.");
        }
        transactionListeners.remove(listener);
    }

    public void removeTopiaTransactionVetoable(TopiaTransactionVetoable vetoable) {
        if (vetoable == null) {
            throw new NullPointerException("listener can not be null.");
        }
        transactionVetoables.remove(vetoable);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener can not be null.");
        }
        propertyChangeListeners.remove(listener);
    }

    public void removeTopiaSchemaListener(TopiaSchemaListener listener) {
        if (listener == null) {
            throw new NullPointerException("listener can not be null.");
        }
        topiaSchemaListeners.remove(listener);
    }

    public void removeTopiaEntitiesVetoable(TopiaEntitiesVetoable vetoable) {
        if (vetoable == null) {
            throw new NullPointerException("listener can not be null.");
        }
        entitiesVetoables.remove(vetoable);
    }

}
