/*
 * *##% 
 * JAXX Action
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
/* 
* ##% Copyright (C) 2007, 2008 Code Lutin, Tony Chemit
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
* ##% */
package org.nuiton.jaxx.action;

import jaxx.runtime.JAXXObject;
import jaxx.runtime.swing.JAXXToggleButton;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.Resource;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import java.awt.event.ActionEvent;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.TreeMap;

/**
 * A simple implementation of {@link ActionFactory} using some {@link ActionProvider} to seek actions.
 * <p/>
 * <p/>
 * An entry is in that form : <code>action.actionName=fqn</code> where
 * <p/>
 * <code>actionName</code> is the key of action used in factory, and
 * <code>fqn</code> is the fully qualified name of the implemented action class.
 * <p/>
 * A special clase is to have for a given entry a key like this : <code>action.:fqn'=fqn</code>, in that case,
 * le fqn' is a classe of type {@link org.nuiton.jaxx.action.ActionNameProvider} which gives us at
 * runtime the names of each entry to put in cache for the givne action fqn.
 *
 * @param <A> the type of action
 * @author chemit
 */
public class ActionFactoryFromProvider<A extends MyAbstractAction> implements ActionFactory<A> {

    protected static Log log = LogFactory.getLog(ActionFactoryFromProvider.class);

    public static <A extends MyAbstractAction> ActionFactory<A> newInstance(Class<A> klazz) {
        return new ActionFactoryFromProvider<A>(klazz);
    }

    /** class of encapsuling action */
    protected Class<A> baseImpl;

    /** dictionary of known actions implementations */
    private Map<String, Class<? extends MyAbstractAction>> impls;

    /** dictionary of instanciated actions */
    private Map<String, A> cache;

    protected final ActionConfigConfigurationResolver actionConfigInitializer;
    protected final ToggleActionConfigConfigurationResolver toggleActionConfigInitializer;
    protected final SelectActionConfigConfigurationResolver selectActionConfigInitializer;

    protected List<AbstractActionConfigurationResolver> configurationResolvers;

    protected ActionFactoryFromProvider(Class<A> baseImpl) {
        this.baseImpl = baseImpl;
        this.impls = init();
        this.cache = new TreeMap<String, A>();
        this.configurationResolvers = new java.util.ArrayList<AbstractActionConfigurationResolver>();

        this.toggleActionConfigInitializer = registerInitializer(ToggleActionConfigConfigurationResolver.class);
        this.actionConfigInitializer = registerInitializer(ActionConfigConfigurationResolver.class);
        this.selectActionConfigInitializer = registerInitializer(SelectActionConfigConfigurationResolver.class);
    }

    @Override
    public Class<A> getBaseClass() {
        return baseImpl;
    }

    @Override
    public void resetCache() {
        cache.clear();
    }

    /*public A get(String actionKey) {
        return cache.get(actionKey);
    }*/

    @Override
    public void loadActions(JAXXObject ui) {
        if (log.isDebugEnabled()) {
            log.debug("for ui " + ui.getClass());
        }
        for (Map.Entry<String, Class<? extends MyAbstractAction>> entry : implsEntrySet()) {
            String actionKey = entry.getKey();
            Object comp = ui.getObjectById(actionKey);
            if (comp == null || !(comp instanceof AbstractButton || comp instanceof JComboBox)) {
                // nothing to do
                continue;
            }
            if (log.isTraceEnabled()) {
                log.trace("detect action " + actionKey);
            }
            if (comp instanceof AbstractButton) {
                AbstractButton component = (AbstractButton) comp;
                A action = newAction(actionKey, component);

                component.setAction(action);

                if (component instanceof JAXXToggleButton) {
                    JAXXToggleButton glueComponent = (JAXXToggleButton) component;
                    glueComponent.setIcon((Icon) action.getValue(Action.SMALL_ICON));
                    Integer integer = (Integer) action.getValue(Action.MNEMONIC_KEY);
                    if (integer != null) {
                        glueComponent.setNormalMnemonic(integer);
                    }
                    glueComponent.setSelectedIcon((Icon) action.getValue(Action.SMALL_ICON + 2));
                    integer = (Integer) action.getValue(Action.MNEMONIC_KEY + 2);
                    if (integer != null) {
                        glueComponent.setGlueMnemonic(integer);
                    }
                    glueComponent.setGlueText((String) action.getValue(Action.NAME + 2));
                    glueComponent.setGlueTooltipText((String) action.getValue(Action.SHORT_DESCRIPTION + 2));

                    glueComponent.setNormalText((String) action.getValue(Action.NAME));
                    glueComponent.setNormalTooltipText((String) action.getValue(Action.SHORT_DESCRIPTION));
                }

                Boolean value = (Boolean) action.getValue("hideActionText");
                component.setHideActionText(value != null && value);
                action.setEnabled(true);
                continue;
            }
            // is JComboBox
            JComboBox component = (JComboBox) comp;
            A action = newAction(actionKey, component);

            component.setAction(action);
            Integer val = (Integer) action.getValue("selectedIndex");
            if (val != null && val != -1 && val < component.getItemCount() && val != component.getSelectedIndex()) {
                component.setSelectedIndex(val);
            }
        }
    }

    /**
     * @param actionKey le nom de l'action tel que définie dans le fichier
     *                  de mapping (sans le prefix action.)
     * @param component le button où rattacher l'action
     * @return une nouvelle instance de l'action associée à sa clef.
     */
    @Override
    public A newAction(String actionKey, JComponent component) {
        // try first in cache
        A result = getActionFromCache(actionKey);
        if (result != null) {
            return result;
        }

        try {
            result = newActionInstance(actionKey);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        if (log.isDebugEnabled()) {
            log.debug("create <" + actionKey + " : " + result + ">");
        }

        // recherche de l'annotation de configuration
        ActionConfigurationResolver<?, ?> configurationResolver = resolveActionConfiguration(result);

        if (configurationResolver != null) {
            configurationResolver.applyConfiguration(component, result);
        }

        try {

            if (configurationResolver != null) {
                if (AbstractButton.class.isAssignableFrom(configurationResolver.getComponentImpl())) {
                    finalizeNewAction((AbstractButton) component, result, configurationResolver);
                }

                if (JComboBox.class.isAssignableFrom(configurationResolver.getComponentImpl())) {
                    finalizeNewAction((JComboBox) component, result, configurationResolver);
                }

                return result;
            }

            if (component == null || component instanceof AbstractButton) {
                finalizeNewAction((AbstractButton) component, result, configurationResolver);
                return result;
            }

            if (component instanceof JComboBox) {
                finalizeNewAction((JComboBox) component, result, configurationResolver);
            }
        } finally {
            // save result in cache
            cache.put(actionKey, result);
        }

        return result;
    }

    @Override
    public A newAction(String actionKey) {
        return newAction(actionKey, null);
    }

    @Override
    public String[] getActionNames() {
        return impls.keySet().toArray(new String[impls.size()]);
    }

    @Override
    public Set<Entry<String, Class<? extends MyAbstractAction>>> implsEntrySet() {
        return impls.entrySet();
    }

    @Override
    public Set<Entry<String, A>> cacheEntrySet() {
        return cache.entrySet();
    }

    @Override
    public void fireAction(String actionKey, Object source, JComponent component) {
        A action = newAction(actionKey, component);
        fireAction0(actionKey, source, action);
    }

    @Override
    public void fireAction(String actionKey, Object source) {
        fireAction(actionKey, source, null);
    }

    /**
     * @param actionKey la clef de l'action
     * @return l'action deja stockee dans le cache d'action, ou <code>null</code> si non trouvée.
     */
    @Override
    public A getActionFromCache(String actionKey) {
        // on vérifie que l'action existe bien
        checkRegistredAction(actionKey);

        A action = null;
        // try in cache
        if (cache.containsKey(actionKey)) {
            // use cached action
            action = cache.get(actionKey);
            if (log.isDebugEnabled()) {
                log.debug("use cache action " + action);
            }
        }
        return action;
    }

    @Override
    public void dispose() {
        if (log.isInfoEnabled()) {
            log.info(this);
        }
        for (String actionKey : getActionNames()) {
            MyAbstractAction action = getActionFromCache(actionKey);
            if (action != null) {
                action.disposeUI();
            }
        }
        resetCache();
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        resetCache();
        impls.clear();
    }

    /**
     * @param component             le button où rattacher l'action
     * @param action                action
     * @param configurationResolver initializer
     */
    protected void finalizeNewAction(AbstractButton component, MyAbstractAction action, ActionConfigurationResolver<?, ?> configurationResolver) {

        if (configurationResolver == null) {
            // no configurationResolver matching,
            if (component != null) {
                action.putValue(Action.ACTION_COMMAND_KEY, component.getName());
                action.putValue(Action.SHORT_DESCRIPTION, component.getToolTipText());
                action.putValue(Action.SMALL_ICON, component.getIcon());
                action.putValue(Action.NAME, component.getText());
                action.putValue(Action.MNEMONIC_KEY, component.getMnemonic());
                action.putValue("hideActionText", component.getHideActionText());
                if (component instanceof JAXXToggleButton) {
                    JAXXToggleButton glueComponent = (JAXXToggleButton) component;
                    action.putValue(Action.SHORT_DESCRIPTION, glueComponent.getNormalTooltipText());
                    action.putValue(Action.NAME, glueComponent.getNormalText());
                    action.putValue(Action.SMALL_ICON, glueComponent.getIcon());
                    action.putValue(Action.MNEMONIC_KEY, glueComponent.getNormalMnemonic());
                    action.putValue(Action.SHORT_DESCRIPTION + 2, glueComponent.getGlueTooltipText());
                    action.putValue(Action.NAME + 2, glueComponent.getGlueText());
                    action.putValue(Action.SMALL_ICON + 2, glueComponent.getSelectedIcon());
                    action.putValue(Action.MNEMONIC_KEY + 2, glueComponent.getGlueMnemonic());
                }
            }

        }

        String text = (String) action.getValue(Action.NAME);
        Integer mnemo = (Integer) action.getValue(Action.MNEMONIC_KEY);
        if (mnemo != null && mnemo != '\0') {
            int pos = text.indexOf((char) mnemo.intValue());
            if (pos == -1) {
                pos = text.indexOf(Character.toLowerCase((char) mnemo.intValue()));
            }
            action.putValue(Action.DISPLAYED_MNEMONIC_INDEX_KEY, pos);
        }

    }

    /**
     * @param component             le select box où rattacher l'action
     * @param action                action
     * @param configurationResolver initializer
     */
    protected void finalizeNewAction(JComboBox component, MyAbstractAction action, ActionConfigurationResolver<?, ?> configurationResolver) {

        if (configurationResolver == null) {
            action.putValue(Action.ACTION_COMMAND_KEY, component.getName());
            action.putValue(Action.SHORT_DESCRIPTION, component.getToolTipText());
            //result.putValue("selectedIndex", component.getSelectedIndex());
        }

    }

    protected ActionConfigurationResolver resolveActionConfiguration(MyAbstractAction action) {
        for (ActionConfigurationResolver resolver : configurationResolvers) {
            if (resolver.resolveConfiguration(action) != null) {
                return resolver;
            }
        }
        return null;
    }

    protected <I extends AbstractActionConfigurationResolver> I registerInitializer(Class<I> initizalizer) {
        try {
            I instance = initizalizer.newInstance();
            configurationResolvers.add(instance);
            return instance;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void fireAction0(String actionKey, Object source, A action) {
        if (action == null) {
            log.warn("could not find action " + actionKey);
            return;
        }
        ActionEvent event = new ActionEvent(source, ActionEvent.ACTION_FIRST, actionKey);
        action.actionPerformed(event);
    }

    protected void checkRegistredAction(String actionKey) {
        if (!impls.containsKey(actionKey)) {
            throw new IllegalStateException("can not find a registered action for key " + actionKey);
        }
    }


    @SuppressWarnings({"unchecked"})
    protected A newActionInstance(String actionKey) throws InstantiationException, IllegalAccessException, InvocationTargetException, NoSuchMethodException {
        Class<? extends MyAbstractAction> klazz = impls.get(actionKey);
        MyAbstractAction result;
        result = klazz.getConstructor(String.class).newInstance(actionKey);
        result.putValue(Action.ACTION_COMMAND_KEY, actionKey);
        if (!getBaseClass().isAssignableFrom(klazz)) {
            // the instanciated action must be boxed in the base Action of the factory
            result = getBaseClass().getConstructor(MyAbstractAction.class).newInstance(result);
        }
        return (A) result;
    }


    public Map<String, Class<? extends MyAbstractAction>> init() {
        if (log.isDebugEnabled()) {
            log.debug("start loading " + this);
        }
        URLClassLoader newCL = fixClassLoader(getClass());
        ClassLoader cl = Thread.currentThread().getContextClassLoader();
        if (newCL != null) {
            // replace current cl by our fiexed cl
            Thread.currentThread().setContextClassLoader(newCL);
        }
        // obtain a ServiceLoader on ActionProvider
        ServiceLoader<ActionProvider> loader = ServiceLoader.load(ActionProvider.class);
        Map<String, Class<? extends MyAbstractAction>> cache = new TreeMap<String, Class<? extends MyAbstractAction>>();

        for (ActionProvider<?> actionProvider : loader) {
            if (log.isDebugEnabled()) {
                log.debug("found " + actionProvider);
            }
            cache.putAll(actionProvider.getClasses());
        }
        if (newCL != null) {
            // to avoid side effects, push back old cl
            Thread.currentThread().setContextClassLoader(cl);
        }
        return cache;
    }

    /**
     * Fix the class loader when application is launched from a java -jar
     * The ServiceLoader seems not to find services from jar manifest...
     * <p/>
     * Our solution is to get all jar from the jar manifest and create a URLClassLoader, this is not perfect but works.
     * <p/>
     * TODO Put this nice code in a ServiceLoaderUtil in lutinutil...
     *
     * @param klass class to use to obtain classloader
     * @return the fixed classloader
     */
    public static URLClassLoader fixClassLoader(Class<?> klass) {
        ClassLoader l = klass.getClassLoader();
        URLClassLoader cl;
        if (!(l instanceof URLClassLoader)) {
            log.warn("using cl is not a URL classloader " + l);
            cl = new URLClassLoader(new URL[0], l);
        } else {
            cl = (URLClassLoader) l;
        }
        if (cl.getURLs().length == 1) {
            // come from a java -jar, must expand all jar to make possible ServiceLoader to work
            try {
                //todo put this in lutinutil ServiceLoaderUtil
                URL[] urls = Resource.getClassPathURLsFromJarManifest(cl.getURLs()[0]);
                URLClassLoader newCL = new URLClassLoader(urls);
                if (log.isTraceEnabled()) {
                    for (URL url : newCL.getURLs()) {
                        log.trace(url);
                    }
                }
                return newCL;
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return null;
    }

}