package org.nuiton.jaxx.runtime.api;

import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import javax.swing.text.AbstractDocument;
import javax.swing.text.JTextComponent;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.lang.reflect.Field;

/**
 * Created on 4/5/15.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 3.0
 */
public class JAXXRuntime {
    private static Field numReaders;

    private static Field notifyingListeners;

    /**
     * Get the first char of a String, or return default value.
     * <p/>
     * Used for example by generated code (i18nMnemonic).
     *
     * @param text         the text to cut
     * @param defaultValue default char value if text is null, or empty
     * @return the first char of the given text or the default value if text is null or empty.
     * @since 2.6.14
     */
    public static char getFirstCharAt(String text, char defaultValue) {
        return text == null || text.trim().length() == 0 ?
               defaultValue : text.charAt(0);
    }

    /**
     * Return parent's container corresponding to the Class clazz
     *
     * @param <O>   type of container to obtain from context
     * @param top   the top container
     * @param clazz desired
     * @return parent's container
     */
    @SuppressWarnings({"unchecked"})
    public static <O extends Container> O getParentContainer(Object top,
                                                             Class<O> clazz) {

        return getParent(top, clazz);
    }

    /**
     * Find a parent of the given {@code top} object using the container api to get up.
     * <p/>
     * Stop on parent when it is of the given{@code clazz} type.
     *
     * @param <O>   type of container to obtain from context
     * @param top   the top container
     * @param clazz desired
     * @return parent's container
     * @since 2.5.14
     */
    public static <O> O getParent(Object top, Class<O> clazz) {
        if (top == null) {
            throw new IllegalArgumentException("top parameter can not be null");
        }
        if (!Container.class.isAssignableFrom(top.getClass())) {
            throw new IllegalArgumentException("top parameter " + top +
                                               " is not a " + Container.class);
        }
        Container parent = ((Container) top).getParent();
        if (parent != null && !clazz.isAssignableFrom(parent.getClass())) {
            parent = (Container) getParent(parent, clazz);
        }
        return (O) parent;
    }

    /**
     * Set the width of the given component
     *
     * @param component the component to resize
     * @param width     the new width to apply
     */
    public static void setComponentWidth(Component component, int width) {
        component.setSize(width, component.getHeight());
        if (component instanceof JComponent) {
            JComponent jcomponent = (JComponent) component;
            jcomponent.setPreferredSize(
                    new Dimension(width,
                                  jcomponent.getPreferredSize().height)
            );
            jcomponent.setMinimumSize(
                    new Dimension(width,
                                  jcomponent.getPreferredSize().height)
            );
            if (jcomponent.isDisplayable()) {
                jcomponent.revalidate();
            }
        }
    }

    /**
     * Set the height of a given component.
     *
     * @param component the component to resize
     * @param height    the new height to apply
     */
    public static void setComponentHeight(Component component, int height) {
        component.setSize(component.getWidth(), height);
        if (component instanceof JComponent) {
            JComponent jcomponent = (JComponent) component;
            jcomponent.setPreferredSize(
                    new Dimension(jcomponent.getPreferredSize().width, height));
            jcomponent.setMinimumSize(
                    new Dimension(jcomponent.getPreferredSize().width, height));
            if (jcomponent.isDisplayable()) {
                jcomponent.revalidate();
            }
        }
    }

    public static void setText(final JTextComponent c, final String text) {
        try {
            // AbstractDocument deadlocks if we try to acquire a write lock while a read lock is held by the current thread
            // If there are any readers, dispatch an invokeLater.  This should only happen in the event of circular bindings.
            // Similarly, circular bindings can result in an "Attempt to mutate in notification" error, which we deal with
            // by checking for the 'notifyingListeners' property.
            AbstractDocument document = (AbstractDocument) c.getDocument();
            if (numReaders == null) {
                numReaders = AbstractDocument.class.getDeclaredField("numReaders");
                numReaders.setAccessible(true);
            }
            if (notifyingListeners == null) {
                notifyingListeners = AbstractDocument.class.getDeclaredField("notifyingListeners");
                notifyingListeners.setAccessible(true);
            }

            if (notifyingListeners.get(document).equals(Boolean.TRUE)) {
                return;
            }

            if ((Integer) numReaders.get(document) > 0) {
                SwingUtilities.invokeLater(new Runnable() {

                    @Override
                    public void run() {
                        if (!c.getText().equals(text)) {
                            c.setText(text);
                        }
                    }
                });
                return;
            }

            String oldText = c.getText();
            if (oldText == null || !oldText.equals(text)) {
                c.setText(text);
            }
        } catch (NoSuchFieldException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        } catch (SecurityException e) {
            c.setText(text);
        }
    }

    /**
     * Compute the string representation of an object.
     * <p/>
     * Return empty string if given object is null
     *
     * @param value the value to write
     * @return the string representation of the given object or an empty string
     * if object is null.
     */
    public static String getStringValue(Object value) {
        String result;
        result = value == null ? "" : value.toString();
        return result;
    }
}
