/*
 * #%L
 * JAXX :: Maven plugin
 * 
 * $Id: AbstractGenerateHelpMojo.java 2598 2013-03-08 09:50:09Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jaxx/tags/jaxx-2.6/jaxx-maven-plugin/src/main/java/org/nuiton/jaxx/plugin/AbstractGenerateHelpMojo.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.jaxx.plugin;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Parameter;
import org.nuiton.i18n.I18nUtil;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * Abstract Mojo to generate help stuff.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.0.0
 */
public abstract class AbstractGenerateHelpMojo extends AbstractJaxxMojo {

    /**
     * The locales to generate for help, separeted by comma.
     * <p/>
     * The first locale given is the default locale.
     *
     * @since 2.0.0
     */
    @Parameter(property = "jaxx.locales", required = true)
    protected String locales;

    /**
     * Where to generate helpIds files.
     *
     * @since 1.3
     */
    @Parameter(property = "jaxx.outputHelpIds", defaultValue = "target/generated-sources/jaxx", required = true)
    private File outputHelpIds;

    /**
     * The store of helpIds generated by the goal {@link GenerateMojo} and then
     * used by the goal {@link GenerateHelpMojo}.
     *
     * @since 1.3
     */
    @Parameter(property = "jaxx.helpIdsFilename", defaultValue = "helpIds.txt", required = true)
    private String helpIdsFilename;

    /**
     * The store of cumulate helpIds generated by the goal {@link GenerateMojo}
     * and then used by the goal {@link GenerateHelpMojo}.
     *
     * @since 1.3
     */
    @Parameter(property = "jaxx.mergeHelpIdsFilename", defaultValue = "helpIds-all.txt", required = true)
    private String mergeHelpIdsFilename;

    /**
     * Flag to generate the search index.
     *
     * @since 1.3
     */
    @Parameter(property = "jaxx.generateSearch", defaultValue = "true", required = true)
    protected boolean generateSearch;

    /** Default locale (the first locale in {@link #localesToTreate}. */
    private Locale defaultLocale;

    /** Locales to treate */
    private Locale[] localesToTreate;

    /**
     * Do the action for the given locale.
     *
     * @param locale          the locale to treate
     * @param isDefaultLocale {@code true} if given locale is de the default
     *                        locale
     * @param source          where are stored help files for the given locale
     * @param localePath      the locale path to use (is {@code default} if
     *                        given locale is default).
     * @throws Exception if any pb
     */
    protected abstract void doActionForLocale(
            Locale locale,
            boolean isDefaultLocale,
            File source,
            String localePath) throws Exception;

    /** Call back after doing all stuff for all locales declared */
    protected abstract void postDoAction();

    /** Call back before doing all stuff for all locales declared */
    protected abstract void preDoAction() throws IOException;

    @Override
    protected void init() throws Exception {

        if (StringUtils.isEmpty(locales)) {
            throw new MojoFailureException(
                    "You must set the 'locales' property properly (was " +
                    locales + ").");
        }

        // check there is a outHelp
        if (getTargetDirectory() == null) {
            throw new MojoFailureException(
                    "You must set the 'outputHelpXXX' property.");
        }

        List<Locale> tmp = new ArrayList<Locale>();
        for (String loc : locales.split(",")) {
            Locale l = I18nUtil.newLocale(loc);
            tmp.add(l);
        }

        if (tmp.isEmpty()) {
            throw new MojoFailureException(
                    "No locale to react, you must set the 'locales' property.");
        }

        localesToTreate = tmp.toArray(new Locale[tmp.size()]);
        defaultLocale = localesToTreate[0];

        createDirectoryIfNecessary(getTargetDirectory());
    }

    @Override
    protected boolean checkSkip() {
        if (!generateHelp) {
            getLog().info("generateHelp flag is off, will skip goal.");
            return false;
        }
        return true;
    }

    @Override
    protected void doAction() throws Exception {

        preDoAction();

        for (Locale locale : localesToTreate) {

            boolean isDefaultLocale = defaultLocale.equals(locale);

            String language = locale.getLanguage();

            String localePath = isDefaultLocale ? "default" : language;

            File source = new File(getTargetDirectory(), localePath);

            createDirectoryIfNecessary(source);

            doActionForLocale(locale, isDefaultLocale, source, localePath);

        }

        postDoAction();
    }

    public File getOutputHelpIds() {
        return outputHelpIds;
    }

    public void setOutputHelpIds(File outputHelpIds) {
        this.outputHelpIds = outputHelpIds;
    }

    public File getHelpIdsStoreFile() {
        return outputHelpIds == null ? null :
               new File(outputHelpIds, helpIdsFilename);
    }

    public File getMergeHelpIdsStoreFile() {
        return outputHelpIds == null ? null :
               new File(outputHelpIds, mergeHelpIdsFilename);
    }

    public String getHelpIdsFilename() {
        return helpIdsFilename;
    }

    public void setHelpIdsFilename(String helpIdsFilename) {
        this.helpIdsFilename = helpIdsFilename;
    }

    public String getMergeHelpIdsFilename() {
        return mergeHelpIdsFilename;
    }

    public void setMergeHelpIdsFilename(String mergeHelpIdsFilename) {
        this.mergeHelpIdsFilename = mergeHelpIdsFilename;
    }

    protected void cleanHelpIdsStore() throws IOException {
        File idsStore = getHelpIdsStoreFile();
        if (idsStore.exists()) {
            if (getLog().isDebugEnabled()) {
                getLog().debug("delete id store " + idsStore);
            }
            boolean b = idsStore.delete();
            if (!b) {
                throw new IOException("could not delete file " + idsStore);
            }
        }
    }

    protected Set<String> loadHelpIds(File file) throws IOException {

        BufferedReader reader;
        Set<String> result = new HashSet<String>();
        reader = new BufferedReader(new InputStreamReader(
                new FileInputStream(file), getEncoding()));
        try {
            String id;
            while ((id = reader.readLine()) != null) {
                id = id.trim();

                if (!id.startsWith("#")) {
                    result.add(id);
                }
            }
            if (isVerbose()) {
                getLog().info("load " + result.size() +
                              " help ids from file " + file);
            }
            return result;
        } finally {
            reader.close();
        }
    }

    protected void storeHelpIds(File file, Set<String> ids) throws IOException {

        createDirectoryIfNecessary(file.getParentFile());

        StringBuilder buffer = new StringBuilder();

        for (String helpId : ids) {
            buffer.append(removeQuote(helpId)).append('\n');
        }
        writeFile(file, buffer.toString(), getEncoding());
        if (isVerbose()) {
            getLog().info("stored " + ids.size() + " help ids in " + file);
        }
    }

    protected String removeQuote(String txt) {
        if (txt.startsWith("\"")) {
            txt = txt.substring(1);
        }
        if (txt.endsWith("\"")) {
            txt = txt.substring(0, txt.length() - 1);
        }
        return txt;
    }

}
