/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.compiler;

import jaxx.CompilerException;
import jaxx.types.TypeManager;

/**
 * Represents a data binding in a JAXX file.  <code>DataBinding</code> uses {@link DataSource} to
 * track changes to a source expression and update the destination.
 */
public class DataBinding {
    private String id;

    /** The DatSource which tracks source expression changes. */
    private DataSource dataSource;

    /** The data binding destination in the form <code>&lt;id&gt;.&lt;propertyName&gt;</code>. */
    private String dest;

    /**
     * A Java snippet which will cause the destination property to be updated with the current value of
     * the binding.
     */
    private String assignment;

    /** The current <code>JAXXCompiler</code>. */
    private JAXXCompiler compiler;


    /**
     * Creates a new data binding.
     *
     * @param source     the Java source code for the data binding expression
     * @param dest       the data binding destination in the form <code>&lt;id&gt;.&lt;propertyName&gt;</code>
     * @param assignment Java snippet which will cause the destination property to be updated with the current value of the binding
     * @param compiler   the current <code>JAXXCompiler</code>
     */
    public DataBinding(String source, String dest, String assignment, JAXXCompiler compiler) {
        this.id = dest;
        this.dataSource = new DataSource(id, source, compiler);
        this.dest = dest;
        this.assignment = assignment;
        this.compiler = compiler;
    }


    public String getId() {
        return id;
    }


    /**
     * Compiles the data binding expression.  This method calls methods in <code>JAXXCompiler</code>
     * to add the Java code that performs the data binding setup.
     *
     * @param quickNoDependencies true to optimize bindings with no dependencies by simply running them at startup time
     * @return <code>true</code> if the expression has dependencies, <code>false</code> otherwise
     * @throws CompilerException if a compilation error occurs
     */
    public boolean compile(boolean quickNoDependencies) throws CompilerException {
        // DataSource.compile handles all of the listener additions
        boolean result = dataSource.compile("new jaxx.runtime.DataBindingListener(" + compiler.getRootObject().getJavaCode() + ", " + TypeManager.getJavaCode(id) + ")");

        if (!result && quickNoDependencies) {
            if (!dest.endsWith(".layout")) // layout is specially handled early in the chain
            {
                compiler.appendInitDataBindings(assignment + JAXXCompiler.getLineSeparator());
            }
            return false; // no dependencies, just a static expression
        }
        if (compiler.haveProcessDataBinding()) {
            compiler.appendProcessDataBinding(" else ");
        } else {
            compiler.appendProcessDataBinding("    ");
        }
        compiler.appendProcessDataBinding("if (" + TypeManager.getJavaCode(id) + ".equals($dest)) {" + JAXXCompiler.getLineSeparator());
        String objectCode = dataSource.getObjectCode();
        if (objectCode != null) {
            compiler.appendProcessDataBinding("    if (" + objectCode + " != null) {" + JAXXCompiler.getLineSeparator());
            compiler.appendProcessDataBinding("    ");
        }
        compiler.appendProcessDataBinding("    " + assignment.trim());
        if (objectCode != null) {
            compiler.appendProcessDataBinding(JAXXCompiler.getLineSeparator() + "    }");
        }
        compiler.appendProcessDataBinding(JAXXCompiler.getLineSeparator() + "    }");
        return true;
    }
}