/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix.generator;

import org.nuiton.guix.CompilerException;
import org.nuiton.guix.parser.JavaParser;
import org.nuiton.guix.parser.JavaParserTreeConstants;
import org.nuiton.guix.parser.ParseException;
import org.nuiton.guix.parser.SimpleNode;
import org.nuiton.guix.tags.TagManager;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.Reader;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

// TODO: need to unify this implementation with the parsing in ScriptManager
public class JavaFileParser {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private final Log log = LogFactory.getLog(JavaFileParser.class);

    private String className;
    private String packageName = null;
    private String superclassName = "java.lang.Object";
    private List<JavaMethod> methods = new ArrayList<JavaMethod>();
    private List<JavaField> fields = new ArrayList<JavaField>();


    public static JavaFile parseJavaFile(String displayName, Reader src) throws ClassNotFoundException {
        // has some limitations -- it reports all members as public, leaves getDeclaredMethod and getDeclaredField
        // undefined, and doesn't report interfaces.  It's safe to leave those the way they are for now, because
        // Guix doesn't look at any of those for non-Guix classes.
        JavaFileParser parser = new JavaFileParser();
        if (log.isDebugEnabled()) {
            log.debug("starting parsing : " + displayName);
        }
        try {
            parser.doParse(displayName, src);
        } catch (Exception e) {
            log.error(e.getMessage());
            throw new RuntimeException(e);
        }
        JavaFile jf = new JavaFile(Modifier.PUBLIC, JavaFile.CLASS, parser.packageName,
                parser.className.substring(parser.className.lastIndexOf(".") + 1), parser.superclassName, new ArrayList<String>(), null);
        for(JavaMethod jm : parser.methods) {
            if(Modifier.isPublic(jm.getModifiers())) {
                jf.addMethod(jm);
            }
        }
        for(JavaField jfi : parser.fields) {
            if(Modifier.isPublic(jfi.getModifiers())) {
                jf.addField(jfi);
            }
        }

        Class superclass = Class.forName(parser.superclassName);
        for(Method m : superclass.getMethods()) {
            JavaArgument[] jas = new JavaArgument[m.getParameterTypes().length];
            for(int i = 0 ; i < m.getParameterTypes().length ; i++) {
                jas[i] = new JavaArgument(m.getParameterTypes()[i].getName(), "arg" + i);
            }
            String[] exceptions = new String[m.getExceptionTypes().length];
            for(int i = 0 ; i < m.getExceptionTypes().length ; i++) {
                exceptions[i] = m.getExceptionTypes()[i].getName();
            }
            jf.addMethod(new JavaMethod(m.getModifiers(), m.getReturnType().getName(), m.getName(), jas, exceptions, "", null));
        }
        for(Field f : superclass.getFields()) {
            jf.addField(new JavaField(f.getModifiers(), f.getType().getName(), f.getName(), null));
        }

        for (Class i : superclass.getInterfaces()) {
            jf.getInterfaces().add(i.getName());
        }

        return jf;
    }

    private void doParse(String displayName, Reader src) {
        try {
            JavaParser p = new JavaParser(src);
            p.CompilationUnit();
            SimpleNode node = p.popNode();
            if (node != null) {
                scanCompilationUnit(node);
                return;
            }
            throw new CompilerException("Internal error: null node parsing Java file from " + src);
        }
        catch (ParseException e) {
            throw new CompilerException("Error parsing Java source code " + displayName + ": " + e.getMessage());
        }
    }


    private void scanCompilationUnit(SimpleNode node) {
        for (int i = 0; i < node.jjtGetNumChildren(); i++) {
            SimpleNode child = node.getChild(i);
            int nodeType = child.getId();
            if (nodeType == JavaParserTreeConstants.JJTPACKAGEDECLARATION) {
                packageName = child.getChild(1).getText().trim();
            } else
            if (nodeType == JavaParserTreeConstants.JJTIMPORTDECLARATION) {
                String text = child.getText().trim();
                if (text.startsWith("import")) {
                    text = text.substring("import".length()).trim();
                }
                if (text.endsWith(";")) {
                    text = text.substring(0, text.length() - 1);
                }
            } else if (nodeType == JavaParserTreeConstants.JJTTYPEDECLARATION) {
                scanCompilationUnit(child);
            } else
            if (nodeType == JavaParserTreeConstants.JJTCLASSORINTERFACEDECLARATION) {
                scanClass(child);
            }
        }
    }


    // scans the main ClassOrInterfaceDeclaration
    private void scanClass(SimpleNode node) {
        boolean isInterface = node.firstToken.image.equals("interface");
        className = node.firstToken.next.image;
        if (packageName != null)
            className = packageName + "." + className;
        for (int i = 0; i < node.jjtGetNumChildren(); i++) {
            SimpleNode child = node.getChild(i);
            int nodeType = child.getId();
            if (nodeType == JavaParserTreeConstants.JJTEXTENDSLIST) {
                if (!isInterface) {
                    assert child.jjtGetNumChildren() == 1 : "expected ExtendsList to have exactly one child for a non-interface class";
                    String rawName = child.getChild(0).getText().trim();
                    superclassName = TagManager.resolveClassName(rawName);
                    if (superclassName == null) {
                        throw new CompilerException("Could not find class: " + rawName);
                    }
                }
            } else
            if (nodeType == JavaParserTreeConstants.JJTCLASSORINTERFACEBODY) {
                scanClassNode(child);
            }
        }
    }


    // scans class body nodes
    private void scanClassNode(SimpleNode node) {
        int nodeType = node.getId();
        if (nodeType == JavaParserTreeConstants.JJTMETHODDECLARATION) {
            String returnType = null;
            String name = null;
            List<JavaArgument> parameterTypes = new ArrayList<JavaArgument>();
            //List<String> parameterNames = new ArrayList<String>();
            for (int i = 0; i < node.jjtGetNumChildren(); i++) {
                SimpleNode child = node.getChild(i);
                int type = child.getId();
                if (type == JavaParserTreeConstants.JJTRESULTTYPE)
                    returnType = TagManager.resolveClassName(child.getText().trim());
                else if (type == JavaParserTreeConstants.JJTMETHODDECLARATOR) {
                    name = child.firstToken.image.trim();
                    SimpleNode formalParameters = child.getChild(0);
                    assert formalParameters.getId() == JavaParserTreeConstants.JJTFORMALPARAMETERS;
                    for (int j = 0; j < formalParameters.jjtGetNumChildren(); j++)
                    {
                        SimpleNode parameter = formalParameters.getChild(j);
                        String rawParameterType = parameter.getChild(1).getText().trim().replaceAll("\\.\\.\\.", "[]");
                        String parameterType = TagManager.resolveClassName(rawParameterType);
                        if (parameterType == null) {
                            throw new CompilerException("could not find class '" + rawParameterType + "'");
                        }
                        parameterTypes.add(new JavaArgument(parameterType, "arg" + j));
                        //parameterNames.add(parameter.getChild(2).getText().trim());
                    }
                }
            }
            methods.add(new JavaMethod(Modifier.PUBLIC, returnType, name, parameterTypes.toArray(new JavaArgument[parameterTypes.size()]), null, "", null)); // TODO: determine the actual modifiers
        } else
        if (nodeType == JavaParserTreeConstants.JJTCLASSORINTERFACEDECLARATION) {
            // TODO: handle inner classes
        } else
        if (nodeType == JavaParserTreeConstants.JJTCONSTRUCTORDECLARATION) {
            // TODO: handle constructors
        } else if (nodeType == JavaParserTreeConstants.JJTFIELDDECLARATION) {
            String text = node.getText();
            String declaration = text;
            int equals = text.indexOf("=");
            if (equals != -1) {
                declaration = declaration.substring(0, equals);
            }
            declaration = declaration.trim();
            String[] declarationTokens = declaration.split("\\s");
            //boolean isFinal = Arrays.asList(declarationTokens).contains("final");
            //boolean isStatic = Arrays.asList(declarationTokens).contains("static");
            String name = declarationTokens[declarationTokens.length - 1];
            if (name.endsWith(";")) {
                name = name.substring(0, name.length() - 1).trim();
            }
            String className = declarationTokens[declarationTokens.length - 2];
            String type = TagManager.resolveClassName(className);
            fields.add(new JavaField(Modifier.PUBLIC, type, name, null)); // TODO: determine the actual modifiers
        } else {
            for (int i = 0; i < node.jjtGetNumChildren(); i++) {
                SimpleNode child = node.getChild(i);
                scanClassNode(child);
            }
        }
    }
}
