/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix.generator;

import java.lang.reflect.Modifier;
import java.util.Comparator;
import org.nuiton.guix.tags.TagHandler;

/**
 * Represents a field in a Java source file being generated for output.  <code>JavaFields</code> are created
 * and added to a {@link JavaFile}, which can then output Java source code.
 */
public class JavaField implements Comparable<JavaField> {

    private int modifiers;
    private String type;
    private String name;
    private String initializer;
    private String javaDoc;
    private TagHandler tagHandler;


    /**
     * Constructs a new <code>JavaField</code>.  The <code>modifiers</code> parameter is a bit mask of the
     * constants from {@link java.lang.reflect.Modifier}, and the <code>type</code> of the field should be
     * represented as it would appear in Java source code.
     *
     * @param modifiers the modifier keywords that should appear as part of the field's declaration
     * @param type      the type of the field as it would appear in Java source code
     * @param name      the field's name
     */
    public JavaField(int modifiers, String type, String name, String javaDoc) {
        this(modifiers, type, name, null, javaDoc, null);
    }

    /**
     * Constructs a new <code>JavaField</code>.  The <code>modifiers</code> parameter is a bit mask of the
     * constants from {@link java.lang.reflect.Modifier}, and the <code>type</code> of the field should be
     * represented as it would appear in Java source code.
     *
     * @param modifiers the modifier keywords that should appear as part of the field's declaration
     * @param type      the type of the field as it would appear in Java source code
     * @param name      the field's name
     */
    public JavaField(int modifiers, String type, String name, String javaDoc, TagHandler tagHanlder) {
        this(modifiers, type, name, null, javaDoc, tagHanlder);
    }


    /**
     * Constructs a new <code>JavaField</code>.  The <code>modifiers</code> parameter is a bit mask of the
     * constants from <code>java.lang.reflect.Modifier</code>, and the <code>type</code> of the field should be
     * represented as it would appear in Java source code.  The <code>initializer</code> is the initial
     * value of the field as it would appear in Java source code, or <code>null</code> to leave it at the
     * default value.
     *
     * @param modifiers   the modifier keywords that should appear as part of the field's declaration
     * @param type        the type of the field as it would appear in Java source code
     * @param name        the field's name
     * @param initializer the initial value of the field, as it would appear in Java source code
     */
    public JavaField(int modifiers, String type, String name, String initializer, String javaDoc, TagHandler tagHandler) {
        this.modifiers = modifiers;
        this.type = type;
        this.name = name;
        this.initializer = initializer;
        this.javaDoc = javaDoc;
        this.tagHandler = tagHandler;
    }


    /**
     * Returns a bit mask describing the modifier keywords which should appear as part of this field's
     * declaration.  See <code>java.lang.reflect.Modifier</code> for more information on decoding this
     * field.
     *
     * @return the modifier bit mask
     */
    public int getModifiers() {
        return modifiers;
    }


    /**
     * Returns the field's name.
     *
     * @return the field's name
     */
    public String getName() {
        return name;
    }


    /**
     * Returns the field's type, as it would be represented in Java source code.
     *
     * @return the field's type
     */
    public String getType() {
        return type;
    }

    public TagHandler getTagHandler() {
        return tagHandler;
    }


    /**
     * Returns the Java source code for this field.
     *
     * @param lineSeparator line separator
     * @return the Java source code for this field
     */
    public String toString(String lineSeparator) {
        StringBuffer result = new StringBuffer();
        if(javaDoc != null && !javaDoc.equals(""))
            result.append("/**\n * ")
                    .append(javaDoc.replace("\n", "\n * "))
                    .append("\n */")
                    .append(lineSeparator);
        result.append(JavaFile.getModifiersText(modifiers));
        result.append(type).append(' ').append(name);
        if (initializer != null) {
            result.append(" = ").append(initializer);
        }
        result.append(';').append(lineSeparator);
        return result.toString();
    }

    public int compareTo(JavaField o) {
        return COMPARATOR.compare(this, o);
    }

    public static final Comparator<JavaField> COMPARATOR = new Comparator<JavaField>() {

        public int compare(JavaField o1, JavaField o2) {

            int result;
            if ((result = compareStatic(o1, o2)) != 0) {
                return result;
            }

            // data sources must be on the last after all other fields
            if ((result = compareDataSource(o1, o2)) != 0) {
                return result;
            }

            // same static
            if ((result = compareVisibility(o1, o2)) != 0) {
                return result;
            }
            // same visibility, test name
            return o1.name.compareTo(o2.name);
        }

        public int compareStatic(JavaField o1, JavaField o2) {
            // first comparator modifiers : static always before none static
            if (Modifier.isStatic(o1.modifiers) && !Modifier.isStatic(o2.modifiers)) {
                return -1;
            }
            if (!Modifier.isStatic(o1.modifiers) && Modifier.isStatic(o2.modifiers)) {
                return 1;
            }
            return 0;
        }

        public int compareDataSource(JavaField o1, JavaField o2) {
            // first comparator modifiers : static always before none static
            if (o1.name.startsWith("$DataSource") && !o2.name.startsWith("$DataSource")) {
                return 1;
            }
            if (!o1.name.startsWith("$DataSource") && o2.name.startsWith("$DataSource")) {
                return -1;
            }
            return 0;
        }

        public int compareVisibility(JavaField o1, JavaField o2) {
            // first comparator modifiers : static always before none static
            if (!Modifier.isPublic(o1.modifiers) && Modifier.isPublic(o2.modifiers)) {
                return 1;
            }

            if (Modifier.isPublic(o1.modifiers) && !Modifier.isPublic(o2.modifiers)) {
                return -1;
            }

            if (Modifier.isProtected(o1.modifiers) && !Modifier.isProtected(o2.modifiers)) {
                return -1;
            }
            if (!Modifier.isProtected(o1.modifiers) && Modifier.isProtected(o2.modifiers)) {
                return 1;
            }

            if (Modifier.isPrivate(o1.modifiers) && !Modifier.isPrivate(o2.modifiers)) {
                return -1;
            }
            if (!Modifier.isPrivate(o1.modifiers) && Modifier.isPrivate(o2.modifiers)) {
                return 1;
            }
            return 0;
        }
    };

    public static JavaField newField(int modifiers, String returnType, String name) {
        return newField(modifiers, returnType, name, null);
    }

    public static JavaField newField(int modifiers, String returnType, String name, String initializer) {
        return new JavaField(modifiers, returnType, name, initializer);
    }
}