/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.nuiton.guix;

//~--- non-JDK imports --------------------------------------------------------

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.nuiton.guix.model.AttributeDescriptor;
import org.nuiton.guix.model.ClassDescriptor;
import org.nuiton.guix.model.GuixModelObject;
import org.nuiton.guix.model.StyleSheet;
import org.nuiton.guix.tags.ScriptHandler;
import org.nuiton.guix.tags.StyleHandler;
import org.nuiton.guix.tags.TagManager;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

//~--- JDK imports ------------------------------------------------------------

import java.io.File;
import java.io.FileReader;
import java.io.IOException;

import java.util.ArrayList;
import java.util.List;

/**
 * Compiles Guix files into model tree
 *
 * @author morin
 */
public class GuixCompiler {

    private static final String STYLE_TAG = "style";
    private static final String SCRIPT_TAG = "script";
    private static final String CONSTRUCTOR_PARAMS_ATTRIBUTE = "constructor";
    private static final String ID_ATTRIBUTE = "id";
    private static final String STYLE_CLASS_ATTRIBUTE = "styleClass";
    private static final String SOURCE_ATTRIBUTE = "source";
    private static final String JAVA_BEAN_ATTRIBUTE = "javaBean";

    /** log */
    private Log log = LogFactory.getLog(GuixCompiler.class);
    /** Integer used to give an id to the objects which don't have one specified by the user */
    private int           index         = 1;
    private StyleHandler  styleHandler  = new StyleHandler();
    private ScriptHandler scriptHandler = new ScriptHandler();

    /** Directory containing the guix files */
    private File baseDir;

    /** Flag to detec if an error occurs while compiling guix file */
    protected boolean failed;

    /**
     * Long representing the date and time of the last modification of any file
     * used during compilation 
     */
    private long lastModification;

    /** GuixLauncher instance which launched this compiler */
    private GuixLauncher launcher;

    /** Root of the model */
    private GuixModelObject rootMO;

    /** Guix file to compile */
    private File   src;
    private String srcPackage;

    private String generationLanguage;
    

    /* -- Constructor methods ------------------------------------------------- */

    /**
     * Creates a new GuixCompiler.
     *
     * @param src       location of file to compile
     * @param launcher  the launcher of the compiler
     */
    public GuixCompiler(File src, GuixLauncher launcher, String srcPackage, String generationLanguage) {
        this.src      = src;
        this.launcher = launcher;

        if (srcPackage != null) {
            this.srcPackage = srcPackage;
        } else {
            this.srcPackage = "";
        }

        if (src != null) {
            baseDir          = src.getParentFile();
            lastModification = src.lastModified();
        }

        this.generationLanguage = generationLanguage;
    }

    /**
     * Compiles the guix file
     *
     * @return the root of the model
     */
    public GuixModelObject compile() {
        if ((src != null) && (launcher != null) && (generationLanguage != null)) {
            try {
                // Creation of the Xml parser
                XmlPullParserFactory factory =
                    XmlPullParserFactory.newInstance(System.getProperty(XmlPullParserFactory.PROPERTY_NAME), null);

                factory.setNamespaceAware(true);

                XmlPullParser xpp = factory.newPullParser();

                xpp.setInput(new FileReader(src));

                // Start parsing
                if (xpp.getEventType() == XmlPullParser.START_DOCUMENT) {

                    // javaDoc of the class to create
                    StringBuffer doc = new StringBuffer();

                    do {
                        xpp.nextToken();

                        if (xpp.getEventType() == XmlPullParser.COMMENT) {
                            doc.append(xpp.getText());
                        }
                    } while (xpp.getEventType() != XmlPullParser.START_TAG);

                    if(xpp.getPrefix() == null || xpp.getPrefix().equalsIgnoreCase(generationLanguage)) {
                        // resolve the package of teh superclass
                        String tagNameSpace = xpp.getNamespace();
                        String tagName;

                        if (xpp.getName().lastIndexOf('.') >= 0) {
                            tagName = xpp.getName().substring(xpp.getName().lastIndexOf('.') + 1);
                        } else {
                            tagName = xpp.getName();
                        }

                        String tagPackageName = resolvePackageName(tagNameSpace, xpp.getName());

                        String id = ((xpp.getAttributeValue("", ID_ATTRIBUTE) != null)
                                 ? xpp.getAttributeValue("", ID_ATTRIBUTE)
                                 : "_" + tagName + index++);

                        // creation of the root GuixModelObject
                        rootMO = new GuixModelObject(id, xpp.getAttributeValue("", CONSTRUCTOR_PARAMS_ATTRIBUTE),
                                    doc.toString(), xpp.getAttributeValue("", STYLE_CLASS_ATTRIBUTE));

                        // the class name is the name of the file minus the extension
                        String className = src.getName().substring(0, src.getName().lastIndexOf('.'));

                        // register the ClassDescriptor
                        ClassDescriptor cd  = new ClassDescriptor(className, srcPackage);
                        cd.setSuperClass(new ClassDescriptor(tagName, tagPackageName));
                        rootMO.setClassDescriptor(launcher.registerClassDescriptor(cd));
                        failed = cd == null;

                        try {
                            ClassLoader.getSystemClassLoader().loadClass(rootMO.getClassDescriptor().getSuperClass().toString());
                        }
                        catch(ClassNotFoundException eee) {
                            try {
                                ClassDescriptor superCD = rootMO.getClassDescriptor().getSuperClass();
                                if(!launcher.getGuixFileClassNames().contains(rootMO.getClassDescriptor().getSuperClass())) {
                                    File f = new File(launcher.getSrcDirectory(), superCD.toString().replace('.', File.separatorChar) + ".guix");
                                    if(!f.exists()) {
                                        f = new File(launcher.getSrcDirectory(), superCD.toString().replace('.', File.separatorChar) + ".jaxx");
                                    }
                                    if(f.exists() && !launcher.isFileAlreadyCompiled(f)) {
                                        GuixCompiler gc = new GuixCompiler(f, launcher, superCD.getPackageName(), generationLanguage);
                                        GuixModelObject rootMO = gc.compile();
                                        launcher.addCompiledFile(f);
                                        launcher.addClassName(superCD.toString());
                                        launcher.addRootModelObject(rootMO, gc.getLastModification());
                                        if (gc.isFailed()) {
                                            failed = true;
                                        }
                                    }
                                    if(f.exists()) {
                                        launcher.addDependency(rootMO, superCD);
                                    }
                                }
                                else {
                                    launcher.addDependency(rootMO, superCD);
                                }
                            }
                            catch(NullPointerException eeee) {
                            }
                        }

                        rootMO.setAttributeDescriptors(getAttributes(xpp));
                        rootMO.setChildren(new ArrayList<GuixModelObject>());

                        // add the stylesheet of the CSS file with the same name as the Guix file
                        File styleFile = new File(src.getParentFile(), className + ".css");

                        if (styleFile.exists()) {
                            rootMO.getCssFiles().add(styleFile);
                            StyleSheet ss = styleHandler.autoDetectStyleFile(styleFile);
                            rootMO.getStyleSheets().add(ss);
                            lastModification = Math.max(lastModification, styleFile.lastModified());
                        }

                        StringBuffer script = new StringBuffer();

                        // add the script in the .script file with the same name as the Guix file
                        File scriptFile = new File(src.getParentFile(), className + ".script");

                        if (scriptFile.exists()) {
                            script.append(scriptHandler.loadScriptFile(scriptFile));
                            lastModification = Math.max(lastModification, scriptFile.lastModified());
                        }

                        // reach the next START_TAG
                        do {
                            xpp.nextToken();
                        } while ((xpp.getEventType() != XmlPullParser.START_TAG)
                                 && (xpp.getEventType() != XmlPullParser.END_DOCUMENT));

                        // if not eof
                        if (xpp.getEventType() == XmlPullParser.START_TAG) {

                            // compile the rest of the file
                            script.append(compile(xpp, rootMO, doc.toString()));
                        }

                        // all the script tags have been recorded
                        rootMO.getClassDescriptor().setScript(script.toString());

                        return rootMO;
                    }
                }
            } catch (XmlPullParserException ex) {
                if (log.isErrorEnabled()) {
                    log.error(ex);
                }
            } catch (IOException ex) {
                if (log.isErrorEnabled()) {
                    log.error(ex);
                }
            }
        }

        failed = true;

        return null;
    }

    /* -- Compile methods ----------------------------------------------------- */

    /**
     *
     * @param xpp           the parser referencing the tag
     * @param previousMO    the <code>GuixModelObject</code> parent of the tag
     * @param javaDoc       the comments above the tag, to add to its javaDoc
     * @return              the scripts already discovered while parsing the file
     */
    protected StringBuffer compile(final XmlPullParser xpp, GuixModelObject previousMO, String javaDoc) {
        StringBuffer    result = new StringBuffer();
        StringBuffer    doc    = new StringBuffer();
        GuixModelObject prev;

        try {
            if(xpp.getPrefix() == null || xpp.getPrefix().equalsIgnoreCase(generationLanguage)) {

                // if the tag is a style tag
                if ((xpp.getName() != null) && xpp.getName().equals(STYLE_TAG)) {
                    File styleFile = null;

                    // the name of the file to load
                    String source = xpp.getAttributeValue("", SOURCE_ATTRIBUTE);

                    // if the source attribute is specified
                    if (source != null) {
                        styleFile = new File(baseDir, source.replace('/', File.separatorChar));
                    }

                    // creates a new stylesheet
                    StyleSheet ss = styleHandler.compileStyle(xpp, styleFile);

                    if (ss != null) {
                        rootMO.getStyleSheets().add(ss);
                    }

                    if ((styleFile != null) && styleFile.exists()) {
                        rootMO.getCssFiles().add(styleFile);
                        lastModification = Math.max(lastModification, styleFile.lastModified());
                    }

                    // the parent is still the same
                    prev = previousMO;
                }    // if the tag is a script tag
                else if ((xpp.getName() != null) && xpp.getName().equals(SCRIPT_TAG)) {
                    File scriptFile = null;

                    // the name of the file to load
                    String source = xpp.getAttributeValue("", SOURCE_ATTRIBUTE);

                    // if the source attribute is specified
                    if (source != null) {
                        scriptFile = new File(baseDir, source.replace('/', File.separatorChar));
                    }

                    // add the script to the result
                    result.append(scriptHandler.compileScript(xpp, scriptFile));

                    if ((scriptFile != null) && scriptFile.exists()) {
                        lastModification = Math.max(lastModification, scriptFile.lastModified());
                    }

                    // the parent is still the same
                    prev = previousMO;
                }    // if the tag is a class tag
                else {
                    String tagNameSpace   = xpp.getNamespace();
                    String tagPackageName = resolvePackageName(tagNameSpace, xpp.getName());
                    String tagName;

                    // if the name of the tag is the fully-qualified class name
                    if (xpp.getName().lastIndexOf('.') >= 0) {
                        tagName = xpp.getName().substring(xpp.getName().lastIndexOf('.') + 1);
                    } else {
                        tagName = xpp.getName();
                    }


                    String id = ((xpp.getAttributeValue("", ID_ATTRIBUTE) != null)
                                 ? xpp.getAttributeValue("", ID_ATTRIBUTE)
                                 : "_" + tagName + index++);

                    // create the GuixModelObject representing the tag
                    GuixModelObject mo = new GuixModelObject(id, xpp.getAttributeValue("", CONSTRUCTOR_PARAMS_ATTRIBUTE),
                            javaDoc, xpp.getAttributeValue("", STYLE_CLASS_ATTRIBUTE));

                    mo.setJavaBean(xpp.getAttributeValue("", JAVA_BEAN_ATTRIBUTE) == null || Boolean.valueOf(xpp.getAttributeValue("", JAVA_BEAN_ATTRIBUTE)));

                    // register the ClassDescriptor
                    ClassDescriptor cd  = launcher.registerClassDescriptor(new ClassDescriptor(tagName, tagPackageName));
                    mo.setClassDescriptor(cd);
                    failed = cd == null;

                    try {
                        ClassLoader.getSystemClassLoader().loadClass(tagPackageName + "." + tagName);
                    }
                    catch(ClassNotFoundException eee) {
                        try {
                            if(!launcher.getGuixFileClassNames().contains(tagPackageName + "." + tagName)) {
                                File f = new File(launcher.getSrcDirectory(), tagPackageName.replace('.', File.separatorChar) + File.separatorChar + tagName + ".guix");
                                if(!f.exists()) {
                                    f = new File(launcher.getSrcDirectory(), tagPackageName.replace('.', File.separatorChar) + File.separatorChar + tagName + ".jaxx");
                                }
                                if(f.exists() && !launcher.isFileAlreadyCompiled(f)) {
                                    GuixCompiler gc = new GuixCompiler(f, launcher, tagPackageName, generationLanguage);
                                    GuixModelObject rootMO = gc.compile();
                                    launcher.addCompiledFile(f);
                                    launcher.addClassName(tagPackageName + "." + tagName);
                                    launcher.addRootModelObject(rootMO, gc.getLastModification());
                                    if (gc.isFailed()) {
                                        failed = true;
                                    }
                                }
                                if(f.exists()) {
                                    launcher.addDependency(rootMO, mo.getClassDescriptor());
                                }
                            }
                            else {
                                launcher.addDependency(rootMO, mo.getClassDescriptor());
                            }
                        }
                        catch(NullPointerException eeee) {
                        }
                    }

                    mo.setParent(previousMO);
                    mo.setAttributeDescriptors(getAttributes(xpp));
                    mo.setChildren(new ArrayList<GuixModelObject>());

                    // add this GuixModelObject to the children of the parent
                    previousMO.getChildren().add(mo);

                    if (!failed) {
                        // parse the children of the tag
                        prev = mo;
                    } else {
                        return null;
                    }
                }
                do {
                    xpp.nextToken();

                    // if the tag has no children
                    if (xpp.getEventType() == XmlPullParser.END_TAG) {
                        prev = prev.getParent();
                    } else if (xpp.getEventType() == XmlPullParser.COMMENT) {
                        // add the comment to the doc of the next tag
                        doc.append(xpp.getText());
                    }
                } while ((xpp.getEventType() != XmlPullParser.START_TAG)
                         && (xpp.getEventType() != XmlPullParser.END_DOCUMENT));
            }
            else {
                prev = previousMO;
                String prefix = xpp.getPrefix();
                String tagName = xpp.getName();
                do {
                    xpp.nextToken();
                }
                while(((xpp.getEventType() != XmlPullParser.END_TAG)
                        || ((xpp.getPrefix() == null ^ prefix == null) || (xpp.getPrefix() != null && prefix != null && !xpp.getPrefix().equals(prefix)))
                        || ((xpp.getName() == null ^ tagName == null) || (xpp.getName() != null && tagName != null && !xpp.getName().equals(tagName))))
                        && (xpp.getEventType() != XmlPullParser.END_DOCUMENT));

                do {
                    xpp.nextToken();
                    if (xpp.getEventType() == XmlPullParser.COMMENT) {
                        // add the comment to the doc of the next tag
                        doc.append(xpp.getText());
                    }
                }
                while((xpp.getEventType() != XmlPullParser.START_TAG)
                         && (xpp.getEventType() != XmlPullParser.END_DOCUMENT));

            }

            // if not eof
            if (xpp.getEventType() == XmlPullParser.START_TAG) {
                // compile the rest of the document
                result.append(compile(xpp, prev, doc.toString()));
            }
        } catch (XmlPullParserException ex) {
            if (log.isErrorEnabled()) {
                log.error(ex);
            }
        } catch (IOException ex) {
            if (log.isErrorEnabled()) {
                log.error(ex);
            }
        }

        return result;
    }

    /**
     * Resolve the package name of the tag using the namespace
     * and the name of the tag.
     *
     * @param tagNameSpace  namespace of the tag
     * @param tagName       name of the tag
     * @return              the name of the package of the tag
     */
    private String resolvePackageName(String tagNameSpace, String tagName) {
        String packageName = null;

        // if there is a namespace
        if ((tagNameSpace != null) && tagNameSpace.endsWith("*")) {
            packageName = tagNameSpace.substring(0, tagNameSpace.length() - 2);
        }

        // if the name of the tag is the fully qualified class name
        if (tagName.lastIndexOf('.') >= 0) {
            if (packageName == null) {
                packageName = tagName.substring(0, tagName.lastIndexOf('.'));
            } else {
                packageName += tagName.substring(0, tagName.lastIndexOf('.'));
            }
        }

        // if nor the namespace nor the fully qualified class name is defined
        if (packageName == null) {
            String fullClassName = TagManager.resolveClassName(tagName);

            packageName = (fullClassName == null || fullClassName.lastIndexOf('.') == -1)
                          ? null
                          : fullClassName.substring(0, fullClassName.lastIndexOf('.'));
        }

        return packageName;
    }

    /**
     * Transforms the attributes of a tag
     * into a list of <code>AttributeDescriptor</code>s.
     *
     * @param xpp   the parser referencing a tag.
     * @return      the list of <code>AttributeDescriptor</code>s corresponding
     *              to the attributes of the tag.
     */
    private List<AttributeDescriptor> getAttributes(XmlPullParser xpp) {
        List<AttributeDescriptor> result = new ArrayList<AttributeDescriptor>();

        for (int i = 0; i < xpp.getAttributeCount(); i++) {
            if (!xpp.getAttributeName(i).equals(ID_ATTRIBUTE) && !xpp.getAttributeName(i).equals(CONSTRUCTOR_PARAMS_ATTRIBUTE)
                    && !xpp.getAttributeName(i).equals(STYLE_CLASS_ATTRIBUTE) && ! xpp.getAttributeName(i).equals(JAVA_BEAN_ATTRIBUTE)) {
                result.add(new AttributeDescriptor(xpp.getAttributeName(i), xpp.getAttributeValue(i)));
            }
        }

        return result;
    }

    /**
     * Return if the compilation failed.
     *
     * @return  true if the compilation failed
     */
    public boolean isFailed() {
        return failed;
    }

    public long getLastModification() {
        return lastModification;
    }
}

