/**
 * *##% guix-compiler
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */
package org.nuiton.guix;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.guix.parser.JavaParserTreeConstants;
import org.nuiton.guix.parser.SimpleNode;

/**
 *
 * @author kmorin
 */
public class BindingUtils {
    /** left brace matcher */
    protected static Matcher leftBraceMatcher = Pattern.compile("^(\\{)|[^\\\\](\\{)").matcher("");
    /** right brace matcher */
    protected static Matcher rightBraceMatcher = Pattern.compile("^(\\})|[^\\\\](\\})").matcher("");
    /** log */
    private static Log log = LogFactory.getLog(BindingUtils.class);

    /**
     * Examine an attribute value for data binding expressions.  Returns a 'cooked' expression which
     * can be used to determine the resulting value.  
     * If the attribute value does not invoke data binding, this method returns <code>null</code>
     *
     * @param stringValue the string value of the property from the XML
     * @return a processed version of the expression
     */
    public static String processDataBindings(String stringValue) {
        int pos = getNextLeftBrace(stringValue, 0);
        if (pos != -1) {
            StringBuffer expression = new StringBuffer();
            int lastPos = 0;
            while (pos != -1 && pos < stringValue.length()) {
                if (pos > lastPos) {
                    if (expression.length() > 0) {
                        expression.append(" + ");
                    }
                    expression.append('"');
                    expression.append(escapeJavaString(stringValue.substring(lastPos, pos)));
                    expression.append('"');
                }

                if (expression.length() > 0) {
                    expression.append(" + ");
                }
                int pos2 = getNextRightBrace(stringValue, pos + 1);
                if (pos2 == -1) {
                    if(log.isErrorEnabled())
                        log.error("unmatched '{' in expression: " + stringValue);
                    return "";
                }
                expression.append(stringValue.substring(pos + 1, pos2));
                pos2++;
                if (pos2 < stringValue.length()) {
                    pos = getNextLeftBrace(stringValue, pos2);
                    lastPos = pos2;
                } else {
                    pos = stringValue.length();
                    lastPos = pos;
                }
            }
            if (lastPos < stringValue.length()) {
                if (expression.length() > 0) {
                    expression.append(" + ");
                }
                expression.append('"');
                expression.append(escapeJavaString(stringValue.substring(lastPos)));
                expression.append('"');
            }

            return expression.toString();
        }
        return null;
    }

    protected static int getNextLeftBrace(String string, int pos) {
        leftBraceMatcher.reset(string);
        return leftBraceMatcher.find(pos) ? Math.max(leftBraceMatcher.start(1), leftBraceMatcher.start(2)) : -1;
    }

    protected static  int getNextRightBrace(String string, int pos) {
        leftBraceMatcher.reset(string);
        rightBraceMatcher.reset(string);
        int openCount = 1;
        int rightPos;
        while (openCount > 0) {
            pos++;
            int leftPos = leftBraceMatcher.find(pos) ? Math.max(leftBraceMatcher.start(1), leftBraceMatcher.start(2)) : -1;
            rightPos = rightBraceMatcher.find(pos) ? Math.max(rightBraceMatcher.start(1), rightBraceMatcher.start(2)) : -1;
            assert leftPos == -1 || leftPos >= pos;
            assert rightPos == -1 || rightPos >= pos;
            if (leftPos != -1 && leftPos < rightPos) {
                pos = leftPos;
                openCount++;
            } else if (rightPos != -1) {
                pos = rightPos;
                openCount--;
            } else {
                openCount = 0;
            }
        }
        return pos;
    }

    /**
     * Escapes a string using standard Java escape sequences, generally in preparation to including it in a string literal
     * in a compiled Java file.
     *
     * @param raw the raw string to be escape
     * @return a string in which all 'dangerous' characters have been replaced by equivalent Java escape sequences
     */
    protected static String escapeJavaString(String raw) {
        StringBuffer out = new StringBuffer(raw);
        for (int i = 0; i < out.length(); i++) {
            char c = out.charAt(i);
            if (c == '\\' || c == '"') {
                out.insert(i, '\\');
                i++;
            } else if (c == '\n') {
                out.replace(i, i + 1, "\\n");
                i++;
            } else if (c == '\r') {
                out.replace(i, i + 1, "\\r");
                i++;
            } else if (c < 32 || c > 127) {
                String value = Integer.toString((int) c, 16);
                while (value.length() < 4) {
                    value = "0" + value;
                }
                out.replace(i, i + 1, "\\u" + value);
                i += 5;
            }
        }
        return out.toString();
    }

    /**
     * Examines a node to identify any dependencies it contains.
     *
     * @param node       node to scan
     * @return the decomposed expression into tokens
     * @throws CompilerException ?
     */
    public static List<String[]> scanNode(SimpleNode node) throws CompilerException {
        List<String[]> bindings = new ArrayList<String[]>();
        switch (node.getId()) {
            case JavaParserTreeConstants.JJTMETHODDECLARATION:
                break;
            case JavaParserTreeConstants.JJTFIELDDECLARATION:
                break;

            default:
                int count = node.jjtGetNumChildren();
                for(int i = 0; i < count; i++) {
                    List<String[]> sss = scanNode(node.getChild(i));
                    if(sss != null) {
                      bindings.addAll(sss);
                    }
                }
                String[] sArray = determineNodeType(node);
                if(sArray != null) {
                    bindings.add(sArray);
                }
        }
        return bindings;
    }

    /**
     * Adds type information to nodes where possible, and as a side effect adds event listeners to nodes which
     * can be tracked.
     *
     * @param expression the node to scan
     * @return the decomposed expression into tokens
     */
    private static String[] determineExpressionType(SimpleNode expression) {
        assert expression.getId() == JavaParserTreeConstants.JJTPRIMARYEXPRESSION;
        SimpleNode prefix = expression.getChild(0);
        if (prefix.jjtGetNumChildren() == 1) {
            int type = prefix.getChild(0).getId();
            if (type == JavaParserTreeConstants.JJTLITERAL || type == JavaParserTreeConstants.JJTEXPRESSION) {
                return null;
            } else if (type == JavaParserTreeConstants.JJTNAME && expression.jjtGetNumChildren() == 1) // name with no arguments after it
            {
                return prefix.getChild(0).getText().trim().split("\\.");
            }
        }

        if (expression.jjtGetNumChildren() == 1) {
            return null;
        }
        else {
            List<String> result = new ArrayList<String>();
            int index = 0;
            for(int i = 0 ; i < expression.jjtGetNumChildren() ; i++) {
                //log.info(expression.getChild(i).getText().trim());
                String token = expression.getChild(i).getText().trim();

                if(token.startsWith("(")) {
                    result.set(index - 1, result.get(index - 1) + token);
                }
                else {
                    int nbToken = 0;
                    while(token.startsWith(".")) {
                        token = token.substring(1);
                    }
                    String[] splitedToken = token.split("\\.");
                    for(int j = splitedToken.length - 1 ; j >= 0 ; j-- ) {
                        if(Character.isUpperCase(splitedToken[j].charAt(0))) {
                            StringBuffer stringBuffer = new StringBuffer();
                            for(int k = 0 ; k < j ; k++) {
                                stringBuffer.append(splitedToken[k]).append(".");
                            }
                            result.add(index, stringBuffer.append(splitedToken[j]).toString());
                            nbToken ++;
                            break;
                        }
                        else {
                            result.add(index, splitedToken[j]);
                            nbToken++;
                        }
                    }
                    index += nbToken;
                }
            }
            return result.toArray(new String[result.size()]);
        }

    }

    /**
     * Adds type information to nodes where possible, and as a side effect adds event listeners to nodes which
     * can be tracked.
     *
     * @param node       node to scan
     * @return the decomposed expression into tokens
     */
    private static String[] determineNodeType(SimpleNode node) {
       String[] result = null;
        switch (node.getId()) {            
            case JavaParserTreeConstants.JJTPRIMARYEXPRESSION:
                result = determineExpressionType(node);
                return result;
            default:
                return null;
        }
    }
}
