/*
 * #%L
 * Pollen :: UI (strust2)
 * 
 * $Id: SavePoll.java 3395 2012-05-28 15:34:46Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.3.1.1/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/SavePoll.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll;

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.opensymphony.xwork2.Preparable;
import com.opensymphony.xwork2.interceptor.annotations.InputConfig;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.interceptor.ParameterAware;
import org.chorem.pollen.bean.ChoiceHelper;
import org.chorem.pollen.bean.PollDateChoice;
import org.chorem.pollen.bean.PollImageChoice;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.ChoiceImpl;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.PersonToListImpl;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountImpl;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.persistence.VotingListImpl;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.services.impl.PreventRuleService;
import org.chorem.pollen.ui.actions.FileUploadAware;
import org.chorem.pollen.ui.converters.DateConverter;
import org.nuiton.util.StringUtil;

import java.io.File;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Creates a new poll.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class SavePoll extends AbstractPollForm implements Preparable, ParameterAware, FileUploadAware {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(SavePoll.class);

    private static final Pattern TEXT_CHOICE_NAME_PATTERN =
            Pattern.compile("textChoice_(\\d+)\\.name");

    private static final Pattern DATE_CHOICE_NAME_PATTERN =
            Pattern.compile("dateChoice_(\\d+)\\.name");

    private static final Pattern IMAGE_CHOICE_NAME_PATTERN =
            Pattern.compile("imageChoice_(\\d+)\\.name");

    /** Flag when there is some errors on the information panel. */
    protected boolean informationsError;

    /** Flag when there is some errors on the options panel. */
    private boolean optionsError;

    /**
     * All the parameters send by request used to build back choices of the
     * poll.
     */
    protected final Map<String, String[]> parameters = Maps.newTreeMap();

    /**
     * Indexed choices  retreive from parameters for the choiceType selected in
     * form. (Other choices are lost).
     */
    private Map<Integer, Choice> choices;

    /**
     * Indexed voting lists retreive from parameters for the pollType selected
     * in form. (Other voting lists are lost).
     */
    private Map<Integer, VotingList> votingLists;

    @Override
    public void prepare() throws Exception {

        prepareFormPage();

        String pollUid = getNonEmptyParameterValue("poll.pollId");

        // Retrieve userAccount to attach to the poll
        UserAccount userAccount = getPollenUserAccount();
        if (userAccount == null) {

            // from parameter userId if defined during update loading
            String userId = getNonEmptyParameterValue("userId");
            if (userId != null) {
                userAccount = getPollService().getEntityById(UserAccount.class, userId);
            }
        }

        poll = getPollService().getPollEditable(pollUid, userAccount, false);

        // If vote is started, prepare choices and votingLists is useless
        // because they can't be updated.
        if (!isVoteStarted()) {

            // Retrieve choiceType from parameters, the poll object will be updated after prepare
            String choiceTypeParam = getNonEmptyParameterValue("poll.choiceType");
            ChoiceType pollChoiceType = ChoiceType.valueOf(choiceTypeParam);
            poll.setChoiceType(pollChoiceType);

            // build poll choices

            switch (pollChoiceType) {

                case TEXT:
                    choices = buildTextChoices();
                    break;
                case DATE:
                    choices = buildDateChoices();
                    break;
                case IMAGE:
                    choices = buildImageChoices();
                    break;
            }

            PollType pollType;
            String pollTypeParam = getNonEmptyParameterValue("poll.pollType");
            if (pollTypeParam == null) {
                pollType = poll.getPollType();

            } else {
                pollType = PollType.valueOf(pollTypeParam);
            }

            switch (pollType) {

                case FREE:

                    // empty voting list
                    votingLists = Maps.newTreeMap();
                    break;
                case RESTRICTED:

                    // restricted voting list
                    votingLists = buildVotingLists(pollType);
                    break;
                case GROUP:

                    // group voting lists
                    votingLists = buildVotingLists(pollType);
                    break;
            }
        }
    }

    @Override
    public String input() {

        Collection<Choice> pollChoices =
                isVoteStarted() ? poll.getChoice() : choices.values();
        Collection<VotingList> pollVotingLists =
                isVoteStarted() ? poll.getVotingList() : votingLists.values();

        loadChoicesAndvotingLists(poll,
                                  pollChoices,
                                  pollVotingLists,
                                  true);
        return INPUT;
    }


    @Override
    public void validate() {

        validateInformations();

        validateOptions();
    }

    @Override
    @InputConfig(methodName = "input")
    public String execute() throws Exception {

        // Save choices and votingLists only if vote is not started
        if (!isVoteStarted()) {

            // Clear previous collections to save those from the form
            poll.clearChoice();
            poll.clearVotingList();

            Map<Integer, Choice> orderedChoices = choices;

            for (Integer index : orderedChoices.keySet()) {
                Choice choice = orderedChoices.get(index);
                poll.addChoice(choice);
            }

            if (!isFreePoll()) {

                for (Integer index : votingLists.keySet()) {
                    VotingList votingList = votingLists.get(index);
                    poll.addVotingList(votingList);
                }
            }
        }

        // Reset maxChoiceNb if limitChoice is unchecked
        if (!isLimitChoice()) {
            poll.setMaxChoiceNb(0);
        }

        poll.clearPreventRule();

        PreventRuleService preventRuleService =
                newService(PreventRuleService.class);
        if (isNotification()) {

            // add a notification rule

            PreventRule rule = preventRuleService.createAddVotePreventRule();
            poll.addPreventRule(rule);
        }

        if (isReminder()) {

            // add a reminder rule

            PreventRule rule = preventRuleService.createRemindPreventRule(
                    getReminderHourCountdown()
            );

            poll.addPreventRule(rule);
        }

        if (isEdit()) {
            getPollService().updatePoll(poll);
            addFlashMessage(_("pollen.information.poll.updated"));

        } else {
            getPollService().createPoll(poll);
            addFlashMessage(_("pollen.information.poll.created"));
        }

        // remove all stuff from session
        getPollenSession().clearDynamicData();

        return SUCCESS;
    }

    @Override
    public void addFile(int index, File file) {
        parameters.put(
                "imageChoice_" + index + ".location",
                new String[]{file.getAbsolutePath()});
    }

    @Override
    public void addFileContentType(int index, String contentType) {
        // not used here
    }

    @Override
    public void addFileName(int index, String fileName) {
        parameters.put("imageChoice_" + index + ".name",
                       new String[]{fileName});
    }

    @Override
    public void setParameters(Map<String, String[]> parameters) {
        this.parameters.putAll(parameters);
    }

    @Override
    public boolean isInformationsError() {
        return informationsError;
    }

    @Override
    public boolean isOptionsError() {
        return optionsError;
    }

    @Override
    public int getSelectedTab() {
        int result;
        if (isInformationsError()) {
            result = 0;
        } else {
            if (isOptionsError()) {
                result = 1;
            } else {
                result = 0;
            }
        }
        return result;
    }

    protected void validateInformations() {

        // -- Title : required -- //
        if (StringUtils.isBlank(poll.getTitle())) {
            addInformationsError("poll.title",
                                 _("pollen.error.poll.required.title"));
        }

        if (isVoteStarted()) {

            // no validation on choices if vote is started

        } else {

            // -- Choice -- //

            if (MapUtils.isEmpty(choices)) {

                // poll must have at least one choice
                addInformationsError("poll.choices",
                                     _("pollen.error.poll.required.one.choice"));
            } else {

                ChoiceType choiceType = poll.getChoiceType();
                int inputChoicesSize = choices.size();
                Set<Object> choiceValues =
                        Sets.newHashSet(ChoiceHelper.toValues(choices.values(),
                                                              choiceType));

                if (inputChoicesSize > choiceValues.size()) {
                    addInformationsError(
                            "poll.choices",
                            _("pollen.error.poll.detected.duplicate.choice.name"));
                }
            }
        }
    }

    protected void validateOptions() {

        if (isVoteStarted()) {

            // no validation on votingLists if vote is started

        } else {

            // -- VotingList -- //

            if (isFreePoll()) {

                // nothing to validate

            } else {

                Set<String> groups = Sets.newHashSet();
                Set<String> voters = Sets.newHashSet();
                Set<String> emails = Sets.newHashSet();

                for (Map.Entry<Integer, VotingList> entry : votingLists.entrySet()) {
                    validateVotingList(entry.getKey(),
                                       entry.getValue(),
                                       groups,
                                       voters,
                                       emails);
                }
            }
        }

        String creatorEmail = poll.getCreator().getEmail();
        if (StringUtils.isNotBlank(creatorEmail) && !StringUtil.isEmail(creatorEmail)) {

            addOptionsError("poll.creator.email",
                            _("pollen.error.email.invalid"));
        }

        if (validateEndDate(poll.getBeginChoiceDate(), poll.getEndChoiceDate())) {

            addOptionsError(
                    "poll.endChoiceDate",
                    _("pollen.error.poll.endChoiceDate.before.beginChoiceDate"));
        }

        if (validateEndDate(poll.getBeginDate(), poll.getEndDate())) {

            addOptionsError("poll.endDate",
                            _("pollen.error.poll.endDate.before.beginDate"));
        }

        if (validateEndDate(poll.getEndChoiceDate(), poll.getEndDate())) {

            addOptionsError("poll.endChoiceDate",
                            _("pollen.error.poll.endChoiceDate.after.endDate"));
        }
    }

    protected void validateVotingList(int votingListNumber,
                                      VotingList votingList,
                                      Set<String> groups,
                                      Set<String> voters,
                                      Set<String> emails) {

        PollType votingListType = poll.getPollType();
        String fieldNamePrefix = "votingList" + votingListType + "_" +
                                 votingListNumber;

        if (isGroupPoll()) {

            // group poll

            // check there is at least one group
            // check no doublon on group names
            // check there is at least one voter on each group
            // check no doublon on voter names
            // check no doublon on voter emails

            { // validate votingList name
                String votingListName = votingList.getName();

                if (StringUtils.isEmpty(votingListName)) {

                    addOptionsError(
                            fieldNamePrefix,
                            _("pollen.error.poll.required.votingList.name"));
                } else {

                    // check no votingList name doublon
                    boolean add = groups.add(votingListName);
                    if (!add) {

                        // name doublon
                        addOptionsError(
                                fieldNamePrefix,
                                _("pollen.error.poll.votingList.name.doublon"));
                    }

                }
            }

            { // validate votingList weight

                if (votingList.getWeight() == 0) {

                    // no weight filled (can be a bad conversion)
                    addOptionsError(
                            fieldNamePrefix,
                            _("pollen.error.poll.votingList.weight.not.valid"));
                }
            }
        }

        // check there is at least one voter

        List<PersonToList> personToLists =
                votingList.getPollAccountPersonToList();

        if (CollectionUtils.isEmpty(personToLists)) {

            // no personToList found for unique votingList 0
            addOptionsError(fieldNamePrefix,
                            _("pollen.error.poll.required.one.personToList"));
        } else {

            // check no doublon on voter names
            // check no doublon on voter emails

            for (int i = 0; i < personToLists.size(); i++) {

                validatePersonList(i,
                                   fieldNamePrefix,
                                   personToLists.get(i),
                                   voters,
                                   emails);
            }
        }
    }

    protected void validatePersonList(int personToListNumber,
                                      String votingListFieldNamePrefix,
                                      PersonToList personToList,
                                      Set<String> voters,
                                      Set<String> emails) {

        String fieldNamePrefix = votingListFieldNamePrefix +
                                 "PersonToList_" + personToListNumber;

        PollAccount pollAccount = personToList.getPollAccount();

        { // validate votingId

            String votingId = pollAccount.getVotingId();
            // check voter is not doublon
            boolean add = voters.add(votingId);
            if (!add) {

                // voter doublon
                addOptionsError(
                        fieldNamePrefix,
                        _("pollen.error.poll.personToList.votingId.doublon"));
            }
        }
        { // validate email
            String email = pollAccount.getEmail();

            if (StringUtils.isEmpty(email)) {
                addOptionsError(
                        fieldNamePrefix,
                        _("pollen.error.email.required"));
            } else {


                boolean validEmail = StringUtil.isEmail(email);
                if (!validEmail) {

                    // not a valid email
                    addOptionsError(
                            fieldNamePrefix,
                            _("pollen.error.email.invalid"));
                } else {

                    // check email not doublon
                    boolean add = emails.add(email);
                    if (!add) {

                        // email doublon
                        addOptionsError(
                                fieldNamePrefix,
                                _("pollen.error.poll.personToList.email.doublon"));
                    }
                }
            }
        }
        { // validate weight
            if (personToList.getWeight() == 0) {

                // no weight filled (can be a bad conversion)
                addOptionsError(
                        fieldNamePrefix,
                        _("pollen.error.poll.personToList.weight.not.valid"));
            }
        }
    }

    protected boolean validateEndDate(Date begin, Date end) {
        return begin != null
               && end != null
               && end.before(begin);
    }

    @Override
    public void addFieldError(String fieldName, String errorMessage) {
        super.addFieldError(fieldName, errorMessage);
        if (log.isDebugEnabled()) {
            log.debug("VALIDATION [" + fieldName + "] : " + errorMessage);
        }
    }

    protected void addInformationsError(String fieldName, String errorMessage) {
        addFieldError(fieldName, errorMessage);
        informationsError = true;
    }

    protected void addOptionsError(String fieldName, String errorMessage) {
        addFieldError(fieldName, errorMessage);
        optionsError = true;
    }

    protected Map<Integer, Choice> buildTextChoices() {
        Map<Integer, Choice> result = Maps.newTreeMap();

        int maxNumber = 0;

        for (String paramName : parameters.keySet()) {

            Matcher matcher = TEXT_CHOICE_NAME_PATTERN.matcher(paramName);
            if (matcher.matches()) {

                // found a text choice name

                String paramValue = getNonEmptyParameterValue(paramName);
                if (paramValue != null) {

                    // can keep this none empty text choice name

                    Integer choiceNumber = Integer.valueOf(matcher.group(1));
                    if (choiceNumber > maxNumber) {
                        maxNumber = choiceNumber;
                    }
                    Choice choice = new ChoiceImpl();
                    createChoice(choice, "textChoice_" + choiceNumber,
                                 paramValue);
                    result.put(choiceNumber, choice);
                }
            }
        }
        result = reindexMap(result, maxNumber);

        int size = result.size();
        if (log.isInfoEnabled()) {
            log.info("nbTextChoices (from request) = " + size);
        }
        logChoice(result);
        return result;
    }

    protected Map<Integer, Choice> buildDateChoices() {
        Map<Integer, Choice> result = Maps.newTreeMap();

        int maxNumber = 0;
        for (String paramName : parameters.keySet()) {

            Matcher matcher = DATE_CHOICE_NAME_PATTERN.matcher(paramName);
            if (matcher.matches()) {

                // found a text choice name

                String paramValue = getNonEmptyParameterValue(paramName);
                if (paramValue != null) {

                    // can keep this none empty text choice name

                    Integer choiceNumber = Integer.valueOf(matcher.group(1));
                    if (choiceNumber > maxNumber) {
                        maxNumber = choiceNumber;
                    }
                    PollDateChoice choice = new PollDateChoice();
                    createDateChoice(choice,
                                     "dateChoice_" + choiceNumber,
                                     paramValue);
                    result.put(choiceNumber, choice);
                }
            }
        }
        result = reindexMap(result, maxNumber);

        int size = result.size();
        if (log.isInfoEnabled()) {
            log.info("nbDateChoices (from request)  = " + size);
        }
        logChoice(result);
        return result;
    }

    protected Map<Integer, Choice> buildImageChoices() {
        Map<Integer, Choice> result = Maps.newTreeMap();

        int maxNumber = 0;
        for (String paramName : parameters.keySet()) {

            Matcher matcher = IMAGE_CHOICE_NAME_PATTERN.matcher(paramName);
            if (matcher.matches()) {

                // found an image choice name

                String paramValue = getNonEmptyParameterValue(paramName);
                if (paramValue != null) {

                    // can keep this none empty text choice name

                    Integer choiceNumber = Integer.valueOf(matcher.group(1));
                    if (choiceNumber > maxNumber) {
                        maxNumber = choiceNumber;
                    }
                    PollImageChoice choice = new PollImageChoice();
                    createImageChoice(choice,
                                      "imageChoice_" + choiceNumber,
                                      paramValue);

                    result.put(choiceNumber, choice);
                }
            }
        }
        result = reindexMap(result, maxNumber);
        int size = result.size();
        if (log.isInfoEnabled()) {
            log.info("nbImageChoices  (from request) = " + size);
        }
        logChoice(result);
        return result;
    }

    protected Map<Integer, VotingList> buildVotingLists(PollType pollType) {
        Map<Integer, VotingList> result = Maps.newTreeMap();

        int maxNumber = 0;

        String votingListPrefix = "votingList" + pollType.name();

        // get all votingList_ parameters
        Set<String> votingListParameterNames = Sets.filter(
                parameters.keySet(),
                new StringStartWithPredicate(votingListPrefix));

        Pattern votingListPattern = Pattern.compile(
                "(" + votingListPrefix + ")_(\\d+)\\.name");

        for (String paramName : votingListParameterNames) {

            Matcher matcher = votingListPattern.matcher(paramName);

            if (matcher.matches()) {

                // found a voting list name

                int votingListNumber = Integer.valueOf(matcher.group(2));

                votingListPrefix += "_" + votingListNumber;
                buildVotingList(paramName,
                                votingListPrefix,
                                votingListNumber,
                                result
                );
                maxNumber = Math.max(maxNumber, votingListNumber);
            }
        }

        result = reindexMap(result, maxNumber);

        int size = result.size();
        if (log.isInfoEnabled()) {
            log.info("nbVotingList [" + pollType + "] (from request) = " + size);
        }

        // add personToList maps to session (but just now, since votingList
        // could have been reindex)
        for (Map.Entry<Integer, VotingList> entry : result.entrySet()) {
            Integer votingListNumber = entry.getKey();
            VotingList votingList = entry.getValue();

            if (!votingList.isPollAccountPersonToListEmpty()) {
                List<PersonToList> personToList =
                        votingList.getPollAccountPersonToList();

                Map<Integer, PersonToList> personToListMap = Maps.newTreeMap();
                int index = 0;
                for (PersonToList toList : personToList) {
                    personToListMap.put(index++, toList);
                }
            }
        }

        return result;
    }

    private double getDoubleValue(String parameterName) {
        String parameterValue = getNonEmptyParameterValue(parameterName);
        double result = 0;
        if (StringUtils.isNotEmpty(parameterValue)) {

            try {
                result = Double.valueOf(parameterValue);
            } catch (NumberFormatException e) {
                //bad conversion, will be treated later
                if (log.isDebugEnabled()) {
                    log.debug("Bad double conversion from param [" +
                              parameterName + "] : " + parameterValue);
                }
            }
        }
        return result;
    }

    private int buildVotingList(String votingListParameterName,
                                String votingListPrefix,
                                int votingListNumber,
                                Map<Integer, VotingList> result) {

        String paramValue = getNonEmptyParameterValue(votingListParameterName);

        VotingList votingList = new VotingListImpl();

        votingList.setName(paramValue);

        double weight = getDoubleValue(votingListPrefix + ".weight");
        votingList.setWeight(weight);

        String topiaId = getNonEmptyParameterValue(votingListPrefix + ".topiaId");
        votingList.setTopiaId(topiaId);

        result.put(votingListNumber, votingList);

        String personToListPrefix = votingListPrefix + "PersonToList_";

        // get all personToList parameters
        Set<String> votingListParameterNames = Sets.filter(
                parameters.keySet(),
                new StringStartWithPredicate(personToListPrefix));

        Pattern personToListNamePattern = Pattern.compile(
                personToListPrefix + "(\\d+)\\.votingId");

        Map<Integer, PersonToList> personToLists = Maps.newTreeMap();

        int maxPersonToListNumber = 0;

        // let's build personToList list
        for (String personToListNameParameter : votingListParameterNames) {

            Matcher matcher = personToListNamePattern.matcher(
                    personToListNameParameter);

            if (matcher.matches()) {

                int personToListNumber = buildPersonToList(
                        personToListPrefix,
                        personToListNameParameter,
                        matcher,
                        votingListNumber,
                        personToLists);

                maxPersonToListNumber = Math.max(maxPersonToListNumber,
                                                 personToListNumber);
            }
        }

        personToLists = reindexMap(personToLists, maxPersonToListNumber);

        for (PersonToList personToList : personToLists.values()) {
            votingList.addPollAccountPersonToList(personToList);
        }

        return votingListNumber;
    }

    private int buildPersonToList(String personToListPrefix,
                                  String paramName,
                                  Matcher personToListMatcher,
                                  int votingListNumber,
                                  Map<Integer, PersonToList> result) {

        String paramValue = getNonEmptyParameterValue(paramName);

        int personToListNumber = 0;

        if (paramValue != null) {

            // found a PersonToList none empty name, keep it

            personToListNumber = Integer.valueOf(personToListMatcher.group(1));

            PersonToList personToList = new PersonToListImpl();

            PollAccount account = new PollAccountImpl();
            personToList.setPollAccount(account);

            account.setVotingId(paramValue);

            String prefix = personToListPrefix + personToListNumber;

            double weight = getDoubleValue(prefix + ".weight");
            personToList.setWeight(weight);

            String topiaId = getNonEmptyParameterValue(prefix + ".topiaId");
            personToList.setTopiaId(topiaId);

            String email = getNonEmptyParameterValue(prefix + ".email");
            account.setEmail(email);

            String accountId = getNonEmptyParameterValue(prefix + ".accountId");
            account.setAccountId(accountId);

            result.put(personToListNumber, personToList);
        }
        return personToListNumber;
    }

    private Choice createImageChoice(PollImageChoice choice,
                                     String prefix,
                                     String name) {
        createChoice(choice, prefix, name);
        String locationName = prefix + ".location";
        String location = getNonEmptyParameterValue(locationName);
        choice.setLocation(location);
        if (log.isInfoEnabled()) {
            log.info("image location [" + name + "] =" + location);
        }
        return choice;
    }

    private Choice createDateChoice(PollDateChoice choice,
                                    String prefix,
                                    String name) {
        createChoice(choice, prefix, name);
        if (StringUtils.isNotEmpty(name)) {
            Date date = DateConverter.convertFromString(name);
            if (date == null) {
                addInformationsError(prefix, _("pollen.error.date.format"));
            }
            choice.setDate(date);
        }
        return choice;
    }

    private Choice createChoice(Choice choice, String prefix, String name) {
        String descriptionName = prefix + ".description";
        String topiaIdName = prefix + ".topiaId";
        String description = getNonEmptyParameterValue(descriptionName);
        String topiaId = getNonEmptyParameterValue(topiaIdName);
        choice.setName(name);
        choice.setDescription(description);
        choice.setTopiaId(topiaId);
        return choice;
    }

    private void logChoice(Map<Integer, Choice> result) {
        for (Map.Entry<Integer, Choice> e : result.entrySet()) {
            Integer choiceId = e.getKey();
            Choice choice = e.getValue();
            if (log.isInfoEnabled()) {
                log.info("Choice [" + choiceId + "] = " +
                         choice.getName() + " -- " +
                         choice.getDescription());
            }
        }
    }

    protected <T> Map<Integer, T> reindexMap(Map<Integer, T> result, int maxNumber) {
        Map<Integer, T> result2;

        if (maxNumber != result.size() - 1) {

            // means there is a hole inside the result (a empty choice was
            // submitted)

            // le'ts remove this
            List<Integer> numbers = Lists.newArrayList(result.keySet());

            Collections.sort(numbers);

            result2 = Maps.newTreeMap();
            int i = 0;
            for (Integer number : numbers) {
                T choice = result.get(number);
                result2.put(i++, choice);
            }
        } else {
            result2 = result;
        }
        return result2;
    }

    protected String getNonEmptyParameterValue(String paramName) {
        String[] paramValues = parameters.get(paramName);
        String result = null;
        if (paramValues != null && paramValues.length == 1) {
            String paramValue = paramValues[0];
            if (StringUtils.isNotEmpty(paramValue)) {
                result = paramValue;
            }
        }
        return result;
    }

    private static class StringStartWithPredicate implements Predicate<String> {
        private final String prefix;

        public StringStartWithPredicate(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public boolean apply(String input) {
            return input.startsWith(prefix);
        }
    }
}
