/*
 * #%L
 * Pollen :: Services
 * $Id: FavoriteListImportLDAP.java 3423 2012-06-05 16:50:22Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4/pollen-services/src/main/java/org/chorem/pollen/services/impl/FavoriteListImportLDAP.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.collect.Lists;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.services.FavoriteListImport;
import org.chorem.pollen.services.PollenServiceSupport;
import org.chorem.pollen.services.exceptions.FavoriteListImportException;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import java.util.List;
import java.util.Properties;

/**
 * LDAP Import of {@link PollAccount}. The behavior is the same than the one
 * from pollen 1.2.x
 * <p/>
 * TODO-fdesbois-2012-04-17 : tests and documentation
 * <p/>
 * Created: 16/04/12
 *
 * @author fdesbois <desbois@codelutin.com>
 * @since 1.3
 */
public class FavoriteListImportLDAP extends PollenServiceSupport implements FavoriteListImport {

    private static final Log log = LogFactory.getLog(FavoriteListImportLDAP.class);

    @Override
    public List<PollAccount> execute(String url) throws FavoriteListImportException {

        long start = System.nanoTime();

        List<PollAccount> results = Lists.newArrayList();
        try {

            // Initialisation du contexte
            Properties env = new Properties();
//            if (server != null) {
//                env.put(Context.INITIAL_CONTEXT_FACTORY,
//                        "com.sun.jndi.ldap.LdapCtxFactory");
//                env.put(Context.PROVIDER_URL, "ldap://" + server + "/");
//            }
            DirContext ictx = new InitialDirContext(env);

            // Recherche en profondeur
            SearchControls control = new SearchControls();
            control.setSearchScope(SearchControls.SUBTREE_SCOPE);

            // Création des comptes avec les résultats de la recherche
            NamingEnumeration<SearchResult> e = ictx.search(url, null,
                                                            control);
            while (e.hasMore()) {
                SearchResult r = e.next();

                if (log.isDebugEnabled()) {
                    log.debug("Result: " + r.getName() + "(object: "
                              + r.getClassName() + ")");
                }

                Attribute nameAttr = r.getAttributes().get("cn");
                Attribute emailAttr = r.getAttributes().get("mail");

                if (nameAttr != null) {
                    PollAccountDAO dao = getDAO(PollAccount.class);
                    PollAccount account = newInstance(dao);
                    account.setVotingId(nameAttr.get().toString());
                    account.setEmail(emailAttr.get().toString());
                    results.add(account);

                    if (log.isDebugEnabled()) {
                        log.debug("New account - name: "
                                  + nameAttr.get().toString() + ", email: "
                                  + emailAttr.get().toString());
                    }
                }
            }
        } catch (NamingException ex) {
            log.error("Exception de nommage lors de l'import depuis LDAP", ex);
            throw new FavoriteListImportException("LDAP", ex);
        }

        long duration = (System.nanoTime() - start) / 1000000000;
        if (log.isInfoEnabled()) {
            log.info(results.size() + " comptes importés en " + duration
                     + " sec.");
        }

        return results;
    }
}
