/*
 * Copyright 2007 Robert Hanson <iamroberthanson AT gmail.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.gwtwidgets.client.ui.canvas.impl;

import java.util.HashMap;
import java.util.Map;

import org.gwtwidgets.client.ui.canvas.Canvas;
import org.gwtwidgets.client.ui.canvas.Font;
import org.gwtwidgets.client.ui.canvas.FontLoadListener;

import com.google.gwt.http.client.Request;
import com.google.gwt.http.client.RequestBuilder;
import com.google.gwt.http.client.RequestCallback;
import com.google.gwt.http.client.Response;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Element;

/**
 * <p>
 * A <code>BitmapFontImpl</code> object encapsulates a font and knows how to
 * draw text on a canvas. This base implementation renders bitmap fonts read from
 * an image and a font descriptor.
 * </p>
 * <p>
 * The font is usually not usable right after instantiation because asynchronous
 * requests are made towards the server in order to retrieve font specific
 * information. If required, load listeners can be attached to the font object
 * in order to poll its load status.
 * </p>
 * <p>
 * BitmapFontImpl objects are expensive to create and maintain as two requests are made to the server for
 * the font bitmap and the font descriptor, the entire bitmap is stored in the browser's cache and
 * a costly parsing of the font descriptor is performed.
 * </p>
 * 
 * @author George Georgovassilis g.georgovassilis[at]gmail.com
 * 
 */
public class BitmapFontImpl extends Font implements RequestCallback {

	protected Map characters = new HashMap();
	protected int blockHeight = 0;
	protected boolean imageLoaded = false;
	protected boolean descriptionLoaded = false;
	protected FontLoadListener listener;
	protected Element bitmap;

	private void parseHeader(String headerLine) {
		String[] parts = headerLine.split(",");
		blockHeight = Utils.readInt(parts[1]);
	}

	private void fireFontLoaded() {
		if ((listener == null) || !(imageLoaded && descriptionLoaded))
			return;
		listener.onLoad(this);
		listener = null;
	}

	private native void addImageLoadListener(Element image) /*-{
	var ref = this;
	var imageRef = image;
	imageRef.onload = function(){
		ref.@org.gwtwidgets.client.ui.canvas.impl.BitmapFontImpl::imageLoaded = true;
		ref.@org.gwtwidgets.client.ui.canvas.impl.BitmapFontImpl::fireFontLoaded()();
		ref = undefined;
		imageRef.onload = null;
		};
	}-*/;

	/**
	 * Allows substitution of a listener in the constructor of extending classes
	 * 
	 * @param listener
	 * @return
	 */

	protected FontLoadListener doWithListener(FontLoadListener listener) {
		return listener;
	}

	/**
	 * Constructs a canvas font and loads required resources from the server.
	 * This method will issue two HTTP requests to the server:
	 * <ul>
	 * <li>A request for reading the bitmap representation of the font. A
	 * relative request is made to <code>bitmapPath</code> to retrieve the
	 * font bitmap as created by the {@link BitmapFontCreator}</li>
	 * <li>The font descriptor is retrieved by a request to
	 * <code>descriptionPath</code></li>
	 * </ul>
	 * 
	 * @param bitmapPath
	 *            URL to the bitmap containing the font's graphical
	 *            representation.
	 * @param descriptionPath
	 *            URL to the font's descriptor.
	 * @param listener
	 *            Optional listener to execute when the font is ready to use.
	 *            Can be null.
	 * @param token
	 *            Token to be passed to the listener
	 */
	public BitmapFontImpl(String bitmapPath, String descriptionPath, FontLoadListener listener) {
		this.bitmap = DOM.createImg();
		this.listener = doWithListener(listener);
		addImageLoadListener(bitmap);
		DOM.setImgSrc(bitmap, bitmapPath);
		RequestBuilder requestBuilder = new RequestBuilder(RequestBuilder.GET, descriptionPath);
		try {
			requestBuilder.sendRequest("", this);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Draws a character. Invoked by
	 * {@link #drawText(String, Canvas, double, double)}. Can be overridden by
	 * extending implementations.
	 * 
	 * @param c
	 *            Character to draw
	 * @param canvas
	 *            Canvas to draw on
	 * @param charOffsetLeft
	 *            Horizontal character offset on the template bitmap
	 * @param charOffsetTop
	 *            Vertical character offset on the template bitmap
	 * @param charWidth
	 *            Glyph width
	 * @param blockHeight
	 *            Glyph height
	 * @param destinationX
	 *            Destination X on canvas
	 * @param destinationY
	 *            Destination Y on canvas
	 * @param destinationWidth
	 *            Width of character to draw
	 * @param destinationHeight
	 *            Height of character to draw
	 */
	protected void drawGlyph(char c, Canvas canvas, double charOffsetLeft, double charOffsetTop, double charWidth, double blockHeight,
			double destinationX, double destinationY, double destinationWidth, double destinationHeight) {
		canvas.drawImage(bitmap, charOffsetLeft, charOffsetTop, charWidth, blockHeight, destinationX, destinationY, destinationWidth,
				destinationHeight);
	}

	public void drawText(String text, Canvas canvas, double x, double y) {
		double offsetX = x;
		double cosR = Math.cos(canvas.getRotation());
		double sinR = Math.sin(canvas.getRotation());
		for (int i = 0; i < text.length(); i++) {
			char c = text.charAt(i);
			int[] args = (int[]) characters.get("" + c);
			if (args == null)
				continue;
			int charWidth = args[0];
			int charOffsetLeft = args[1];
			int charOffsetTop = args[2];

			double r = offsetX - x;
			double rotX = r * cosR;
			double rotY = r * sinR;

			drawGlyph(c, canvas, charOffsetLeft, charOffsetTop, charWidth, blockHeight, rotX + x, rotY + y, charWidth, blockHeight);

			offsetX += charWidth;
		}
	}

	public void onError(Request request, Throwable exception) {
		if (listener == null)
			return;
		listener.onFail(exception);
	}

	public void onResponseReceived(Request request, Response response) {
		String[] lines = Utils.split(response.getText(), "\n");
		parseHeader(lines[0]);

		int offsetTop = 0;
		int charNum = 0;

		for (int lineNo = 1; lineNo < lines.length; lineNo++) {
			String parts[] = Utils.split(lines[lineNo], ",");
			offsetTop += Utils.readInt(parts[0]);

			for (int i = 1; i < parts.length; i++) {
				int c = Utils.readInt(parts[i]);
				if (c < 0) {
					charNum = -c;
					i++;
				} else
					charNum++;
				int charWidth = Utils.readInt(parts[i++]);
				int offsetLeft = Utils.readInt(parts[i]);

				int[] args = new int[] { charWidth, offsetLeft, offsetTop };
				characters.put("" + (char)charNum, args);
			}
		}
		descriptionLoaded = true;
		fireFontLoaded();
	}

}