/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: AppModule.java 625 2010-07-24 12:35:32Z fdesbois $
 * $HeadURL: svn+ssh://fdesbois@labs.libre-entreprise.org/svnroot/suiviobsmer/tags/wao-1.5.2/wao-ui/src/main/java/fr/ifremer/wao/ui/services/AppModule.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.wao.ui.services;

import fr.ifremer.wao.WaoContextImpl;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.bean.ContactFilter;
import fr.ifremer.wao.bean.ContactFilterImpl;
import fr.ifremer.wao.bean.PieChartData;
import fr.ifremer.wao.bean.PieChartDataImpl;
import fr.ifremer.wao.service.ServiceBoat;
import fr.ifremer.wao.service.ServiceBoatImpl;
import fr.ifremer.wao.service.ServiceCartography;
import fr.ifremer.wao.service.ServiceCartographyImpl;
import fr.ifremer.wao.service.ServiceChart;
import fr.ifremer.wao.service.ServiceContact;
import fr.ifremer.wao.service.ServiceContactImpl;
import fr.ifremer.wao.service.ServiceNews;
import fr.ifremer.wao.service.ServiceNewsImpl;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceReferentialImpl;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.service.ServiceSamplingImpl;
import fr.ifremer.wao.service.ServiceSynthesis;
import fr.ifremer.wao.service.ServiceSynthesisImpl;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.service.ServiceUserImpl;
import fr.ifremer.wao.ui.pages.PieChart;
import org.apache.tapestry5.SymbolConstants;
import org.apache.tapestry5.ioc.Configuration;
import org.apache.tapestry5.ioc.MappedConfiguration;
import org.apache.tapestry5.ioc.OrderedConfiguration;
import org.apache.tapestry5.ioc.ServiceBinder;
import org.apache.tapestry5.ioc.annotations.EagerLoad;
import org.apache.tapestry5.ioc.annotations.InjectService;
import org.apache.tapestry5.ioc.services.Coercion;
import org.apache.tapestry5.ioc.services.CoercionTuple;
import org.apache.tapestry5.ioc.services.RegistryShutdownHub;
import org.apache.tapestry5.services.ApplicationStateContribution;
import org.apache.tapestry5.services.ApplicationStateCreator;
import org.apache.tapestry5.services.ApplicationStateManager;
import org.apache.tapestry5.services.ComponentRequestFilter;
import org.apache.tapestry5.services.PageRenderLinkSource;
import org.apache.tapestry5.services.Request;
import org.apache.tapestry5.services.RequestFilter;
import org.apache.tapestry5.services.RequestHandler;
import org.apache.tapestry5.services.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

/**
 * This module is automatically included as part of the Tapestry IoC Registry,
 * it's a good place to configure and extend Tapestry, or to place your own
 * service definitions.
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
public class AppModule {

    private static final Logger logger =
            LoggerFactory.getLogger(AppModule.class);

    public static void bind(ServiceBinder binder) {
        binder.bind(ContactModelFactory.class);
    }

    /**
     * Use EagerLoad to build manager at startup. The manager will control
     * the application lifecycle (start and stop).
     *
     * @param hub to register the manager in the tapestry registry shutdown
     * @return the new instance of WaoManager
     */
    @EagerLoad
    public WaoManager buildWaoManager(RegistryShutdownHub hub) {
        WaoManager manager = new WaoManager(new WaoContextImpl());
        hub.addRegistryShutdownListener(manager);
        return manager;
    }

    /**
     * Execute WaoManager when registry startup.
     * The manager will be instantiated using
     * {@link #buildWaoManager(org.apache.tapestry5.ioc.services.RegistryShutdownHub) }.
     * The serviceUser is needed to start the application.
     *
     * @param configuration to add the manager to registry startup
     * @param manager to launch at startup
     * @param serviceUser needed for manager start
     */
    public static void contributeRegistryStartup(
            OrderedConfiguration<Runnable> configuration,
            WaoManager manager, ServiceUser serviceUser) {
        manager.setServiceUser(serviceUser);
        configuration.add("WaoStartup", manager);
    }

    public ServiceNews buildServiceNews(WaoManager manager) {
        ServiceNewsImpl instance = new ServiceNewsImpl();
        instance.setContext(manager.getContext());
        return instance;
    }

    public ServiceReferential buildServiceReferential(WaoManager manager) {
        ServiceReferentialImpl instance = new ServiceReferentialImpl();
        instance.setContext(manager.getContext());
        return instance;
    }

    public ServiceBoat buildServiceBoat(WaoManager manager) {
        ServiceBoatImpl instance = new ServiceBoatImpl();
        instance.setContext(manager.getContext());
        return instance;
    }

    public ServiceUser buildServiceUser(WaoManager manager) {
        ServiceUserImpl instance = new ServiceUserImpl();
        instance.setContext(manager.getContext());
        return instance;
    }

    public ServiceSampling buildServiceSampling(WaoManager manager) {
        ServiceSamplingImpl instance = new ServiceSamplingImpl();
        instance.setContext(manager.getContext());
        return instance;
    }

    public ServiceContact buildServiceContact(WaoManager manager) {
        ServiceContactImpl instance = new ServiceContactImpl();
        instance.setContext(manager.getContext());
        return instance;
    }

    public ServiceSynthesis buildServiceSynthesis(WaoManager manager) {
        ServiceSynthesisImpl instance = new ServiceSynthesisImpl();
        instance.setContext(manager.getContext());
        return instance;
    }

    public ServiceChart buildServiceChart(PageRenderLinkSource pageRender) {
        ServiceChartImpl instance = new ServiceChartImpl(pageRender);
        return instance;
    }

    public ServiceChart buildServiceChartServlet() {
        ServiceChartServlet instance = new ServiceChartServlet();
        return instance;
    }

    public ServiceAuthentication buildServiceAuthentication(
            ApplicationStateManager stateManager) {
        ServiceAuthentication instance =
                new ServiceAuthenticationImpl(stateManager);
        return instance;
    }

    public ServiceCartography buildServiceCartography(WaoManager manager,
                                                      @InjectService("serviceChart") ServiceChart serviceChart,
                                                      ServiceSynthesis serviceSynthesis) {
        ServiceCartographyImpl instance = new ServiceCartographyImpl();
        instance.setContext(manager.getContext());
        instance.setServiceChart(serviceChart);
        instance.setServiceSynthesis(serviceSynthesis);
        return instance;
    }

    public static void contributeApplicationDefaults(
            MappedConfiguration<String, String> configuration) {
        // Contributions to ApplicationDefaults will override any contributions to
        // FactoryDefaults (with the same key). Here we're restricting the supported
        // locales to just "en" (English). As you add localised message catalogs and other assets,
        // you can extend this list of locales (it's a comma separated series of locale names;
        // the first locale name is the default when there's no reasonable match).

        configuration.add(SymbolConstants.SUPPORTED_LOCALES, "fr");

        // The factory default is true but during the early stages of an application
        // overriding to false is a good idea. In addition, this is often overridden
        // on the command line as -Dtapestry.production-mode=false
        configuration.add(SymbolConstants.PRODUCTION_MODE, "false");
    }

    /**
     * This is a service definition, the service will be named "TimingFilter".
     * The interface, RequestFilter, is used within the RequestHandler service
     * pipeline, which is built from the RequestHandler service configuration.
     * Tapestry IoC is responsible for passing in an appropriate Logger
     * instance. Requests for static resources are handled at a higher level, so
     * this filter will only be invoked for Tapestry related requests.
     *
     * <p>
     * Service builder methods are useful when the implementation is inline as
     * an inner class (as here) or require some other kind of special
     * initialization. In most cases, use the static bind() method instead.
     *
     * <p>
     * If this method was named "build", then the service id would be taken from
     * the service interface and would be "RequestFilter". Since Tapestry
     * already defines a service named "RequestFilter" we use an explicit
     * service id that we can reference inside the contribution method.
     * @param log
     * @return
     */
    public RequestFilter buildTimingFilter(final Logger log) {
        return new RequestFilter() {
            @Override
            public boolean service(Request request, Response response,
                    RequestHandler handler) throws IOException {
                long startTime = System.currentTimeMillis();

                try {
                    // The responsibility of a filter is to invoke the corresponding method
                    // in the handler. When you chain multiple filters together, each filter
                    // received a handler that is a bridge to the next filter.

                    return handler.service(request, response);
                } finally {
                    long elapsed = System.currentTimeMillis() - startTime;

                    log.info(String.format("Request time: %d ms", elapsed));
                }
            }
        };
    }

    /**
     * Used to instanciate SessionState ConnectedUser when session expires.
     *
     * @param configuration to add the ApplicationStateCreator.
     * @param serviceAuthentication used to instantiate the new user
     */
    public void contributeApplicationStateManager(MappedConfiguration<Class<?>,
            ApplicationStateContribution> configuration,
            final ServiceAuthentication serviceAuthentication) {
        ApplicationStateCreator<ConnectedUser> creator =
                new ApplicationStateCreator<ConnectedUser>() {
            @Override
            public ConnectedUser create() {
                return serviceAuthentication.getNewUserInstance();
            }
        };

        configuration.add(ConnectedUser.class,
                new ApplicationStateContribution("session", creator));
    }

    /**
     * Contribution to ComponentRequestHandler service. The
     * {@link RequiresAuthenticationFilter} is added to the configuration of
     * the service to provide authentication managment on user connected.
     *
     * @param configuration
     * @see ServiceAuthentication
     * @see RequiresAuthenticationFilter
     * @see ComponentRequestFilter
     */
    public static void contributeComponentRequestHandler(
        OrderedConfiguration<ComponentRequestFilter> configuration) {
        configuration.addInstance("RequiresAuthentication",
                RequiresAuthenticationFilter.class);
    }

    /**
     * Contribute to Coercion of Tapestry for PieChartData used as page context.
     *
     * @param configuration Coercion configuration
     * @see PieChart
     */
    public static void contributeTypeCoercer(
            Configuration<CoercionTuple<?, ?>> configuration) {
        
        Coercion<String, PieChartData> coercion1 = new Coercion<String, PieChartData>() {

            @Override
            public PieChartData coerce(String input) {
                PieChartData data = new PieChartDataImpl();
                data.setData(input);
                return data;
            }
        };
        Coercion<PieChartData, String> coercion2 = new Coercion<PieChartData, String>() {

            @Override
            public String coerce(PieChartData input) {
                return input.getData();
            }
        };

        // coercions for String <-> PieChartData
        configuration.add(new CoercionTuple<String, PieChartData>(
                String.class, PieChartData.class, coercion1));
        configuration.add(new CoercionTuple<PieChartData, String>(
                PieChartData.class, String.class, coercion2));
    }

}
