package fr.ifremer.tutti.ui.swing.util.editor;

/*
 * #%L
 * Tutti :: UI
 * $Id: AttachmentCellComponent.java 432 2013-02-15 20:28:55Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/editor/AttachmentCellComponent.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.ui.swing.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableModel;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.ComponentMover;
import jaxx.runtime.swing.ComponentResizer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.decorator.Decorator;

import javax.swing.AbstractCellEditor;
import javax.swing.BorderFactory;
import javax.swing.JDialog;
import javax.swing.JTable;
import javax.swing.border.LineBorder;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import java.awt.Color;
import java.awt.Component;
import java.awt.Frame;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Collection;

import static org.nuiton.i18n.I18n._;

/**
 * Component to edit a cell containing attachments
 *
 * @author kmorin
 * @since 0.2
 */
public class AttachmentCellComponent extends DefaultTableCellRenderer {

    /** Logger. */
    private static final Log log = LogFactory.getLog(AttachmentCellComponent.class);

    private static final long serialVersionUID = 1L;

    public AttachmentCellComponent() {
        setHorizontalAlignment(CENTER);
        setIcon(SwingUtil.createActionIcon("edit-attachment"));
    }

    @Override
    protected void setValue(Object value) {
        // do nothing
    }

    public static TableCellRenderer newRender(Decorator<Attachment> decorator, String noneText) {
        return new AttachmentCellRenderer(decorator, noneText);
    }

    public static TableCellEditor newEditor(AttachmentEditorUI ui) {
        return new AttachmentCellEditor(ui);
    }

    public static class AttachmentCellEditor extends AbstractCellEditor implements TableCellEditor {

        private static final long serialVersionUID = 1L;

        protected final AttachmentCellComponent component;

        protected final AttachmentEditorUI ui;

        protected Frame frame;

        protected JTable table;

        protected AbstractTuttiTableModel<AbstractTuttiBeanUIModel> tableModel;

        protected ColumnIdentifier<AbstractTuttiBeanUIModel> columnIdentifier;

        protected AbstractTuttiBeanUIModel editRow;

        protected Integer rowIndex;

        protected Integer columnIndex;

        public AttachmentCellEditor(AttachmentEditorUI ui) {
            this.ui = ui;

            component = new AttachmentCellComponent();
            component.setBorder(new LineBorder(Color.BLACK));
            component.addKeyListener(new KeyAdapter() {
                @Override
                public void keyReleased(KeyEvent e) {
                    if (e.getKeyCode() == KeyEvent.VK_ENTER ||
                        e.getKeyCode() == KeyEvent.VK_SPACE) {
                        e.consume();
                        startEdit();
                    }
                }
            });

            component.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    e.consume();
                    startEdit();
                }
            });
        }

        protected void startEdit() {
            if (frame == null) {
                frame = SwingUtil.getParentContainer(ui, Frame.class);
            }

            ui.setBorder(BorderFactory.createTitledBorder(_(columnIdentifier.getHeaderI18nKey())));
            ui.setBean(editRow);
            ui.setProperty(columnIdentifier.getPropertyName());

            JDialog editor = new JDialog(frame, true);
            editor.setUndecorated(true);
            editor.add(ui);
            editor.setResizable(true);
            editor.pack();

            ComponentResizer cr = new ComponentResizer();
            cr.registerComponent(editor);
            ComponentMover cm = new ComponentMover();
            cm.setDragInsets(cr.getDragInsets());
            cm.registerComponent(editor);

            editor.addWindowListener(new WindowAdapter() {

                @Override
                public void windowClosed(WindowEvent e) {
                    Component ui = (Component) e.getSource();
                    if (log.isInfoEnabled()) {
                        log.info("Destroy ui " + ui);
                    }
                    JAXXUtil.destroy(ui);
                }
            });

            // Computes the location of bottom left corner of the cell
            Component comp = component;
            int x = 0;
            int y = component.getHeight();
            while (comp != null) {
                x += comp.getX();
                y += comp.getY();
                comp = comp.getParent();
            }
            // if the editor is too big on the right,
            // then align its right side to the right side of the cell
            if (x + editor.getWidth() > frame.getX() + frame.getWidth()) {
                x = x - editor.getWidth() + component.getWidth();
            }
            editor.setLocation(x, y);
            editor.setVisible(true);

            int r = rowIndex;
            int c = columnIndex;

            // stop edition
            stopCellEditing();

            // reselect this cell
            AbstractSelectTableAction.doSelectCell(table, r, c);
            table.requestFocus();
        }

        @Override
        public Component getTableCellEditorComponent(JTable table,
                                                     Object value,
                                                     boolean isSelected,
                                                     int row,
                                                     int column) {
            if (tableModel == null) {
                tableModel = (AbstractTuttiTableModel<AbstractTuttiBeanUIModel>) table.getModel();
                this.table = table;
                columnIdentifier = tableModel.getPropertyName(column);
            }

            rowIndex = row;
            columnIndex = column;

            editRow = tableModel.getEntry(row);

            return component;
        }

        @Override
        public Object getCellEditorValue() {

            Preconditions.checkNotNull(editRow, "No editRow found in editor.");

            String propertyName = columnIdentifier.getPropertyName();
            Object result = TuttiUIUtil.getProperty(editRow, propertyName);
            if (log.isInfoEnabled()) {
                log.info("editor value (" + propertyName + "): " + result);
            }

            return result;
        }

        @Override
        public boolean stopCellEditing() {
            boolean b = super.stopCellEditing();
            if (b) {
                rowIndex = null;
                editRow = null;
                columnIndex = null;
            }
            return b;
        }

        @Override
        public void cancelCellEditing() {
            super.cancelCellEditing();
            rowIndex = null;
            columnIndex = null;
            editRow = null;
        }
    }

    public static class AttachmentCellRenderer implements TableCellRenderer {

        protected final AttachmentCellComponent component;

        public static final String TEXT_PATTERN = "<html><body>%s</body></html>";

        private final String noneText;

        private final Decorator<Attachment> decorator;

        protected String propertyName;

        public AttachmentCellRenderer(Decorator<Attachment> decorator,
                                      String noneText) {
            component = new AttachmentCellComponent();
            this.noneText = noneText;
            this.decorator = decorator;
        }

        @Override
        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row,
                                                       int column) {
            AttachmentCellComponent result =
                    (AttachmentCellComponent) component.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

            if (table != null) {

                String textValue;
                if (value == null) {
                    if (propertyName == null) {
                        AbstractTuttiTableModel tableModel =
                                (AbstractTuttiTableModel) table.getModel();
                        ColumnIdentifier columnIdentifier = tableModel.getPropertyName(column);
                        propertyName = columnIdentifier.getPropertyName();
                    }
                    // use HTML to show the tooltip in italic
                    textValue = "<i>" + _(noneText) + "</i>";

                } else {
                    Collection<Attachment> attachments = (Collection<Attachment>) value;
                    StringBuilder sb = new StringBuilder();
                    for (Attachment attachment : attachments) {
                        sb.append("<br/>").append(decorator.toString(attachment));
                    }
                    // use html to display the tooltip on several lines
                    textValue = sb.substring(5);
                }
                String text = String.format(TEXT_PATTERN, textValue);
                boolean editable = table.isCellEditable(row, column);
                result.setEnabled(editable);
                result.setToolTipText(text);
            }

            return result;
        }
    }
}
