/*
 * *##% 
 * SuiviObsmer :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package fr.ifremer.suiviobsmer.ui.pages;

import fr.ifremer.suiviobsmer.SuiviObsmerBusinessException;
import fr.ifremer.suiviobsmer.SuiviObsmerContext;
import fr.ifremer.suiviobsmer.SuiviObsmerException;
import fr.ifremer.suiviobsmer.bean.BoatFilter;
import fr.ifremer.suiviobsmer.bean.ContactFilter;
import fr.ifremer.suiviobsmer.bean.ContactFilterImpl;
import fr.ifremer.suiviobsmer.bean.ContactState;
import fr.ifremer.suiviobsmer.bean.ImportResults;
import fr.ifremer.suiviobsmer.entity.Boat;
import fr.ifremer.suiviobsmer.entity.Contact;
import fr.ifremer.suiviobsmer.entity.SampleRow;
import fr.ifremer.suiviobsmer.entity.User;
import fr.ifremer.suiviobsmer.services.ServiceBoat;
import fr.ifremer.suiviobsmer.services.ServiceContact;
import fr.ifremer.suiviobsmer.services.ServiceReferential;
import fr.ifremer.suiviobsmer.services.ServiceSampling;
import fr.ifremer.suiviobsmer.ui.base.AbstractFilteredPage;
import fr.ifremer.suiviobsmer.ui.data.BusinessUtils;
import fr.ifremer.suiviobsmer.ui.base.GenericSelectModel;
import fr.ifremer.suiviobsmer.ui.base.SuiviObsmerPage;
import fr.ifremer.suiviobsmer.ui.components.Layout;
import fr.ifremer.suiviobsmer.ui.services.ContactModelFactory;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.ComponentResources;
import org.apache.tapestry5.Field;
import org.apache.tapestry5.OptionModel;
import org.apache.tapestry5.SelectModel;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.Form;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.internal.OptionModelImpl;
import org.apache.tapestry5.internal.SelectModelImpl;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.services.BeanModelSource;
import org.apache.tapestry5.services.Response;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.slf4j.Logger;

/**
 * Contacts
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois
 * @version $Revision: 212 $
 *
 * Mise a jour: $Date: 2010-01-17 16:36:58 +0100 (dim. 17 janv. 2010) $
 * par : $Author: fdesbois $
 */
@IncludeStylesheet("context:css/contacts.css")
public class Contacts extends AbstractFilteredPage implements SuiviObsmerPage {

    @Override
    public boolean isOnlyForAdmin() {
        return false;
    }

    @Inject
    private Logger log;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private User user;
    
    @Inject
    private ServiceContact serviceContact;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceBoat serviceBoat;

    @Persist("flash")
    private String editableContactId;

    void setupRender() throws SuiviObsmerException {
        initFilter();
        contacts = null;
        getContacts();
        contactsForm.clearErrors();
    }

    /**************************** CONTACT FILTERS *****************************/

    @Persist
    private ContactFilter contactFilter;

    @InjectComponent
    private Zone filtersZone;

    @InjectComponent
    private Zone importExportZone;

    @Property
    @Persist("flash")
    private boolean filtersVisible;

    private boolean reset;

    public ContactFilter getContactFilter() throws SuiviObsmerException {
        if (contactFilter == null) {
            contactFilter = new ContactFilterImpl();
//            if (!user.getAdmin()) {
//                contactFilter.setCompany(user.getCompany());
//            }
        }
        return contactFilter;
    }

    @Override
    protected BoatFilter getFilter() throws SuiviObsmerException {
        return getContactFilter();
    }
    
    @Override
    protected void resetFilter() {
        contactFilter = null;
    }

//    @Override
//    protected Zone getSampleRowFilterZone() {
//        return filtersZone;
//    }
//
//    @Override
//    protected Zone getObserverFilterZone() {
//        return filtersZone;
//    }

    @Override
    protected boolean getAvailableDataForFiltersOnly() {
        return true;
    }

    void onSuccessFromSearchBoat() {

    }

    Object onActionFromShowFilters() {
        contactFilter = null;
        return filtersZone.getBody();
    }

    Object onActionFromShowImportExport() {
        return importExportZone.getBody();
    }

    void onSelectedFromReset() {
        reset = true;
    }

    @Log
    void onSuccessFromFiltersForm() throws SuiviObsmerException {
        if (reset) {
            contactFilter = null;
        } else {
            filtersVisible = true;
        }
    }

    /**************************** CONTACT IMPORT/EXPORT ***********************/

    @Property
    private UploadedFile contactsCsvFile;

    @Log
    void onSuccessFromImportContacts() throws SuiviObsmerException {
        //importBoatsForm.clearErrors();
        try {
            ImportResults result = serviceContact.importContactCsv(user, contactsCsvFile.getStream());
            // Suppress persitant list of contacts
            contacts = null;
            layout.getFeedBack().addInfo(result.getNbRowsImported() + " contacts importés,  " +
                    result.getNbRowsRefused() + " refusés.");
            for (String error : result.getErrors()) {
                layout.getFeedBack().addInfo(error);
            }
        } catch (SuiviObsmerBusinessException eee) {
            layout.getFeedBack().addError(eee.getMessage());
        }
        //return importBoatsForm.getHasErrors() ? importBoatsForm : this;
    }

    StreamResponse onActionFromExportShowContacts() {
        return new StreamResponse() {

            @Override
            public String getContentType() {
                return "text/csv;charset=utf-8";
            }

            @Override
            public InputStream getStream() throws IOException {
                InputStream result = null;
                try {
                    result = serviceContact.exportContactCsv(user, getContacts().values());
                } catch (SuiviObsmerException eee) {
                    throw new IOException(eee);
                }
                return result;
            }

            @Override
            public void prepareResponse(Response response) {
            }
        };
    }

    /**************************** CONTACT LIST ********************************/

    @Inject
    private BeanModelSource beanModelSource;

    @Inject
    private ComponentResources resources;

    @Inject
    private ContactModelFactory contactModelFactory;

    @Persist
    private Map<String, Contact> contacts;

    @Property
    private Contact contact;

    private BeanModel<Contact> contactModel;

    @Inject
    private PropertyAccess propertyAccess;

    private GenericSelectModel<User> userSelectModel;

    @Property
    @Persist("flash")
    private String contactUserId;
    
    private ContactState contactState;

    private boolean even = true;

    public Map<String, Contact> getContacts() throws SuiviObsmerException {
        if (contacts == null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getContactsByFilter]");
            }
            contacts = serviceContact.getContactsByFilter(getContactFilter());
        }
        return contacts;
    }

    public BeanModel<Contact> getContactModel() {
        if (contactModel == null) {
            contactModel = user.getAdmin() ?
                contactModelFactory.buildAdminContactModel(beanModelSource, resources) :
                contactModelFactory.buildContactModel(beanModelSource, resources);
        }
        return contactModel;
    }

    public ContactState getContactState() {
        if (contactState == null) {
            // initialization
            this.contactState = ContactState.createContactStateEnum(contact.getState());
        }
        return this.contactState;
    }

    public GenericSelectModel<User> getUserSelectModel() {
        if (userSelectModel == null) {
            List<User> users = user.getCompany().getUser();
            if (log.isDebugEnabled()) {
                log.debug("Nb users : " + users.size());
            }
            userSelectModel = new GenericSelectModel<User>(users, User.class, "fullName", "id", propertyAccess);
        }
        return userSelectModel;
    }

    public void setContactState(ContactState contactState) {
        this.contactState = contactState;
    }

    public String getSampleRowDescription() {
        return BusinessUtils.getTooltipSampleRow(contact.getSampleRow());
    }

    public String getBoatDescription() {
        return BusinessUtils.getTooltipBoat(contact.getBoat());
    }

    public String getGridClass() {
        return user.getAdmin() ? "admin" : "user";
    }

    public String getRowClass() {
        String result = BusinessUtils.getCSSColorClassForContact(contact, user.getAdmin());
        if (contact.getTopiaId().equals(contactSelectedId)) {
            result = "selected";
        }
        return result;
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("dd/MM/yyyy");
    }

    public boolean isEditionMode() {
        return contact.getTopiaId().equals(editableContactId);
    }

    public boolean isEmpty(Boolean validation) {
        return validation == null;
    }

    /**************************** CONTACT ROW ACTION **************************/

    /**
     * Flag to know if it's only edition (=true) or save action (=false)
     */
    private boolean contactEdited;

    private boolean contactDeleted;

    public boolean hasActions() {
        return !user.getAdmin() && contact.getValidationCompany() == null;
    }

    public boolean canValidate() {
        ContactState state = ContactState.createContactStateEnum(contact.getState());
        boolean boardingDone = state.equals(ContactState.BOARDING_DONE) && contact.getDataInputDate() != null;
        if (!user.getAdmin()) {
            return contact.getValidationCompany() == null && (state.isUnfinishedState() || boardingDone);
        }
        return contact.getValidationProgram() == null && BooleanUtils.isTrue(contact.getValidationCompany());
    }

    public boolean canUnvalidate() {
        if (!user.getAdmin()) {
            return contact.getValidationCompany() != null &&
                    contact.getValidationProgram() == null;
        }
        return contact.getValidationProgram() != null;
    }

    void onSelectedFromAcceptContact(String contactId) throws SuiviObsmerException {
        contact = getContacts().get(contactId);        
        if (user.getAdmin()) {
            contact.setValidationProgram(Boolean.TRUE);
        } else {
            // For company accepted, addRealTideTime
            contact.getSampleRow().addRealTideTime(contact);
            contact.setValidationCompany(Boolean.TRUE);
        }
    }

    void onSelectedFromRefuseContact(String contactId) throws SuiviObsmerException {
        contact = getContacts().get(contactId);
        if (user.getAdmin()) {
            // For program refused, removeRealTideTime
            contact.getSampleRow().removeRealTideTime(contact);
            contact.setValidationProgram(Boolean.FALSE);
        } else {
            contact.setValidationCompany(Boolean.FALSE);
        }
    }

    void onSelectedFromUnvalidateContact(String contactId) throws SuiviObsmerException {
        contact = getContacts().get(contactId);
        if (user.getAdmin()) {
            // For program unvalidate from previous refused validation, addRealTideTime
            if (BooleanUtils.isFalse(contact.getValidationProgram())) {
                contact.getSampleRow().addRealTideTime(contact);
            }
            contact.setValidationProgram(null);
        } else {
            // For company unvalidate from previous accepted validation, removeRealTideTime
            if (BooleanUtils.isTrue(contact.getValidationCompany())) {
                contact.getSampleRow().removeRealTideTime(contact);
            }
            contact.setValidationCompany(null);
        }
    }

    void onSelectedFromEditContact(String contactId) throws SuiviObsmerException {
        contact = getContacts().get(contactId);
        contactUserId = contact.getUser().getTopiaId();
        editableContactId = contactId;
        contactSelectedId = contactId;
        contactEdited = true;
    }

    void onSelectedFromDeleteContact(String contactId) throws SuiviObsmerException {
        contact = getContacts().get(contactId);
        contactDeleted = true;
    }

    void onSelectedFromSaveContact(String contactId) throws SuiviObsmerException {
        contact = getContacts().get(contactId);
        contact.setState(contactState.toString());
        User contactUser = getUserSelectModel().findObject(contactUserId);
        contact.setUser(contactUser);
    }

    void onSelectedFromCancelEditContact() throws SuiviObsmerException {
        contactEdited = true;
    }

//    @Log
//    void onSelectedFromSaveComment(String contactId) throws SuiviObsmerException {
//        contact = getContacts().get(contactId);
//        if (log.isInfoEnabled()) {
//            log.info("Comment : " + contact.getComment());
//        }
//        contactEdited = true;
//    }

    /**************************** CONTACT SAVE ********************************/

    @Persist("flash")
    private String contactSelectedId;

    @InjectComponent
    private Form contactsForm;

    @InjectComponent
    private Field beginDate;

    @InjectComponent
    private Field endDate;

    @InjectComponent
    private Field nbObservants;

    @InjectComponent
    private Field comment;

    @InjectComponent
    private Field inputDate;

    @Log
    void onValidateFormFromContactsForm() {
        contactsForm.clearErrors();
        // Validation for saving contact depends on contactState (only edition form)
        if (!contactEdited && contactState != null) {
            if (log.isInfoEnabled()) {
                log.info("For state : " + contactState);
            }

            Date begin = contact.getTideBeginDate();
            Date end = contact.getTideEndDate();
            Date input = contact.getDataInputDate();
            SampleRow row = contact.getSampleRow();
            DateFormat dateFormat = new SimpleDateFormat("MM/yyyy");

            if (begin != null && !row.isValid(begin)) {
                contactsForm.recordError(beginDate,
                        "La date de début de la marée doit correspondre à un mois valide (non vide) de la ligne " + row.getCode());
            }

//            if (end != null && !DateUtils.between(end, row.getPeriodBegin(), row.getPeriodEnd())) {
//                contactsForm.recordError(endDate,
//                        "La date de fin de la marée doit être comprise dans la période de la ligne : du " +
//                        dateFormat.format(row.getPeriodBegin()) + " au " + dateFormat.format(row.getPeriodEnd()));
//            }

            if (begin != null && end != null && end.before(begin)) {
                contactsForm.recordError(endDate, "La date de fin de la marée ne peut pas être antérieure à celle de début");
            }

            Date current = SuiviObsmerContext.getCurrentDate();

            if (end != null && end.after(current)) {
                contactsForm.recordError(endDate, "La date de fin de la marée ne peut pas être postérieure à la date du jour");
            }

            if (end != null && input != null && end.after(input)) {
                contactsForm.recordError(inputDate, "La date de saisie des données ne peut pas être antérieure à la date de fin de la marée");
            }

            if (input != null && input.after(current)) {
                contactsForm.recordError(inputDate, "La date de saisie des données ne peut pas être postérieure à la date du jour");
            }

            // Non abouti, Refus ou Refus Définitif
            if (contactState.isUnfinishedState()) {
                if (!StringUtils.isEmpty(contact.getComment())) {
                    // RAZ des champs
                    contact.setTideBeginDate(null);
                    contact.setTideEndDate(null);
                    contact.setNbObservants(0);
                    contact.setMammalsCapture(false);
                    contact.setMammalsObservation(false);
                } else {
                    contactsForm.recordError(comment, "Le commentaire est obligatoire pour l'état '" + contactState + "'");
                }
            // Embarquement Réalisé
            } else if (contactState.equals(ContactState.BOARDING_DONE)) {
                if (begin == null) {
                    contactsForm.recordError(beginDate, "La date de début de marée est obligatoire pour l'état '" + contactState + "'");
                }
                if (end == null) {
                    contactsForm.recordError(endDate, "La date de fin de marée est obligatoire pour l'état '" + contactState + "'");
                }
                if (contact.getNbObservants() == 0) {
                    contactsForm.recordError(nbObservants, "Il ne peut y avoir aucun observateur pour l'état '" + contactState + "'");
                }
            }
        }
    }

    @Log
    Object onSuccessFromContactsForm() {
        if (!contactEdited) {            
            try {
                if (log.isDebugEnabled()) {
                    log.debug("Contact save : " + contact);
                }
                serviceContact.saveContact(contact, contactDeleted);
                contactSelectedId = contact.getTopiaId();
            } catch (SuiviObsmerException eee) {
                layout.getFeedBack().addError(eee.getMessage());
            }
        }
        return this;
    }

    @Log
    Object onFailureFromContactsForm() {
        if (log.isDebugEnabled()) {
            log.debug("Contact can't be saved with errors");
        }
        // The contact is not saved, the contact must be editable to show form and correct errors
        contactEdited = true;
        editableContactId = contact.getTopiaId();
        return contactsForm;
    }

    public void createNewContact(Boat boat, SampleRow sampleRow) throws SuiviObsmerException {
        contact = serviceContact.getNewContact(user, sampleRow, boat);
        serviceContact.saveContact(contact, Boolean.FALSE);
        contactSelectedId =  contact.getTopiaId();
    }
}
