/*
 * *##% 
 * SuiviObsmer :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package fr.ifremer.suiviobsmer.ui.base;

import fr.ifremer.suiviobsmer.SuiviObsmerException;
import fr.ifremer.suiviobsmer.bean.BoatFilter;
import fr.ifremer.suiviobsmer.entity.Company;
import fr.ifremer.suiviobsmer.entity.SampleRow;
import fr.ifremer.suiviobsmer.entity.User;
import fr.ifremer.suiviobsmer.services.ServiceBoat;
import fr.ifremer.suiviobsmer.services.ServiceReferential;
import fr.ifremer.suiviobsmer.services.ServiceSampling;
import fr.ifremer.suiviobsmer.services.ServiceUser;
import java.util.ArrayList;
import java.util.List;
import org.apache.tapestry5.OptionModel;
import org.apache.tapestry5.SelectModel;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.internal.OptionModelImpl;
import org.apache.tapestry5.internal.SelectModelImpl;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.slf4j.Logger;

/**
 * AbstractFilteredPage
 *
 * Created: 14 janv. 2010
 *
 * This class uses the BoatFilter bean (also SamplingFilter and UserFilter which are supertypes).
 *
 * Known implementations are {@link fr.ifremer.suiviobsmer.ui.pages.Synthesis } and {@link fr.ifremer.suiviobsmer.ui.pages.Contacts }
 *
 * You must override the three abstract methods and call {@link #initFilter() } in the setupRender of the page.
 * <pre>
 * - BoatName uses an Autocomplete mixins :
 *      <t:textfield t:id="boatName" t:value="filter.boatName" t:mixins="Autocomplete"/>
 * - SampleRow select value is sampleRowId :
 *      <input t:type="select" t:id="sampleRow" t:model="sampleRowSelectModel" value="sampleRowId" />
 * - Company select value is companyId :
 *      <input t:type="select" t:id="company" t:model="companySelectModel" value="companyId" />
 * - Observer select value is observerId :
 *      <input t:type="select" t:id="observer" t:model="observerSelectModel" value="observerId" />
 * - Select change, use JavaScript :
 *      <script type="text/javascript">
 *         Event.observe('facadeName', 'change', function() { $('filtersForm').submit(); } );
 *         Event.observe('company', 'change', function() { $('filtersForm').submit(); } );
 *      </script>
 * </pre>
 *
 * @author fdesbois
 * @version $Revision: 222 $
 *
 * Mise a jour: $Date: 2010-01-17 20:49:21 +0100 (dim. 17 janv. 2010) $
 * par : $Author$
 */
public abstract class AbstractFilteredPage {

    @Inject
    private Logger log;

    @SessionState
    private User user;

    @Inject
    private PropertyAccess propertyAccess;
    
    @Inject
    private ServiceReferential serviceReferential;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceBoat serviceBoat;

    @Inject
    private ServiceUser serviceUser;

    @Persist
    private GenericSelectModel<SampleRow> sampleRowSelectModel;

    @Property
    private String sampleRowId;
    
    @Persist
    private SelectModel facadeSelectModel;

    @Persist
    private SelectModel sectorSelectModel;

    @Persist
    private GenericSelectModel<Company> companySelectModel;

    @Property
    private String companyId;

    @Persist
    private GenericSelectModel<User> observerSelectModel;

    @Property
    private String observerId;

    protected void initFilter() throws SuiviObsmerException {

        facadeSelectModel = null;
        getFacadeSelectModel();
        sectorSelectModel = null;
        getSectorSelectModel();
        sampleRowSelectModel = null;
        getSampleRowSelectModel();

        if (getFilter().getSampleRow() != null) {
            sampleRowId = getFilter().getSampleRow().getTopiaId();
        }
        
        if (user.getAdmin()) {
            companySelectModel = null;
            getCompanySelectModel();
            if (getFilter().getCompany() != null) {
                companyId = getFilter().getCompany().getTopiaId();
            }
        } else {
            if (log.isDebugEnabled()) {
                log.debug("Company filtered : " + user.getCompany());
            }
            getFilter().setCompany(user.getCompany());
        }

        observerSelectModel = null;
        getObserverSelectModel();

        if (getFilter().getObserver() != null) {
            observerId = getFilter().getObserver().getTopiaId();
        }
    }

    public SelectModel getFacadeSelectModel() throws SuiviObsmerException {
        if (facadeSelectModel == null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getFacades]");
            }
            List<OptionModel> options = new ArrayList<OptionModel>();
            for (String facadeName : serviceReferential.getFacades()) {
                options.add(new OptionModelImpl(facadeName,facadeName));
            }
            facadeSelectModel = new SelectModelImpl(null, options);
        }
        return facadeSelectModel;
    }

    public SelectModel getSectorSelectModel() throws SuiviObsmerException {
        if (sectorSelectModel == null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getSectors]");
            }
            List<OptionModel> options = new ArrayList<OptionModel>();
            for (String sectorName : serviceReferential.getSectors(getFilter().getFacadeName())) {
                options.add(new OptionModelImpl(sectorName,sectorName));
            }
            sectorSelectModel = new SelectModelImpl(null, options);
        }
        return sectorSelectModel;
    }

    public GenericSelectModel<SampleRow> getSampleRowSelectModel() throws SuiviObsmerException {
        if (sampleRowSelectModel == null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getSampleRowsForUser]");
            }
            List<SampleRow> sampleRows = serviceSampling.getSampleRowsForUser(user, getAvailableDataForFiltersOnly());
            sampleRowSelectModel = new GenericSelectModel<SampleRow>(sampleRows, SampleRow.class,
                "code", "topiaId", propertyAccess);
        }
        return sampleRowSelectModel;
    }

    public GenericSelectModel<Company> getCompanySelectModel() throws SuiviObsmerException {
        if (companySelectModel == null) {
            if (log.isDebugEnabled()) {
                log.debug("BUSINESS REQUEST [getCompanies]");
            }
            List<Company> companies = serviceUser.getCompanies(getAvailableDataForFiltersOnly());
            companySelectModel = new GenericSelectModel<Company>(companies,Company.class,
                    "name","topiaId",propertyAccess);
        }
        return companySelectModel;
    }

    public GenericSelectModel<User> getObserverSelectModel() throws SuiviObsmerException {
        if (observerSelectModel == null) {
            List<User> observers =  new ArrayList<User>();
            if (getFilter().getCompany() != null) {
                observers = getFilter().getCompany().getUser();
            } else if (user.getAdmin()) {
                if (log.isInfoEnabled()) {
                    log.info("BUSINESS REQUEST [getObservers]");
                }
                observers = serviceUser.getObservers(getAvailableDataForFiltersOnly());
            }
            observerSelectModel = new GenericSelectModel<User>(observers,User.class,
                    "fullName","topiaId",propertyAccess);
        }
        return observerSelectModel;
    }

    protected abstract BoatFilter getFilter() throws SuiviObsmerException;

    protected abstract void resetFilter();

//    protected abstract Zone getSampleRowFilterZone();
//
//    protected abstract Zone getObserverFilterZone();

    protected abstract boolean getAvailableDataForFiltersOnly();

    public String[] onProvideCompletionsFromBoatName(String input) throws SuiviObsmerException {
        if (log.isInfoEnabled()) {
            log.info("BUSINESS REQUEST [getBoatNamesStartWith]");
        }
        List<String> results = serviceBoat.getBoatNamesStartWith(input);
        return results.toArray(new String[0]);
    }

//    @Log
//    public Object onChangeFromFacadeName(String value) throws SuiviObsmerException {
//        if (!StringUtils.isEmpty(value.trim())) {
//            // Reset filter to avoid strange behavior (cause of no form submit)
//            resetFilter();// = null;
//            // set facadeName selected in filter
//            getFilter().setFacadeName(value);
//            // Reset sectors to use facadeName selected
//            sectorSelectModel = null;
//            return getSampleRowFilterZone().getBody();
//        }
//        return null;
//    }
//
//    @Log
//    public Object onChangeFromCompany(String value) throws SuiviObsmerException {
//        if (!StringUtils.isEmpty(value.trim())) {
//            // Reset filter to avoid strange behavior (cause of no form submit)
//            resetFilter();
//            // set company selected in filter
//            getFilter().setCompany(getCompanySelectModel().findObject(companyId));
//            // Reset observers to use company selected
//            observerSelectModel = null;
//            return getObserverFilterZone().getBody();
//        }
//        return null;
//    }

    @Log
    public void onSuccess() throws SuiviObsmerException {
        SampleRow row = getSampleRowSelectModel().findObject(sampleRowId);
        getFilter().setSampleRow(row);
        if (user.getAdmin()) {
            Company company = getCompanySelectModel().findObject(companyId);
            getFilter().setCompany(company);
        }

        User observer = getObserverSelectModel().findObject(observerId);
        getFilter().setObserver(observer);
    }
   
}
