/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published by the 
 * Free Software Foundation; either version 3 of the License, 
 * or (at your option) any later version.
 * <p/>
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 * 
 */
package de.intarsys.pdf.platform.cwt.image.swt;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.eclipse.swt.SWTException;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.eclipse.swt.graphics.PaletteData;

import de.intarsys.cwt.image.ImageException;
import de.intarsys.cwt.image.ImageTools;
import de.intarsys.cwt.swt.image.ImageConverterAwt2Swt;
import de.intarsys.pdf.cos.COSStream;
import de.intarsys.pdf.filter.Filter;
import de.intarsys.pdf.pd.PDImage;
import de.intarsys.pdf.platform.cwt.color.swt.SwtColorSpace;
import de.intarsys.pdf.platform.cwt.color.swt.SwtColorSpaceFactory;
import de.intarsys.pdf.platform.cwt.image.awt.ImageConverterPdf2Awt;
import de.intarsys.pdf.platform.cwt.image.awt.PlatformImageTools;

public class ImageConverterPdf2Swt {

	final private PDImage pdImage;

	private ImageData imageData;

	public ImageConverterPdf2Swt(PDImage pPdImage) {
		pdImage = pPdImage;
	}

	protected ImageData createImageData() {
		if (!pdImage.hasTransparency()) {
			try {
				ImageData result = createPrimaryImageData();
				// todo 2 @ehk alpha handling
				return result;
			} catch (ImageException ex) {
				//
			} catch (Exception ex) {
				//
			}
		}
		// try via AWT
		ImageConverterPdf2Awt converter1 = new ImageConverterPdf2Awt(
				getPDImage());
		BufferedImage bufferedImage = converter1.getBufferedImage();
		ImageConverterAwt2Swt converter2 = new ImageConverterAwt2Swt(
				bufferedImage);
		return converter2.getImageData();
	}

	protected ImageData createImageDataFromJPEG(COSStream cosStream)
			throws IOException {
		String colorSpaceType;
		InputStream stream;
		ImageData[] images;

		stream = new ByteArrayInputStream(cosStream.getEncodedBytes());
		try {
			colorSpaceType = ImageTools.extractJPEGColorSpaceType(stream);
		} catch (ImageException ex) {
			/*
			 * if color space cannot be extracted assume YCbCr and try decoding;
			 * might fail later
			 */
			colorSpaceType = "YCbCr"; //$NON-NLS-1$
		}
		if (!"GRAY".equals(colorSpaceType) && !"RGB".equals(colorSpaceType) //$NON-NLS-1$ //$NON-NLS-2$
				&& !"YCbCr".equals(colorSpaceType)) { //$NON-NLS-1$
			// cannot be decoded by swt
			throw new ImageException("SWT non-RGB/GRAY JPEG not supported");
		}
		if (!PlatformImageTools.matchColorSpace(pdImage, colorSpaceType)) {
			throw new ImageException(
					"PD color space does not match JPEG metadata");
		}

		stream.reset();
		try {
			images = new ImageLoader().load(stream);
		} catch (SWTException ex) {
			throw new ImageException("SWT JPEG decoding error");
		}
		if (images[0].data.length == 0) {
			// didn't work
			throw new ImageException("SWT JPEG decoding error");
		}
		return images[0];
	}

	protected ImageData createPrimaryImageData() {
		PaletteData palette;
		ImageData imageData;

		if (pdImage.cosGetStream().hasFilter(Filter.CN_Filter_JPXDecode)) {
			// no direct conversion possible
			throw new ImageException("SWT JPX stream not supported");
		}
		if (pdImage.cosGetStream().hasFilter(Filter.CN_Filter_DCTDecode)
				|| pdImage.cosGetStream().hasFilter(Filter.CN_Filter_DCT)) {
			try {
				return createImageDataFromJPEG(pdImage.cosExtractJPEGStream());
			} catch (IOException ex) {
				throw new ImageException("SWT JPEG decoding error");
			}
		}
		try {
			SwtColorSpace pfColorSpace = SwtColorSpaceFactory.get()
					.createPlatformColorSpace(pdImage.getColorSpace());
			palette = pfColorSpace.getPalette(pdImage);
			int numComponents = pfColorSpace.getNumComponents();
			imageData = new ImageData(pdImage.getWidth(), pdImage.getHeight(),
					pdImage.getBitsPerComponent() * numComponents, palette,
					(pdImage.getBitsPerComponent() * numComponents + 7) / 8,
					pdImage.getAdjustedBytes(numComponents));
			return imageData;
		} catch (UnsupportedOperationException ex) {
			// expected if color space cannot be converted
			throw new ImageException("SWT color space not supported", ex);
		}
	}

	public ImageData getImageData() throws ImageException {
		if (imageData == null) {
			imageData = createImageData();
		}
		return imageData;
	}

	public PDImage getPDImage() {
		return pdImage;
	}
}
